<!--
%CopyrightBegin%

SPDX-License-Identifier: Apache-2.0

Copyright Ericsson AB 2023-2025. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

%CopyrightEnd%
-->
# Errors and Error Handling

## Terminology

Errors can roughly be divided into four different types:

- **Compile-time errors** - When the compiler fails to compile the program, for
  example a syntax error.

- **Logical errors** - When a program does not behave as intended, but does not
  crash. An example is that nothing happens when a button in a graphical user
  interface is clicked.

- **[](){: #run-time-errors } Run-time errors** - 
  When a crash occurs. An example is when an operator is applied to arguments of
  the wrong type. The Erlang programming language has built-in features for
  handling of run-time errors. A run-time error can also be emulated by calling
  [`error(Reason)`](`erlang:error/1`). Run-time errors are exceptions of class
  `error`.

- **[](){: #generated-errors } Generated errors**
  When the code itself calls [`exit/1`](`erlang:exit/1`) or
  [`throw/1`](`erlang:throw/1`). Generated errors are exceptions of class `exit`
  or `throw`.

When an exception occurs in Erlang, execution of the process that evaluated the
erroneous expression is stopped. This is referred to as a _failure_, that
execution or evaluation _fails_, or that the process _fails_, _terminates_, or
_exits_. Notice that a process can terminate/exit for other reasons than a
failure.

A process that terminates emits an _exit signal_ with an _exit reason_ that
describes why the process terminated. Normally, some information about any
erroneous termination is printed to the terminal. See
[Process Termination](ref_man_processes.md#process-termination) in the Processes
chapter for more details on termination.

## Exceptions

Exceptions are [run-time errors](errors.md#run-time-errors) or
[generated errors](errors.md#generated-errors) and are of three different
classes, with different origins. The [try](expressions.md#try) expression can
distinguish between the different classes, whereas the
[catch](expressions.md#catch-and-throw) expression cannot. `try` and `catch` are described
in [Expressions](expressions.md).

| Class   | Origin                                                                           |
| ------- | -------------------------------------------------------------------------------- |
| `error` | Run-time error, for example, `1+a`, or the process called [`error/1`](`error/1`) |
| `exit`  | The process called [`exit/1`](`exit/1`)                                          |
| `throw` | The process called [`throw/1`](`throw/1`)                                        |

_Table: Exception Classes._

All of the above exceptions can also be generated by calling `erlang:raise/3`.

An exception consists of its class, an exit reason (see
[Exit Reason](errors.md#exit_reasons)), and a stack trace (which aids in finding
the code location of the exception).

The stack trace can be bound to a variable from within a `try` expression for
any exception class, or as part of the exit reason when a run-time error is
caught by a `catch`. Example:

```erlang
> {'EXIT',{test,Stacktrace}} = (catch error(test)), Stacktrace.
[{shell,apply_fun,3,[]},
 {erl_eval,do_apply,6,[]},
 ...]
> try throw(test) catch Class:Reason:Stacktrace -> Stacktrace end.
[{shell,apply_fun,3,[]},
 {erl_eval,do_apply,6,[]},
 ...]
```

[](){: #stacktrace }

### The call-stack back trace (stacktrace)

The stack back-trace ([_stacktrace_](`t:erlang:stacktrace/0`)) is a list that
contains `{Module, Function, Arity, ExtraInfo}` and/or `{Fun, Arity, ExtraInfo}`
tuples. The field `Arity` in the tuple can be the argument list of that function
call instead of an arity integer, depending on the exception.

`ExtraInfo` is a (possibly empty) list of two-element tuples in any order that
provides additional information about the exception. The first element is an
atom describing the type of information in the second element. The following
items can occur:

- **`error_info`** - The second element of the tuple is a map providing
  additional information about what caused the exception. This information can
  be created by calling [`error/3`](`erlang:error/3`) and is used by
  `erl_error:format_exception/4`.

- **`file`** - The second element of the tuple is a string (list of characters)
  representing the filename of the source file of the function.

- **`line`** - The second element of the tuple is the line number (an
  integer > 0) in the source file where the exception occurred or the function
  was called.

> #### Warning {: .warning }
>
> Developers should rely on stacktrace entries only for debugging purposes.
>
> The VM performs tail call optimization, which does not add new entries to the
> stacktrace, and also limits stacktraces to a certain depth. Furthermore,
> compiler options, optimizations, and future changes may add or remove
> stacktrace entries, causing any code that expects the stacktrace to be in a
> certain order or contain specific items to fail.
>
> The only exception to this rule is the class `error` with the reason `undef`
> which is guaranteed to include the `Module`, `Function` and `Arity` of the
> attempted function as the first stacktrace entry.

## Handling of Run-time Errors in Erlang

### Error Handling Within Processes

It is possible to prevent run-time errors and other exceptions from
causing the process to terminate by using [`try`](expressions.md#try)
or [`catch`](expressions.md#catch-and-throw).

### Error Handling Between Processes

Processes can monitor other processes and detect process terminations, see
[Processes](ref_man_processes.md#errors).

[](){: #exit_reasons }

## Exit Reasons

When a run-time error occurs, that is an exception of class `error`. The exit
reason is a tuple `{Reason,Stack}`, where `Reason` is a term indicating the type
of error:

- **`badarg`** - Bad argument. The argument is of wrong data type, or
    is otherwise badly formed.

- **`badarith`** - An argument for an arithmetic expression was not numeric,
    or the expression does not evaluate to finite number.

- **`{badmatch,V}`** - Evaluation of a match expression failed. The
    value `V` did not match.

- **`function_clause`** - No matching function clause is found when
    evaluating a function call.

- **`{case_clause,V}`** - No matching branch is found when evaluating
    a `case` expression. The value `V` did not match.

- **`if_clause`** - No true branch is found when evaluating an `if`
    expression.

- **`{try_clause,V}`** - No matching branch is found when evaluating
    the of-section of a `try` expression. The value `V` did not
    match.

- **`undef`** - The function cannot be found when evaluating a
    function call.

- **`{badfun,F}`** - `F` was expected to a be a fun, but is not.

- **`{badarity,{Fun,Args}}`** - A fun is applied to the wrong number of
    arguments.

- **`timeout_value`** - The timeout value in a `receive...after`
    expression is evaluated to something else than an integer or
    `infinity`.

- **`noproc`** - Trying to create [link](`link/1`) or
    [monitor](`monitor/2`) to a non-existing process or port.

- **`noconnection`** - A link or monitor to a remote process was
    broken because a connection between the nodes could not be
    established or was severed.

- **`{nocatch,V}`** - Trying to evaluate a `throw `outside a
    `catch`. `V` is the thrown term.

- **`system_limit`** - A system limit has been reached. See
    [System Limits in the Efficiency Guide](`e:system:system_limits.md`)
    for information about system limits.

`Stack` is the stack of function calls being evaluated when the error occurred,
given as a list of tuples `{Module,Name,Arity,ExtraInfo}` with the most recent
function call first. The most recent function call tuple can in some cases be
`{Module,Name,[Arg],ExtraInfo}`.
