import ora from 'ora';
import chalk from 'chalk';
import { parse } from '@swc/core';
import { mkdir, writeFile, readFile } from 'node:fs/promises';
import { dirname, extname } from 'node:path';
import { findKeys } from './key-finder.js';
import { getTranslations } from './translation-manager.js';
import { validateExtractorConfig, ExtractorError } from '../../utils/validation.js';
import { extractKeysFromComments } from '../parsers/comment-parser.js';
import { ConsoleLogger } from '../../utils/logger.js';
import { loadRawJson5Content, serializeTranslationFile } from '../../utils/file-utils.js';
import { shouldShowFunnel, recordFunnelShown } from '../../utils/funnel-msg-tracker.js';

/**
 * Main extractor function that runs the complete key extraction and file generation process.
 *
 * This is the primary entry point that:
 * 1. Validates configuration
 * 2. Sets up default sync options
 * 3. Finds all translation keys across source files
 * 4. Generates/updates translation files for all locales
 * 5. Provides progress feedback via spinner
 * 6. Returns whether any files were updated
 *
 * @param config - The i18next toolkit configuration object
 * @param logger - Logger instance for output (defaults to ConsoleLogger)
 * @returns Promise resolving to boolean indicating if any files were updated
 *
 * @throws {ExtractorError} When configuration validation fails or extraction process encounters errors
 *
 * @example
 * ```typescript
 * const config = await loadConfig()
 * const updated = await runExtractor(config)
 * if (updated) {
 *   console.log('Translation files were updated')
 * }
 * ```
 */
async function runExtractor(config, { isWatchMode = false, isDryRun = false, syncPrimaryWithDefaults = false, syncAll = false } = {}, logger = new ConsoleLogger()) {
    config.extract.primaryLanguage ||= config.locales[0] || 'en';
    config.extract.secondaryLanguages ||= config.locales.filter((l) => l !== config?.extract?.primaryLanguage);
    // Ensure default function and component names are set if not provided.
    config.extract.functions ||= ['t', '*.t'];
    config.extract.transComponents ||= ['Trans'];
    validateExtractorConfig(config);
    const plugins = config.plugins || [];
    const spinner = ora('Running i18next key extractor...\n').start();
    try {
        const { allKeys, objectKeys } = await findKeys(config, logger);
        spinner.text = `Found ${allKeys.size} unique keys. Updating translation files...`;
        const results = await getTranslations(allKeys, objectKeys, config, { syncPrimaryWithDefaults, syncAll });
        let anyFileUpdated = false;
        for (const result of results) {
            if (result.updated) {
                anyFileUpdated = true;
                if (!isDryRun) {
                    // prefer explicit outputFormat; otherwise infer from file extension per-file
                    const effectiveFormat = config.extract.outputFormat ?? (result.path.endsWith('.json5') ? 'json5' : 'json');
                    const rawContent = effectiveFormat === 'json5'
                        ? (await loadRawJson5Content(result.path)) ?? undefined
                        : undefined;
                    const fileContent = serializeTranslationFile(result.newTranslations, effectiveFormat, config.extract.indentation, rawContent);
                    await mkdir(dirname(result.path), { recursive: true });
                    await writeFile(result.path, fileContent);
                    logger.info(chalk.green(`Updated: ${result.path}`));
                }
            }
        }
        // Run afterSync hooks from plugins
        if (plugins.length > 0) {
            spinner.text = 'Running post-extraction plugins...';
            for (const plugin of plugins) {
                await plugin.afterSync?.(results, config);
            }
        }
        spinner.succeed(chalk.bold('Extraction complete!'));
        // Show the funnel message only if files were actually changed.
        if (anyFileUpdated)
            await printLocizeFunnel();
        return anyFileUpdated;
    }
    catch (error) {
        spinner.fail(chalk.red('Extraction failed.'));
        // Re-throw or handle error
        throw error;
    }
}
/**
 * Processes an individual source file for translation key extraction.
 *
 * This function:
 * 1. Reads the source file
 * 2. Runs plugin onLoad hooks for code transformation
 * 3. Parses the code into an Abstract Syntax Tree (AST) using SWC
 * 4. Extracts keys from comments using regex patterns
 * 5. Traverses the AST using visitors to find translation calls
 * 6. Runs plugin onVisitNode hooks for custom extraction logic
 *
 * @param file - Path to the source file to process
 * @param config - The i18next toolkit configuration object
 * @param logger - Logger instance for output
 * @param allKeys - Map to accumulate found translation keys
 *
 * @throws {ExtractorError} When file processing fails
 *
 * @internal
 */
async function processFile(file, plugins, astVisitors, pluginContext, config, logger = new ConsoleLogger()) {
    try {
        let code = await readFile(file, 'utf-8');
        // Run onLoad hooks from plugins with error handling
        for (const plugin of plugins) {
            try {
                const result = await plugin.onLoad?.(code, file);
                if (result !== undefined) {
                    code = result;
                }
            }
            catch (err) {
                logger.warn(`Plugin ${plugin.name} onLoad failed:`, err);
                // Continue with the original code if the plugin fails
            }
        }
        // Determine parser options from file extension so .ts is not parsed as TSX
        const fileExt = extname(file).toLowerCase();
        const isTypeScriptFile = fileExt === '.ts' || fileExt === '.tsx' || fileExt === '.mts' || fileExt === '.cts';
        const isTSX = fileExt === '.tsx';
        const isJSX = fileExt === '.jsx';
        let ast;
        try {
            ast = await parse(code, {
                syntax: isTypeScriptFile ? 'typescript' : 'ecmascript',
                tsx: isTSX,
                jsx: isJSX,
                decorators: true,
                dynamicImport: true,
                comments: true,
            });
        }
        catch (err) {
            // Fallback for .ts files with JSX (already present)
            if (fileExt === '.ts' && !isTSX) {
                try {
                    ast = await parse(code, {
                        syntax: 'typescript',
                        tsx: true,
                        decorators: true,
                        dynamicImport: true,
                        comments: true,
                    });
                    logger.info?.(`Parsed ${file} using TSX fallback`);
                }
                catch (err2) {
                    throw new ExtractorError('Failed to process file', file, err2);
                }
                // Fallback for .js files with JSX
            }
            else if (fileExt === '.js' && !isJSX) {
                try {
                    ast = await parse(code, {
                        syntax: 'ecmascript',
                        jsx: true,
                        decorators: true,
                        dynamicImport: true,
                        comments: true,
                    });
                    logger.info?.(`Parsed ${file} using JSX fallback`);
                }
                catch (err2) {
                    throw new ExtractorError('Failed to process file', file, err2);
                }
            }
            else {
                throw new ExtractorError('Failed to process file', file, err);
            }
        }
        // "Wire up" the visitor's scope method to the context.
        // This avoids a circular dependency while giving plugins access to the scope.
        pluginContext.getVarFromScope = astVisitors.getVarFromScope.bind(astVisitors);
        // Pass BOTH file and code
        astVisitors.setCurrentFile(file, code);
        // 3. FIRST: Visit the AST to build scope information
        astVisitors.visit(ast);
        // 4. THEN: Extract keys from comments with scope resolution (now scope info is available)
        if (config.extract.extractFromComments !== false) {
            extractKeysFromComments(code, pluginContext, config, astVisitors.getVarFromScope.bind(astVisitors));
        }
    }
    catch (error) {
        logger.warn(`${chalk.yellow('Skipping file due to error:')} ${file}`);
        const err = error;
        const msg = typeof err?.message === 'string' && err.message.trim().length > 0
            ? err.message
            : (typeof err === 'string' ? err : '') || err?.toString?.() || 'Unknown error';
        logger.warn(`  ${msg}`);
        // If message is missing, stack is often the only useful clue
        if ((!err?.message || String(err.message).trim() === '') && err?.stack) {
            logger.warn(`  ${String(err.stack)}`);
        }
    }
}
/**
 * Simplified extraction function that returns translation results without file writing.
 * Used primarily for testing and programmatic access.
 *
 * @param config - The i18next toolkit configuration object
 * @returns Promise resolving to array of translation results
 *
 * @example
 * ```typescript
 * const results = await extract(config)
 * for (const result of results) {
 *   console.log(`${result.path}: ${result.updated ? 'Updated' : 'No changes'}`)
 * }
 * ```
 */
async function extract(config, { syncPrimaryWithDefaults = false } = {}) {
    config.extract.primaryLanguage ||= config.locales[0] || 'en';
    config.extract.secondaryLanguages ||= config.locales.filter((l) => l !== config?.extract?.primaryLanguage);
    config.extract.functions ||= ['t', '*.t'];
    config.extract.transComponents ||= ['Trans'];
    const { allKeys, objectKeys } = await findKeys(config);
    return getTranslations(allKeys, objectKeys, config, { syncPrimaryWithDefaults });
}
/**
 * Prints a promotional message for the locize saveMissing workflow.
 * This message is shown after a successful extraction that resulted in changes.
 */
async function printLocizeFunnel() {
    if (!(await shouldShowFunnel('extract')))
        return;
    console.log(chalk.yellow.bold('\n💡 Tip: Tired of running the extractor manually?'));
    console.log('   Discover a real-time "push" workflow with `saveMissing` and Locize AI,');
    console.log('   where keys are created and translated automatically as you code.');
    console.log(`   Learn more: ${chalk.cyan('https://www.locize.com/blog/i18next-savemissing-ai-automation')}`);
    console.log(`   Watch the video: ${chalk.cyan('https://youtu.be/joPsZghT3wM')}`);
    return recordFunnelShown('extract');
}

export { extract, processFile, runExtractor };
