// Copyright 2015 The Gogs Authors. All rights reserved.
// SPDX-License-Identifier: MIT

package git

import (
	"crypto/sha1"
	"encoding/hex"
	"strconv"
	"strings"
	"sync"

	"code.gitea.io/gitea/modules/util"
)

// ObjectCache provides thread-safe cache operations.
type ObjectCache[T any] struct {
	lock  sync.RWMutex
	cache map[string]T
}

func newObjectCache[T any]() *ObjectCache[T] {
	return &ObjectCache[T]{cache: make(map[string]T, 10)}
}

// Set adds obj to cache
func (oc *ObjectCache[T]) Set(id string, obj T) {
	oc.lock.Lock()
	defer oc.lock.Unlock()

	oc.cache[id] = obj
}

// Get gets cached obj by id
func (oc *ObjectCache[T]) Get(id string) (T, bool) {
	oc.lock.RLock()
	defer oc.lock.RUnlock()

	obj, has := oc.cache[id]
	return obj, has
}

// ParseBool returns the boolean value represented by the string as per git's git_config_bool
// true will be returned for the result if the string is empty, but valid will be false.
// "true", "yes", "on" are all true, true
// "false", "no", "off" are all false, true
// 0 is false, true
// Any other integer is true, true
// Anything else will return false, false
func ParseBool(value string) (result, valid bool) {
	// Empty strings are true but invalid
	if len(value) == 0 {
		return true, false
	}
	// These are the git expected true and false values
	if strings.EqualFold(value, "true") || strings.EqualFold(value, "yes") || strings.EqualFold(value, "on") {
		return true, true
	}
	if strings.EqualFold(value, "false") || strings.EqualFold(value, "no") || strings.EqualFold(value, "off") {
		return false, true
	}
	// Try a number
	intValue, err := strconv.ParseInt(value, 10, 32)
	if err != nil {
		return false, false
	}
	return intValue != 0, true
}

func HashFilePathForWebUI(s string) string {
	h := sha1.New()
	_, _ = h.Write([]byte(s))
	return hex.EncodeToString(h.Sum(nil))
}

func SplitCommitTitleBody(commitMessage string, titleRuneLimit int) (title, body string) {
	title, body, _ = strings.Cut(commitMessage, "\n")
	title, title2 := util.EllipsisTruncateRunes(title, titleRuneLimit)
	if title2 != "" {
		if body == "" {
			body = title2
		} else {
			body = title2 + "\n" + body
		}
	}
	return title, body
}
