"use strict";
// Copyright 2021-2024 Buf Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeImportPathRelative = exports.deriveImportPath = exports.relativePathRE = exports.makeImportPath = exports.rewriteImportPath = void 0;
const protobuf_1 = require("@bufbuild/protobuf");
const cache = new WeakMap();
/**
 * Apply import rewrites to the given import path, and change all .js extensions
 * to the given import extension.
 */
function rewriteImportPath(importPath, rewriteImports, importExtension) {
    let ri = cache.get(rewriteImports);
    if (ri === undefined) {
        ri = rewriteImports.map(({ pattern, target }) => {
            return {
                pattern: starToRegExp(pattern),
                target,
            };
        });
        cache.set(rewriteImports, ri);
    }
    for (const { pattern, target } of ri) {
        if (pattern.test(importPath)) {
            if (exports.relativePathRE.test(importPath)) {
                importPath =
                    target.replace(/\/$/, "") + importPath.replace(exports.relativePathRE, "/");
            }
            else {
                importPath = target;
            }
            break;
        }
    }
    if (importExtension != ".js" && importPath.endsWith(".js")) {
        importPath =
            importPath.substring(0, importPath.length - 3) + importExtension;
    }
    return importPath;
}
exports.rewriteImportPath = rewriteImportPath;
function starToRegExp(star) {
    const r = ["^"];
    for (let i = 0; i < star.length; i++) {
        switch (star[i]) {
            case "*":
                if (star[i + 1] === "*" && star[i + 2] === "/") {
                    i += 2;
                    r.push("([^\\/]+\\/)*");
                    break;
                }
                r.push("[^\\/]*");
                break;
            case ".":
            case "+":
            case "?":
            case "^":
            case "$":
            case "{":
            case "}":
            case "(":
            case ")":
            case "|":
            case "[":
            case "]":
            case "\\":
                r.push("\\", star[i]);
                break;
            default:
                r.push(star[i]);
                break;
        }
    }
    r.push("$");
    return new RegExp(r.join(""));
}
/**
 * Returns the import path for files generated by the base type generator
 * protoc-gen-es.
 */
function makeImportPath(file, bootstrapWkt, filesToGenerate) {
    // Well-known types are published with the runtime package. We usually want
    // the generated code to import them from the runtime package, with the
    // following exceptions:
    // 1. We are bootstrapping the runtime package via the plugin option
    //    "bootstrap_wkt". In that case, we cannot refer to the runtime package
    //    itself.
    // 2. We were explicitly asked to generate the well-known type.
    if (!bootstrapWkt &&
        !filesToGenerate.includes(file) &&
        protobuf_1.codegenInfo.wktSourceFiles.includes(file.name + ".proto")) {
        return protobuf_1.codegenInfo.packageName;
    }
    return "./" + file.name + "_pb.js";
}
exports.makeImportPath = makeImportPath;
exports.relativePathRE = /^\.{1,2}\//;
/**
 * Derives an ECMAScript module import path from a file path. For example,
 * the path `foo/bar.ts` is transformed into `./foo/bar.js`.
 */
function deriveImportPath(filename) {
    let importPath = filename.replace(/\.(js|ts|d.ts)$/, ".js");
    if (!exports.relativePathRE.test(importPath)) {
        importPath = "./" + importPath;
    }
    return importPath;
}
exports.deriveImportPath = deriveImportPath;
/**
 * Makes an import path relative to the file importing it. For example,
 * consider the following files:
 * - foo/foo.js
 * - baz.js
 * If foo.js wants to import baz.js, we return ../baz.js
 */
function makeImportPathRelative(importer, importPath) {
    if (!exports.relativePathRE.test(importPath)) {
        // We don't touch absolute imports, like @bufbuild/protobuf
        return importPath;
    }
    let a = importer
        .replace(/^\.\//, "")
        .split("/")
        .filter((p) => p.length > 0)
        .slice(0, -1);
    let b = importPath
        .replace(/^\.\//, "")
        .split("/")
        .filter((p) => p.length > 0);
    let matchingPartCount = 0;
    for (let l = Math.min(a.length, b.length); matchingPartCount < l; matchingPartCount++) {
        if (a[matchingPartCount] !== b[matchingPartCount]) {
            break;
        }
    }
    a = a.slice(matchingPartCount);
    b = b.slice(matchingPartCount);
    const c = a
        .map(() => "..")
        .concat(b)
        .join("/");
    return exports.relativePathRE.test(c) ? c : "./" + c;
}
exports.makeImportPathRelative = makeImportPathRelative;
