// <copyright file="EventHandler.cs" company="Selenium Committers">
// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.
// </copyright>

using System;
using System.Threading.Tasks;

namespace OpenQA.Selenium.BiDi;

internal abstract class EventHandler(string eventName)
{
    public string EventName { get; } = eventName;

    public abstract ValueTask InvokeAsync(EventArgs args);
}

internal class AsyncEventHandler<TEventArgs>(string eventName, Func<TEventArgs, Task> func)
    : EventHandler(eventName) where TEventArgs : EventArgs
{
    private readonly Func<TEventArgs, Task> _func = func;

    public override async ValueTask InvokeAsync(EventArgs args)
    {
        await _func((TEventArgs)args).ConfigureAwait(false);
    }
}

internal class SyncEventHandler<TEventArgs>(string eventName, Action<TEventArgs> action)
    : EventHandler(eventName) where TEventArgs : EventArgs
{
    private readonly Action<TEventArgs> _action = action;

    public override ValueTask InvokeAsync(EventArgs args)
    {
        _action((TEventArgs)args);

        return default;
    }
}
