/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.support;

import java.util.ArrayDeque;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public abstract class RetryableAction<Response> {
    private final Logger logger;
    private final AtomicBoolean isDone = new AtomicBoolean(false);
    private final ThreadPool threadPool;
    private final long initialDelayMillis;
    private final long maxDelayBoundMillis;
    private final long timeoutMillis;
    private final long startMillis;
    private final ActionListener<Response> finalListener;
    private final Executor executor;
    private volatile Scheduler.ScheduledCancellable retryTask;

    public RetryableAction(Logger logger, ThreadPool threadPool, TimeValue initialDelay, TimeValue timeoutValue, ActionListener<Response> listener, Executor executor) {
        this(logger, threadPool, initialDelay, TimeValue.MAX_VALUE, timeoutValue, listener, executor);
    }

    public RetryableAction(Logger logger, ThreadPool threadPool, TimeValue initialDelay, TimeValue maxDelayBound, TimeValue timeoutValue, ActionListener<Response> listener, Executor executor) {
        this.logger = logger;
        this.threadPool = threadPool;
        this.initialDelayMillis = initialDelay.getMillis();
        this.maxDelayBoundMillis = maxDelayBound.getMillis();
        if (this.initialDelayMillis < 1L) {
            throw new IllegalArgumentException("Initial delay was less than 1 millisecond: " + String.valueOf(initialDelay));
        }
        if (this.maxDelayBoundMillis < this.initialDelayMillis) {
            throw new IllegalArgumentException("Max delay bound [" + String.valueOf(maxDelayBound) + "] cannot be less than the initial delay [" + String.valueOf(initialDelay) + "]");
        }
        this.timeoutMillis = timeoutValue.getMillis();
        this.startMillis = threadPool.relativeTimeInMillis();
        this.finalListener = ActionListener.assertOnce(listener);
        this.executor = executor;
    }

    public void run() {
        RetryingListener retryingListener = new RetryingListener(this.initialDelayMillis, null);
        Runnable runnable = this.createRunnable(retryingListener);
        this.executor.execute(runnable);
    }

    public void cancel(Exception e) {
        if (this.isDone.compareAndSet(false, true)) {
            Scheduler.ScheduledCancellable localRetryTask = this.retryTask;
            if (localRetryTask != null) {
                localRetryTask.cancel();
            }
            this.onFinished();
            this.finalListener.onFailure(e);
        }
    }

    private Runnable createRunnable(RetryingListener retryingListener) {
        return new ActionRunnable<Response>(retryingListener){

            @Override
            protected void doRun() {
                RetryableAction.this.retryTask = null;
                if (!RetryableAction.this.isDone.get()) {
                    RetryableAction.this.tryAction(this.listener);
                }
            }

            @Override
            public void onRejection(Exception e) {
                RetryableAction.this.retryTask = null;
                this.onFailure(e);
            }
        };
    }

    public abstract void tryAction(ActionListener<Response> var1);

    public abstract boolean shouldRetry(Exception var1);

    protected long calculateDelayBound(long previousDelayBound) {
        return Math.min(previousDelayBound * 2L, this.maxDelayBoundMillis);
    }

    public void onFinished() {
    }

    private class RetryingListener
    implements ActionListener<Response> {
        private static final int MAX_EXCEPTIONS = 4;
        private final long delayMillisBound;
        private ArrayDeque<Exception> caughtExceptions;

        private RetryingListener(long delayMillisBound, ArrayDeque<Exception> caughtExceptions) {
            this.delayMillisBound = delayMillisBound;
            this.caughtExceptions = caughtExceptions;
        }

        @Override
        public void onResponse(Response response) {
            if (RetryableAction.this.isDone.compareAndSet(false, true)) {
                RetryableAction.this.onFinished();
                RetryableAction.this.finalListener.onResponse(response);
            }
        }

        @Override
        public void onFailure(Exception e) {
            if (RetryableAction.this.shouldRetry(e)) {
                long elapsedMillis = RetryableAction.this.threadPool.relativeTimeInMillis() - RetryableAction.this.startMillis;
                long remainingMillis = RetryableAction.this.timeoutMillis - elapsedMillis;
                if (remainingMillis <= 0L) {
                    RetryableAction.this.logger.debug(() -> Strings.format((String)"retryable action timed out after %s", (Object[])new Object[]{TimeValue.timeValueMillis((long)elapsedMillis)}), (Throwable)e);
                    this.onFinalFailure(e);
                } else {
                    long twentyPercent;
                    this.addException(e);
                    long nextDelayMillisBound = RetryableAction.this.calculateDelayBound(this.delayMillisBound);
                    RetryingListener retryingListener = new RetryingListener(nextDelayMillisBound, this.caughtExceptions);
                    Runnable runnable = RetryableAction.this.createRunnable(retryingListener);
                    int range = Math.toIntExact((this.delayMillisBound + 1L) / 2L);
                    long delayMillis = (long)Randomness.get().nextInt(range) + this.delayMillisBound - (long)range + 1L;
                    long millisExceedingTimeout = delayMillis - remainingMillis;
                    if (millisExceedingTimeout > 0L && millisExceedingTimeout > (twentyPercent = (long)((double)RetryableAction.this.timeoutMillis * 0.2))) {
                        int tenPercent = Math.toIntExact((long)((double)RetryableAction.this.timeoutMillis * 0.1));
                        int delayBeyondTimeout = Randomness.get().nextInt(tenPercent) + tenPercent;
                        delayMillis = remainingMillis + (long)delayBeyondTimeout;
                    }
                    assert (delayMillis > 0L);
                    if (!RetryableAction.this.isDone.get()) {
                        TimeValue delay = TimeValue.timeValueMillis((long)delayMillis);
                        RetryableAction.this.logger.debug(() -> Strings.format((String)"retrying action that failed in %s", (Object[])new Object[]{delay}), (Throwable)e);
                        try {
                            RetryableAction.this.retryTask = RetryableAction.this.threadPool.schedule(runnable, delay, RetryableAction.this.executor);
                        }
                        catch (EsRejectedExecutionException ree) {
                            this.onFinalFailure(ree);
                        }
                    }
                }
            } else {
                this.onFinalFailure(e);
            }
        }

        public String toString() {
            return this.getClass().getName() + "/" + String.valueOf(RetryableAction.this.finalListener);
        }

        private void onFinalFailure(Exception e) {
            this.addException(e);
            if (RetryableAction.this.isDone.compareAndSet(false, true)) {
                RetryableAction.this.onFinished();
                RetryableAction.this.finalListener.onFailure(this.buildFinalException());
            }
        }

        private Exception buildFinalException() {
            Exception suppressed;
            Exception topLevel = this.caughtExceptions.removeFirst();
            while ((suppressed = this.caughtExceptions.pollFirst()) != null) {
                topLevel.addSuppressed(suppressed);
            }
            return topLevel;
        }

        private void addException(Exception e) {
            if (this.caughtExceptions != null) {
                if (this.caughtExceptions.size() == 4) {
                    this.caughtExceptions.removeLast();
                }
            } else {
                this.caughtExceptions = new ArrayDeque(4);
            }
            this.caughtExceptions.addFirst(e);
        }
    }
}

