/*******************************************************************************
 * Copyright (c) 1998, 2010 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the 
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0 
 * which accompanies this distribution. 
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at 
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     11/06/2009-2.0 Guy Pelletier 
 *       - 286317: UniqueConstraint xml element is changing (plus couple other fixes, see bug)
 ******************************************************************************/  
package org.eclipse.persistence.internal.jpa.metadata.mappings;

import org.eclipse.persistence.exceptions.ValidationException;
import org.eclipse.persistence.internal.jpa.metadata.MetadataDescriptor;
import org.eclipse.persistence.internal.jpa.metadata.MetadataHelper;
import org.eclipse.persistence.internal.jpa.metadata.MetadataLogger;
import org.eclipse.persistence.internal.jpa.metadata.ORMetadata;
import org.eclipse.persistence.internal.jpa.metadata.accessors.MetadataAccessor;
import org.eclipse.persistence.internal.jpa.metadata.accessors.mappings.MappingAccessor;
import org.eclipse.persistence.internal.jpa.metadata.accessors.objects.MetadataAccessibleObject;
import org.eclipse.persistence.internal.jpa.metadata.accessors.objects.MetadataAnnotation;
import org.eclipse.persistence.mappings.CollectionMapping;

/**
 * Object to hold onto map key metadata.
 * 
 * @author Guy Pelletier
 * @since EclipseLink 2.0
 */
public class MapKeyMetadata extends ORMetadata {
    private String m_name;

    /**
     * INTERNAL:
     */
    public MapKeyMetadata() {
        super("<map-key>");
    }

    /**
     * INTERNAL:
     */
    public MapKeyMetadata(MetadataAccessibleObject accessibleObject) {
        super(null, accessibleObject);
    }
    
    /**
     * INTERNAL:
     */
    public MapKeyMetadata(MetadataAnnotation mapKey, MetadataAccessibleObject accessibleObject) {
        super(mapKey, accessibleObject);

        m_name = (String) mapKey.getAttributeString("name");
    }
    
    /**
     * INTERNAL:
     * Return true if a name has been specified.
     */
    public boolean hasName() {
        return m_name != null && ! m_name.equals("");
    }
    
    /**
     * INTERNAL:
     * Process a map key for a 1-M or M-M mapping. Will return the map key
     * method name that should be use, null otherwise.
     */
    public String process(CollectionMapping mapping, MappingAccessor mappingAccessor) {
        MetadataDescriptor referenceDescriptor = mappingAccessor.getReferenceDescriptor();
        MetadataLogger logger = mappingAccessor.getLogger();
        
        if ((! hasName()) && referenceDescriptor.hasCompositePrimaryKey()) {
            // No persistent property or field name has been provided, and the 
            // reference class has a composite primary key class.  Return null,
            // internally, EclipseLink will use an instance of the composite 
            // primary key class as the map key.
            return null;
        } else {
            // A persistent property or field name may have have been provided. 
            // If one has not we will default to the primary key of the reference 
            // class. The primary key cannot be composite at this point.
            String fieldOrPropertyName = MetadataHelper.getName(m_name, referenceDescriptor.getIdAttributeName(), logger.MAP_KEY_ATTRIBUTE_NAME, logger, mappingAccessor.getAnnotatedElementName());
    
            // Look up the referenceAccessor
            MetadataAccessor referenceAccessor = referenceDescriptor.getAccessorFor(fieldOrPropertyName);
            if (referenceAccessor == null) {
                // 266912: relax validation for MappedSuperclass descriptors when the map key is an unresolved generic type
                if (referenceDescriptor.isMappedSuperclass()) {
                    return null;
                } else {
                    throw ValidationException.couldNotFindMapKey(fieldOrPropertyName, referenceDescriptor.getJavaClass(), mapping);
                }
            }
        
            return referenceAccessor.getAccessibleObjectName();
        }
    }
    
    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public String getName() {
        return m_name;
    }

    /**
     * INTERNAL:
     * Used for OX mapping.
     */
    public void setName(String name) {
        m_name = name;
    }
}
