/*********************************************************************
* Copyright (c) 2005, 2019 SAP SE
*
* This program and the accompanying materials are made
* available under the terms of the Eclipse Public License 2.0
* which is available at https://www.eclipse.org/legal/epl-2.0/
*
* Contributors:
*    SAP SE - initial API, implementation and documentation
*    mwenz - Bug 505659 - NullPointerException below RenameActionProvider.fillContextMenu
*
* SPDX-License-Identifier: EPL-2.0
**********************************************************************/
package org.eclipse.graphiti.examples.common;

import java.io.IOException;
import java.util.Collections;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.edit.command.CommandParameter;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.graphiti.examples.common.util.T;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.navigator.CommonActionProvider;
import org.eclipse.ui.navigator.ICommonMenuConstants;

public class RenameActionProvider extends CommonActionProvider {

	@Override
	public void fillContextMenu(IMenuManager menu) {
		super.fillContextMenu(menu);
		ISelection selection = getContext().getSelection();
		if (!selection.isEmpty()) {
			if (selection instanceof IStructuredSelection) {
				IStructuredSelection sel = (IStructuredSelection) selection;
				Object el = sel.getFirstElement();
				if (el instanceof EClass) {
					EClass eclass = (EClass) el;
					String platformString = eclass.eResource().getURI().toPlatformString(true);
					if (platformString != null) {
						Path path = new Path(platformString);
						IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
						if (file != null) {
							IProject project = file.getProject();
							try {
								if (project.hasNature(ExampleProjectNature.NATURE_ID)) {
									menu.appendToGroup(ICommonMenuConstants.GROUP_ADDITIONS, getAction(eclass));
								}
							} catch (CoreException e) {
								T.racer().error("Error while checking project nature for project " + project.getName(),
										e);
							}
						}
					}
				}
			}
		}
	}

	private IAction getAction(EClass eo) {
		return new RenameAction(eo);
	}

	private static class RenameAction extends Action {

		private EClass eclass;

		public RenameAction(EClass eo) {
			this.eclass = eo;
		}

		@Override
		public void run() {
			InputDialog inputDialog = new InputDialog(null, Messages.RenameActionProvider_ProvideNameTitle, Messages.RenameActionProvider_ProvideNameDescription, eclass.getName(), null);
			int open = inputDialog.open();
			if (open == Dialog.OK) {
				String newName = inputDialog.getValue();
				Resource resource = eclass.eResource();
				ResourceSet resourceSet = resource.getResourceSet();
				TransactionalEditingDomain domain = TransactionalEditingDomain.Factory.INSTANCE.createEditingDomain(resourceSet);
				try{
				if (domain != null){
					Command setCommand = domain.createCommand(SetCommand.class, new CommandParameter(eclass,
							EcorePackage.Literals.ENAMED_ELEMENT__NAME, newName));
					domain.getCommandStack().execute(setCommand);
					try {
						resource.save(Collections.emptyMap());
					} catch (IOException e) {
						e.printStackTrace();
					}
				}
				}finally{
					domain.dispose();
				}
			}
		}

		@Override
		public String getText() {
			return Messages.RenameActionProvider_RenameEClassText;
		}
	}

}
