/**
 * Copyright (c) 2008, 2015, 2021 Anatoliy Tischenko, Borland Software Corporation, CEA LIST, Artal and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Anatoliy Tischenko - Initial API and implementation
 * Artem Tikhomirov (Borland) - initial API and implementation
 * Michael Golubev (Montages) - #386838 - migrate to Xtend2
 * Etienne Allogo (ARTAL) - etienne.allogo@artal.fr - Bug 569174 : 1.4 Merge papyrus extension templates into codegen.xtend
 * Etienne Allogo (ARTAL) - etienne.allogo@artal.fr - Bug 569174 : L1.2 clean up providers
 */
package metamodel;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.Arrays;
import org.eclipse.emf.codegen.ecore.genmodel.GenClass;
import org.eclipse.emf.codegen.ecore.genmodel.GenClassifier;
import org.eclipse.emf.codegen.ecore.genmodel.GenEnum;
import org.eclipse.emf.codegen.ecore.genmodel.GenFeature;
import org.eclipse.emf.codegen.ecore.genmodel.impl.GenClassImpl;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;
import xpt.GenModelUtils_qvto;

@Singleton
@SuppressWarnings("all")
public class MetaModel {
  @Inject
  @Extension
  private MetaModel_qvto _metaModel_qvto;

  @Inject
  @Extension
  private GenModelUtils_qvto _genModelUtils_qvto;

  /**
   * Does instanceof check.
   */
  public CharSequence IsInstance(final GenClass xptSelf, final String accessor) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(accessor);
    _builder.append(" instanceof ");
    String _qualifiedInterfaceName = this._metaModel_qvto.getQualifiedInterfaceName(xptSelf);
    _builder.append(_qualifiedInterfaceName);
    return _builder;
  }

  /**
   * Shorthand, negates IsInstance - handy if you consider
   * different approaches for generated and dynamic models: false == instanceof
   *  vs. !MetaModelFacility.isInstance
   */
  public CharSequence NotInstance(final GenClass xptSelf, final String accessor) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("false == ");
    CharSequence _IsInstance = this.IsInstance(xptSelf, accessor);
    _builder.append(_IsInstance);
    return _builder;
  }

  /**
   * Special kind of instanceof check, that compares eContainer() of the object.
   * Since metaClass may be an external interface, eContainer() might need cast to EObject
   */
  public CharSequence IsContainerInstance(final GenClass it, final String _object, final GenClass metaClass) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _eObjectFeature = this.getEObjectFeature(metaClass, _object, "eContainer()");
    _builder.append(_eObjectFeature);
    _builder.append(" instanceof ");
    String _qualifiedInterfaceName = this._metaModel_qvto.getQualifiedInterfaceName(it);
    _builder.append(_qualifiedInterfaceName);
    return _builder;
  }

  protected CharSequence getEObjectFeature(final GenClass it, final String _object, final String feature) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isExternalInterface = it.isExternalInterface();
      if (_isExternalInterface) {
        _builder.append("((org.eclipse.emf.ecore.EObject) ");
        _builder.append(_object);
        _builder.append(").");
        _builder.append(feature);
      } else {
        _builder.append(_object);
        _builder.append(".");
        _builder.append(feature);
      }
    }
    return _builder;
  }

  public CharSequence getFeatureValue(final GenFeature it, final String containerVar, final GenClass containerClass) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isExternalInterface = containerClass.isExternalInterface();
      if (_isExternalInterface) {
        _builder.append("((");
        CharSequence _featureTargetType = this.featureTargetType(it);
        _builder.append(_featureTargetType);
        _builder.append(") ((org.eclipse.emf.ecore.EObject) ");
        _builder.append(containerVar);
        _builder.append(").eGet(");
        CharSequence _MetaFeature = this.MetaFeature(it);
        _builder.append(_MetaFeature);
        _builder.append("))");
      } else {
        _builder.append(containerVar);
        _builder.append(".");
        String _getAccessor = it.getGetAccessor();
        _builder.append(_getAccessor);
        _builder.append("()");
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public CharSequence featureTargetType(final GenFeature it) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isListType = it.isListType();
      if (_isListType) {
        _builder.append("java.util.List");
      } else {
        CharSequence _QualifiedClassName = this.QualifiedClassName(it.getGenClass().getGenPackage().getGenModel().findGenClassifier(it.getEcoreFeature().getEType()));
        _builder.append(_QualifiedClassName);
      }
    }
    return _builder;
  }

  /**
   * FIXME leave only one version of these two getFeatureValue
   * 
   * A slightly more sophisticated version of getFeatureValue().
   * @param containerName the name of the container
   * @param feature the feature whose value is in interest
   * @param containerMetaClass the <code>GenClass</code> of the container, or <code>null</code>, if the container is declared as an <code>org.eclipse.emf.ecore.EObject</code>.
   * @param needsCastToResultType whether the cast to the result type is required (this parameter is only used if the <code>org.eclipse.emf.ecore.EClass</code> this feature belongs to is an external interface).
   */
  public CharSequence getFeatureValue(final GenFeature it, final String containerVar, final GenClass container, final boolean needsCastToResultType) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isExternalInterface = it.getGenClass().isExternalInterface();
      if (_isExternalInterface) {
        {
          if (needsCastToResultType) {
            _builder.append("((");
            CharSequence _featureTargetType = this.featureTargetType(it);
            _builder.append(_featureTargetType);
            _builder.append(") ");
          }
        }
        String _parenthesizedCast = this._metaModel_qvto.parenthesizedCast(containerVar, container, null);
        _builder.append(_parenthesizedCast);
        _builder.append(".eGet(");
        CharSequence _MetaFeature = this.MetaFeature(it);
        _builder.append(_MetaFeature);
        _builder.append(")");
        {
          if (needsCastToResultType) {
            _builder.append(")");
          }
        }
      } else {
        String _parenthesizedCast_1 = this._metaModel_qvto.parenthesizedCast(containerVar, container, it.getGenClass());
        _builder.append(_parenthesizedCast_1);
        _builder.append(".");
        String _getAccessor = it.getGetAccessor();
        _builder.append(_getAccessor);
        _builder.append("()");
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public CharSequence modifyFeature(final GenFeature it, final String targetVar, final GenClass targetType, final String value) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isListType = it.isListType();
      if (_isListType) {
        CharSequence _featureValue = this.getFeatureValue(it, targetVar, targetType);
        _builder.append(_featureValue);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append(".add(");
        _builder.append(value, "\t");
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      } else {
        CharSequence _setFeatureValue = this.setFeatureValue(it, targetVar, targetType, value);
        _builder.append(_setFeatureValue);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  public CharSequence replaceFeatureValue(final GenFeature it, final String targetVar, final GenClass targetType, final String oldValue, final String newValue) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isListType = it.isListType();
      if (_isListType) {
        CharSequence _featureValue = this.getFeatureValue(it, targetVar, targetType);
        _builder.append(_featureValue);
        _builder.append(".remove(");
        _builder.append(oldValue);
        _builder.append(");");
      }
    }
    _builder.newLineIfNotEmpty();
    CharSequence _modifyFeature = this.modifyFeature(it, targetVar, targetType, newValue);
    _builder.append(_modifyFeature);
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public CharSequence moveFeatureValue(final GenFeature it, final String oldTarget, final String newTarget, final GenClass targetType, final String value) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isListType = it.isListType();
      if (_isListType) {
        CharSequence _featureValue = this.getFeatureValue(it, oldTarget, targetType);
        _builder.append(_featureValue);
        _builder.append(".remove(");
        _builder.append(value);
        _builder.append(");");
      } else {
        CharSequence _setFeatureValue = this.setFeatureValue(it, oldTarget, targetType, "null");
        _builder.append(_setFeatureValue);
        _builder.append(";");
      }
    }
    _builder.newLineIfNotEmpty();
    CharSequence _modifyFeature = this.modifyFeature(it, newTarget, targetType, value);
    _builder.append(_modifyFeature);
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public CharSequence setFeatureValue(final GenFeature it, final String targetVar, final GenClass targetType, final String valueVar) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _setFeatureValue = this.setFeatureValue(it, targetVar, targetType, valueVar, false);
    _builder.append(_setFeatureValue);
    return _builder;
  }

  public CharSequence setFeatureValue(final GenFeature it, final String targetVar, final GenClass targetType, final String valueVar, final boolean isPlainObjectValue) {
    CharSequence _xifexpression = null;
    boolean _isExternalInterface = targetType.isExternalInterface();
    if (_isExternalInterface) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("((org.eclipse.emf.ecore.EObject) ");
      _builder.append(targetVar);
      _builder.append(").eSet(");
      CharSequence _MetaFeature = this.MetaFeature(it);
      _builder.append(_MetaFeature);
      _builder.append(", ");
      _builder.append(valueVar);
      _builder.append(")");
      _xifexpression = _builder;
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append(targetVar);
      _builder_1.append(".set");
      String _accessorName = it.getAccessorName();
      _builder_1.append(_accessorName);
      _builder_1.append("(");
      CharSequence _setFeatureArgs = this.setFeatureArgs(it, valueVar, isPlainObjectValue);
      _builder_1.append(_setFeatureArgs);
      _builder_1.append(")");
      _xifexpression = _builder_1;
    }
    return _xifexpression;
  }

  protected CharSequence setFeatureArgs(final GenFeature it, final String valueVar, final boolean isPlainObjectValue) {
    CharSequence _xifexpression = null;
    if ((!isPlainObjectValue)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(valueVar);
      _xifexpression = _builder;
    } else {
      CharSequence _xifexpression_1 = null;
      boolean _isPrimitiveType = this._genModelUtils_qvto.isPrimitiveType(it);
      if (_isPrimitiveType) {
        StringConcatenation _builder_1 = new StringConcatenation();
        CharSequence _unwrapObjectToPrimitiveValue = this.unwrapObjectToPrimitiveValue(it, valueVar);
        _builder_1.append(_unwrapObjectToPrimitiveValue);
        _xifexpression_1 = _builder_1;
      } else {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("(");
        CharSequence _featureTargetType = this.featureTargetType(it);
        _builder_2.append(_featureTargetType);
        _builder_2.append(")");
        _builder_2.append(valueVar);
        _xifexpression_1 = _builder_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  protected CharSequence unwrapObjectToPrimitiveValue(final GenFeature it, final String valueVar) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("((");
    CharSequence _featureTargetType = this.featureTargetType(it);
    _builder.append(_featureTargetType);
    _builder.append(") ");
    _builder.append(valueVar);
    _builder.append(").");
    String _instanceClassName = it.getEcoreFeature().getEType().getInstanceClassName();
    _builder.append(_instanceClassName);
    _builder.append("Value()");
    return _builder;
  }

  public CharSequence MetaClass(final GenClassifier it) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedPackageInterfaceName = this._metaModel_qvto.getQualifiedPackageInterfaceName(it.getGenPackage());
    _builder.append(_qualifiedPackageInterfaceName);
    _builder.append(".eINSTANCE.get");
    String _classifierAccessorName = this._metaModel_qvto.getClassifierAccessorName(it);
    _builder.append(_classifierAccessorName);
    _builder.append("()");
    return _builder;
  }

  public CharSequence MetaFeature(final GenFeature it) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedPackageInterfaceName = this._metaModel_qvto.getQualifiedPackageInterfaceName(it.getGenClass().getGenPackage());
    _builder.append(_qualifiedPackageInterfaceName);
    _builder.append(".eINSTANCE.get");
    String _featureAccessorName = this._metaModel_qvto.getFeatureAccessorName(it);
    _builder.append(_featureAccessorName);
    _builder.append("()");
    return _builder;
  }

  /**
   * SomeFactory.eINSTANCE.createBlaBla();
   * NB: for map entries, the resulting type is EObject, not the qualified interface name. If cast is needed, use (un)parenthesizedCast() extension.
   * @see GenClassImpl#hasFactoryInterfaceCreateMethod() for details why map entries should be treated differently
   */
  public CharSequence NewInstance(final GenClass it) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isMapEntry = it.isMapEntry();
      if (_isMapEntry) {
        String _qualifiedFactoryInterfaceName = this._metaModel_qvto.getQualifiedFactoryInterfaceName(it.getGenPackage());
        _builder.append(_qualifiedFactoryInterfaceName);
        _builder.append(".");
        String _factoryInstanceName = this._metaModel_qvto.getFactoryInstanceName(it.getGenPackage());
        _builder.append(_factoryInstanceName);
        _builder.append(".create(");
        CharSequence _MetaClass = this.MetaClass(it);
        _builder.append(_MetaClass);
        _builder.append(")");
      } else {
        _builder.newLineIfNotEmpty();
        String _qualifiedFactoryInterfaceName_1 = this._metaModel_qvto.getQualifiedFactoryInterfaceName(it.getGenPackage());
        _builder.append(_qualifiedFactoryInterfaceName_1);
        _builder.append(".");
        String _factoryInstanceName_1 = this._metaModel_qvto.getFactoryInstanceName(it.getGenPackage());
        _builder.append(_factoryInstanceName_1);
        _builder.append(".create");
        String _name = it.getEcoreClass().getName();
        _builder.append(_name);
        _builder.append("()");
      }
    }
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * // XXX Need to take into account possible GenClass from generated and always available code
   * // e.g. Notation or Ecore
   * // FIXME be consistent on final line feed - e.g. NewInstance adds a LF, while modifyFeature not, hence together they look odd.
   */
  public CharSequence NewInstance(final GenClass it, final String varName) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedInterfaceName = this._metaModel_qvto.getQualifiedInterfaceName(it);
    _builder.append(_qualifiedInterfaceName);
    _builder.append(" ");
    _builder.append(varName);
    _builder.append(" = ");
    {
      boolean _isMapEntry = it.isMapEntry();
      if (_isMapEntry) {
        _builder.append("(");
        String _qualifiedInterfaceName_1 = this._metaModel_qvto.getQualifiedInterfaceName(it);
        _builder.append(_qualifiedInterfaceName_1);
        _builder.append(") ");
      }
    }
    CharSequence _NewInstance = this.NewInstance(it);
    _builder.append(_NewInstance);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * Ensures value is of type EObject, may be no-op if context GenClass is compatible with EObject.
   * Note, injected value is not surrounded with parenthesis, may need to introduce another
   * template to accomplish that if needed.
   */
  public CharSequence DowncastToEObject(final GenClass it, final String value) {
    StringConcatenation _builder = new StringConcatenation();
    {
      boolean _isExternalInterface = it.isExternalInterface();
      if (_isExternalInterface) {
        _builder.append("(org.eclipse.emf.ecore.EObject) ");
      }
    }
    _builder.append(value);
    return _builder;
  }

  /**
   * Declares new variable of appropriate type and assigns casted value to it.
   */
  public CharSequence DeclareAndAssign(final GenClass it, final String assignee, final String value) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedInterfaceName = this._metaModel_qvto.getQualifiedInterfaceName(it);
    _builder.append(_qualifiedInterfaceName);
    _builder.append(" ");
    _builder.append(assignee);
    _builder.append(" = (");
    String _qualifiedInterfaceName_1 = this._metaModel_qvto.getQualifiedInterfaceName(it);
    _builder.append(_qualifiedInterfaceName_1);
    _builder.append(") ");
    _builder.append(value);
    _builder.append(";");
    return _builder;
  }

  /**
   * third boolean parameter is to indicate the value is not EObject, so may
   * need extra cast in case dynamic model instances are in use.
   */
  protected CharSequence _DeclareAndAssign(final GenClass it, final String assignee, final String value, final boolean isPlainObjectValue) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _DeclareAndAssign = this.DeclareAndAssign(it, assignee, value);
    _builder.append(_DeclareAndAssign);
    return _builder;
  }

  protected CharSequence _DeclareAndAssign(final GenClassifier it, final String assignee, final String value, final boolean isPlainObjectValue) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedClassName = this._metaModel_qvto.getQualifiedClassName(it);
    _builder.append(_qualifiedClassName);
    _builder.append(" ");
    _builder.append(assignee);
    _builder.append(" = (");
    String _qualifiedClassName_1 = this._metaModel_qvto.getQualifiedClassName(it);
    _builder.append(_qualifiedClassName_1);
    _builder.append(") ");
    _builder.append(value);
    _builder.append(";");
    return _builder;
  }

  /**
   * @see IsContainerInstance
   */
  public CharSequence DeclareAndAssignContainer(final GenClass it, final String assignee, final String _object, final GenClass metaClass) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedInterfaceName = this._metaModel_qvto.getQualifiedInterfaceName(it);
    _builder.append(_qualifiedInterfaceName);
    _builder.append(" ");
    _builder.append(assignee);
    _builder.append(" = (");
    String _qualifiedInterfaceName_1 = this._metaModel_qvto.getQualifiedInterfaceName(it);
    _builder.append(_qualifiedInterfaceName_1);
    _builder.append(") ");
    CharSequence _eObjectFeature = this.getEObjectFeature(metaClass, _object, "eContainer()");
    _builder.append(_eObjectFeature);
    _builder.append(";");
    return _builder;
  }

  /**
   * Declares new variable of context type and assignes a value obtained from 'src',
   * which is of type 'srcMetaClass', via 'srcFeature'
   * 
   * XXX in certain scenarions may need extra cast of the feature value
   */
  public CharSequence DeclareAndAssign(final GenClass it, final String assignee, final String src, final GenClass srcMetaClass, final GenFeature srcFeature) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedInterfaceName = this._metaModel_qvto.getQualifiedInterfaceName(it);
    _builder.append(_qualifiedInterfaceName);
    _builder.append(" ");
    _builder.append(assignee);
    _builder.append(" = ");
    CharSequence _featureValue = this.getFeatureValue(srcFeature, src, srcMetaClass);
    _builder.append(_featureValue);
    _builder.append(";");
    return _builder;
  }

  /**
   * Same as DeclareAndAssign, with extra operation applied to source object
   */
  public CharSequence DeclareAndAssign2(final GenClass it, final String assignee, final String src, final GenClass srcMetaClass, final GenFeature srcFeature, final String srcExt, final boolean needCast) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedInterfaceName = this._metaModel_qvto.getQualifiedInterfaceName(it);
    _builder.append(_qualifiedInterfaceName);
    _builder.append(" ");
    _builder.append(assignee);
    _builder.append(" = ");
    CharSequence _featureValue = this.getFeatureValue(srcFeature, src, srcMetaClass);
    _builder.append(_featureValue);
    _builder.append(".");
    _builder.append(srcExt);
    _builder.append(";");
    return _builder;
  }

  /**
   * Cast value of type EObject to specific type. Would be no-op with dynamic model instances,
   * therefore, the fact eObjectValue is actually EObject is essential
   */
  public CharSequence CastEObject(final GenClass xptSelf, final String eObjectValue) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("(");
    String _qualifiedInterfaceName = this._metaModel_qvto.getQualifiedInterfaceName(xptSelf);
    _builder.append(_qualifiedInterfaceName);
    _builder.append(") ");
    _builder.append(eObjectValue);
    return _builder;
  }

  /**
   * Qualified interface name of the generated EClass, or EObject for dynamic models.
   * Use whenever class name is inevitable (e.g. method arguments)
   * SHOULD NEVER APPEAR in instanceof or any other similar comparison operation
   */
  protected CharSequence _QualifiedClassName(final GenClass xptSelf) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedInterfaceName = xptSelf.getQualifiedInterfaceName();
    _builder.append(_qualifiedInterfaceName);
    return _builder;
  }

  protected CharSequence _QualifiedClassName(final GenEnum xptSelf) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedName = xptSelf.getQualifiedName();
    _builder.append(_qualifiedName);
    return _builder;
  }

  protected CharSequence _QualifiedClassName(final GenClassifier xptSelf) {
    StringConcatenation _builder = new StringConcatenation();
    String _qualifiedClassName = this._metaModel_qvto.getQualifiedClassName(xptSelf);
    _builder.append(_qualifiedClassName);
    return _builder;
  }

  @XbaseGenerated
  public CharSequence DeclareAndAssign(final GenClassifier it, final String assignee, final String value, final boolean isPlainObjectValue) {
    if (it instanceof GenClass) {
      return _DeclareAndAssign((GenClass)it, assignee, value, isPlainObjectValue);
    } else if (it != null) {
      return _DeclareAndAssign(it, assignee, value, isPlainObjectValue);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(it, assignee, value, isPlainObjectValue).toString());
    }
  }

  @XbaseGenerated
  public CharSequence QualifiedClassName(final GenClassifier xptSelf) {
    if (xptSelf instanceof GenEnum) {
      return _QualifiedClassName((GenEnum)xptSelf);
    } else if (xptSelf instanceof GenClass) {
      return _QualifiedClassName((GenClass)xptSelf);
    } else if (xptSelf != null) {
      return _QualifiedClassName(xptSelf);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(xptSelf).toString());
    }
  }
}
