/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 *   Ansgar Radermacher (CEA LIST) - Larger refactoring
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.IASTNode;
import org.eclipse.cdt.core.dom.ast.IASTParameterDeclaration;
import org.eclipse.cdt.core.dom.ast.IASTStandardFunctionDeclarator;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateDeclaration;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateParameter;
import org.eclipse.cdt.core.model.IMethodDeclaration;
import org.eclipse.cdt.core.model.IMethodTemplateDeclaration;
import org.eclipse.cdt.core.parser.IToken;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Const;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.ConstInit;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Friend;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Inline;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Variadic;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Virtual;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Volatile;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.profile.standard.Create;
import org.eclipse.uml2.uml.profile.standard.Destroy;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * functions to create and update methods
 */
@SuppressWarnings("all")
public class MethodUtils {
  /**
   * Create a UML operation from a CDT method declaration
   */
  public static Object createMethod(final IMethodDeclaration method, final org.eclipse.uml2.uml.Class classifier) {
    Object _xblockexpression = null;
    {
      IASTFunctionDeclarator declarator = ASTUtils.getDeclarator(method);
      Object _xifexpression = null;
      if ((declarator != null)) {
        Object _xblockexpression_1 = null;
        {
          final Operation op = classifier.createOwnedOperation(method.getElementName(), null, null);
          ReverseUtils.setXmlID(op);
          _xblockexpression_1 = MethodUtils.updateMethod(classifier, op, method);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }

  /**
   * Update an existing operation from a CDT method declaration
   */
  public static Object updateMethod(final org.eclipse.uml2.uml.Class classifier, final Operation op, final IMethodDeclaration method) {
    try {
      Object _xblockexpression = null;
      {
        op.setName(method.getElementName());
        StereotypeUtil.unapply(op, Inline.class);
        StereotypeUtil.unapply(op, Friend.class);
        StereotypeUtil.unapply(op, Virtual.class);
        StereotypeUtil.unapply(op, Volatile.class);
        StereotypeUtil.unapply(op, Create.class);
        StereotypeUtil.unapply(op, Destroy.class);
        StereotypeUtil.unapply(op, Const.class);
        op.getOwnedParameters().clear();
        List<String> keywords = MethodUtils.getKeywords(method);
        String body = MethodUtils.getBody(method);
        op.setVisibility(ASTUtils.convertVisibility(method.getVisibility()));
        op.setIsStatic(method.isStatic());
        ReverseUtils.applyStereotype(op, method.isInline(), Inline.class);
        ReverseUtils.applyStereotype(op, method.isFriend(), Friend.class);
        ReverseUtils.applyStereotype(op, method.isVirtual(), Virtual.class);
        ReverseUtils.applyStereotype(op, method.isVolatile(), Volatile.class);
        ReverseUtils.applyStereotype(op, method.isConst(), Const.class);
        boolean _isVirtual = method.isVirtual();
        if (_isVirtual) {
          boolean _isPureVirtual = method.isPureVirtual();
          if (_isPureVirtual) {
            op.setIsAbstract(true);
          }
        }
        ReverseUtils.applyStereotype(op, method.isConstructor(), Create.class);
        ReverseUtils.applyStereotype(op, method.isDestructor(), Destroy.class);
        try {
          String signature = method.getSignature();
          Pattern pattern = Pattern.compile("([\\s]*)(\\.\\.\\.)([\\s]*)(\\))");
          Matcher matcher = pattern.matcher(signature);
          ReverseUtils.applyStereotype(op, matcher.find(), Variadic.class);
        } catch (final Throwable _t) {
          if (_t instanceof Exception) {
            final Exception e = (Exception)_t;
            e.printStackTrace();
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
        if ((method instanceof IMethodTemplateDeclaration)) {
          for (int i = 0; (i < ((List<String>)Conversions.doWrapArray(((IMethodTemplateDeclaration)method).getTemplateParameterTypes())).size()); i++) {
            GetOrCreateP1.getOrCreateTemplateParameter(op, ((IMethodTemplateDeclaration)method).getTemplateParameterTypes()[i], keywords.get(i));
          }
        }
        IASTDeclSpecifier declSpecifier = ASTUtils.getDeclSpecifier(method);
        IASTStandardFunctionDeclarator declarator = ASTUtils.getDeclarator(method);
        if (((!method.isConstructor()) && (!method.isDestructor()))) {
          String _returnType = method.getReturnType();
          boolean _tripleNotEquals = (_returnType != null);
          if (_tripleNotEquals) {
            Type parameterType = MethodUtils.getParameterTemplateType(op, method, ASTUtils.getQualifiedName(declSpecifier));
            if ((parameterType == null)) {
              parameterType = ReverseUtils.getUMLType(ASTUtils.getQualifiedName(declSpecifier), method);
            }
            Parameter ret = op.createOwnedParameter("ret", parameterType);
            ret.setDirection(ParameterDirectionKind.RETURN_LITERAL);
            ReverseUtils.analyzeDeclaration(declarator, ret.getType(), ret, ReverseUtils.Cpp_LangID);
            PkgDependencies.createDependency(classifier, ret.getType());
            if ((((parameterType != null) && parameterType.getName().equals("void")) && ret.getAppliedStereotypes().isEmpty())) {
              ret.destroy();
            }
            ReverseUtils.applyStereotype(ret, declSpecifier.isConst(), Const.class);
            ReverseUtils.applyStereotype(ret, declSpecifier.isVolatile(), Volatile.class);
          } else {
            Parameter ret_1 = op.createOwnedParameter("ret", ReverseUtils.getUMLType("void", method));
            ret_1.setDirection(ParameterDirectionKind.RETURN_LITERAL);
            ReverseUtils.analyzeDeclaration(declarator, ret_1.getType(), ret_1, ReverseUtils.Cpp_LangID);
            boolean _isEmpty = ret_1.getAppliedStereotypes().isEmpty();
            if (_isEmpty) {
              ret_1.destroy();
            }
          }
        }
        IASTParameterDeclaration[] _parameters = declarator.getParameters();
        for (final IASTParameterDeclaration param : _parameters) {
          {
            Type parameterType_1 = MethodUtils.getParameterTemplateType(op, method, ASTUtils.getCppTypeName(param.getDeclSpecifier()));
            if ((parameterType_1 == null)) {
              parameterType_1 = ReverseUtils.getUMLType(param.getDeclSpecifier(), method);
            }
            final Parameter opParam = op.createOwnedParameter(param.getDeclarator().getName().toString(), parameterType_1);
            ReverseUtils.applyStereotype(opParam, param.getDeclSpecifier().isConst(), Const.class);
            ReverseUtils.analyzeDeclaration(param.getDeclarator(), opParam.getType(), opParam, ReverseUtils.Cpp_LangID);
            ReverseUtils.applyStereotype(opParam, param.getDeclSpecifier().isVolatile(), Volatile.class);
          }
        }
        if ((body != null)) {
          boolean _isConstructor = method.isConstructor();
          if (_isConstructor) {
            String initStr = ReverseUtils.getMemberInit(method);
            boolean _isEmpty_1 = initStr.isEmpty();
            boolean _not = (!_isEmpty_1);
            if (_not) {
              StereotypeUtil.apply(op, ConstInit.class);
              ConstInit _stereotypeApplication = UMLUtil.<ConstInit>getStereotypeApplication(op, ConstInit.class);
              _stereotypeApplication.setInitialisation(initStr);
            }
          }
          OpaqueBehavior ob = null;
          int _size = op.getMethods().size();
          boolean _equals = (_size == 0);
          if (_equals) {
            Behavior _createOwnedBehavior = classifier.createOwnedBehavior(op.getName(), UMLPackage.Literals.OPAQUE_BEHAVIOR);
            ob = ((OpaqueBehavior) _createOwnedBehavior);
            ob.setSpecification(op);
            ob.setIsReentrant(false);
          } else {
            Behavior _get = op.getMethods().get(0);
            ob = ((OpaqueBehavior) _get);
            boolean _equals_1 = ob.getName().equals(op.getName());
            boolean _not_1 = (!_equals_1);
            if (_not_1) {
              ob.setName(op.getName());
            }
          }
          int _size_1 = ob.getBodies().size();
          boolean _equals_2 = (_size_1 == 0);
          if (_equals_2) {
            ob.getLanguages().add(ReverseUtils.Cpp_LangID);
            ob.getBodies().add("");
          }
          for (int i = 0; (i < ob.getLanguages().size()); i++) {
            boolean _equals_3 = ob.getLanguages().get(i).equals(ReverseUtils.Cpp_LangID);
            if (_equals_3) {
              int _size_2 = ob.getBodies().size();
              boolean _lessThan = (i < _size_2);
              if (_lessThan) {
                ob.getBodies().set(i, body);
              }
            }
          }
        }
        final IASTNode node = ASTUtils.findEnclosingNode(method);
        CommentUtils.addComment(op, node);
        Object _xifexpression = null;
        if ((node instanceof IASTFunctionDefinition)) {
          _xifexpression = null;
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public static Type getParameterTemplateType(final Operation op, final IMethodDeclaration imethod, final String typeName) {
    Type ret = null;
    if ((imethod instanceof IMethodTemplateDeclaration)) {
      final Function1<String, Boolean> _function = (String it) -> {
        return Boolean.valueOf(it.equals(typeName));
      };
      boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<String>filter(((Iterable<String>)Conversions.doWrapArray(((IMethodTemplateDeclaration)imethod).getTemplateParameterTypes())), _function));
      boolean _not = (!_isEmpty);
      if (_not) {
        ret = GetOrCreateP1.getOrCreateTemplateParameter(op, typeName, "class");
      }
    }
    return ret;
  }

  public static List<String> getKeywords(final IMethodDeclaration method) {
    try {
      IASTNode node = ASTUtils.findEnclosingNode(method);
      List<String> keywords = new ArrayList<String>();
      if ((node instanceof ICPPASTTemplateDeclaration)) {
        IASTNode[] _children = ((ICPPASTTemplateDeclaration)node).getChildren();
        for (final IASTNode child : _children) {
          if ((child instanceof ICPPASTTemplateParameter)) {
            IToken token = ((ICPPASTTemplateParameter)child).getSyntax();
            String keyword = "class";
            while ((token != null)) {
              {
                int _type = token.getType();
                boolean _equals = (_type == IToken.t_typename);
                if (_equals) {
                  keyword = "typename";
                }
                token = token.getNext();
              }
            }
            keywords.add(keyword);
          }
        }
      }
      return keywords;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public static String getBody(final IMethodDeclaration method) {
    IASTFunctionDeclarator declarator = null;
    IASTNode node = ASTUtils.findEnclosingNode(method);
    String body = null;
    if ((node instanceof ICPPASTFunctionDefinition)) {
      declarator = ((ICPPASTFunctionDefinition) node).getDeclarator();
      body = BatchReverseFunctionBody.getBody(method.getTranslationUnit(), ((ICPPASTFunctionDefinition) node));
    } else {
      if ((node instanceof IASTFunctionDeclarator)) {
      }
    }
    return body;
  }
}
