package org.eclipse.papyrus.designer.languages.cpp.reverse;

import com.google.common.collect.Iterables;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Level;
import org.eclipse.cdt.core.dom.ast.ASTTypeUtil;
import org.eclipse.cdt.core.dom.ast.ExpansionOverlapsBoundaryException;
import org.eclipse.cdt.core.dom.ast.IASTCompositeTypeSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclaration;
import org.eclipse.cdt.core.dom.ast.IASTElaboratedTypeSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTEnumerationSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.IASTName;
import org.eclipse.cdt.core.dom.ast.IASTNamedTypeSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTNode;
import org.eclipse.cdt.core.dom.ast.IASTNodeSelector;
import org.eclipse.cdt.core.dom.ast.IASTSimpleDeclaration;
import org.eclipse.cdt.core.dom.ast.IASTStandardFunctionDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTTranslationUnit;
import org.eclipse.cdt.core.dom.ast.IBinding;
import org.eclipse.cdt.core.dom.ast.IField;
import org.eclipse.cdt.core.dom.ast.IProblemBinding;
import org.eclipse.cdt.core.dom.ast.ITypedef;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTNameSpecifier;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTNamespaceDefinition;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTQualifiedName;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateDeclaration;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateId;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateParameter;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTVisibilityLabel;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPBinding;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPSpecialization;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPTemplateDefinition;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPTemplateInstance;
import org.eclipse.cdt.core.model.ISourceRange;
import org.eclipse.cdt.core.model.ISourceReference;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.parser.IToken;
import org.eclipse.cdt.core.parser.ast.ASTAccessVisibility;
import org.eclipse.cdt.internal.core.dom.parser.cpp.ICPPUnknownMemberClass;
import org.eclipse.cdt.internal.core.model.ASTStringUtil;
import org.eclipse.uml2.uml.VisibilityKind;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class ASTUtils {
  /**
   * Obtain a declSpecifier from a source reference
   */
  public static IASTDeclSpecifier getDeclSpecifier(final ISourceReference sourceRef) {
    return ASTUtils.getDeclSpecifier(ASTUtils.findEnclosingNode(sourceRef));
  }

  /**
   * Obtain a declSpecifier from a node
   */
  public static IASTDeclSpecifier getDeclSpecifier(final IASTNode node) {
    if ((node instanceof IASTDeclSpecifier)) {
      return ((IASTDeclSpecifier)node);
    } else {
      if ((node instanceof IASTFunctionDefinition)) {
        return ((IASTFunctionDefinition)node).getDeclSpecifier();
      } else {
        if ((node instanceof IASTSimpleDeclaration)) {
          return ((IASTSimpleDeclaration)node).getDeclSpecifier();
        } else {
          if ((node instanceof ICPPASTTemplateDeclaration)) {
            return ASTUtils.getDeclSpecifier(((ICPPASTTemplateDeclaration)node).getDeclaration());
          }
        }
      }
    }
    return null;
  }

  /**
   * Obtain a function declaration (typically for operation analysis) from a source reference
   */
  public static IASTStandardFunctionDeclarator getDeclarator(final ISourceReference sourceRef) {
    IASTNode node = ASTUtils.findEnclosingNode(sourceRef);
    if ((node instanceof ICPPASTFunctionDefinition)) {
      IASTFunctionDeclarator _declarator = ((ICPPASTFunctionDefinition)node).getDeclarator();
      return ((IASTStandardFunctionDeclarator) _declarator);
    } else {
      if ((node instanceof IASTSimpleDeclaration)) {
        return IterableExtensions.<IASTStandardFunctionDeclarator>head(Iterables.<IASTStandardFunctionDeclarator>filter(((Iterable<?>)Conversions.doWrapArray(((IASTSimpleDeclaration)node).getDeclarators())), IASTStandardFunctionDeclarator.class));
      } else {
        if ((node instanceof ICPPASTTemplateDeclaration)) {
          IASTDeclaration declaration = ((ICPPASTTemplateDeclaration)node).getDeclaration();
          if ((declaration instanceof IASTSimpleDeclaration)) {
            return IterableExtensions.<IASTStandardFunctionDeclarator>head(Iterables.<IASTStandardFunctionDeclarator>filter(((Iterable<?>)Conversions.doWrapArray(((IASTSimpleDeclaration)declaration).getDeclarators())), IASTStandardFunctionDeclarator.class));
          }
        }
      }
    }
    return null;
  }

  /**
   * get a node-selector from a translation unit. The selector is cached to improve
   * performance. It skips already indexed header files
   */
  public static IASTNodeSelector getSelector(final ITranslationUnit unit) {
    try {
      IASTTranslationUnit ast = null;
      final Map<ITranslationUnit, IASTTranslationUnit> tuToASTtuMap = ReverseData.current.tuToASTtuMap;
      if ((unit != null)) {
        IASTTranslationUnit _get = tuToASTtuMap.get(unit);
        boolean _tripleNotEquals = (_get != null);
        if (_tripleNotEquals) {
          return tuToASTtuMap.get(unit).getNodeSelector(null);
        } else {
          ast = unit.getAST(ReverseData.current.index, ITranslationUnit.AST_SKIP_INDEXED_HEADERS);
          if ((ast != null)) {
            tuToASTtuMap.put(unit, ast);
            return ast.getNodeSelector(null);
          }
        }
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * get the AST node associated with a source reference
   */
  public static IASTNode findEnclosingNode(final ISourceReference sourceRef) {
    try {
      IASTNodeSelector selector = ASTUtils.getSelector(sourceRef.getTranslationUnit());
      ISourceRange range = sourceRef.getSourceRange();
      if ((selector != null)) {
        return selector.findEnclosingNode(range.getStartPos(), range.getLength());
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Get a qualified name from a source reference via the AST node
   * The function is based on the qualified names in bindings, obtained
   * via decl-specifiers or AST names-
   * In case of a function definition, the parent binding is used in order
   * to obtain the type to which the method belongs.
   * If no suitable name is obtained, null is returned.
   */
  public static String getQualifiedName(final ISourceReference sourceRef) {
    IASTNode node = ASTUtils.findEnclosingNode(sourceRef);
    if ((node instanceof IASTName)) {
      return ASTUtils.getQualifiedName(((IASTName)node));
    }
    if ((node instanceof ICPPASTNamespaceDefinition)) {
      return ASTUtils.getQualifiedName(((ICPPASTNamespaceDefinition)node).getName());
    }
    if ((node instanceof ICPPASTTemplateDeclaration)) {
      node = ((ICPPASTTemplateDeclaration)node).getDeclaration();
    }
    if ((node instanceof IASTFunctionDefinition)) {
      final IBinding binding = ((IASTFunctionDefinition)node).getDeclarator().getName().resolveBinding().getOwner();
      if ((binding instanceof ICPPBinding)) {
        return ASTTypeUtil.getQualifiedName(((ICPPBinding)binding));
      }
    }
    final IASTDeclSpecifier declSpecifier = ASTUtils.getDeclSpecifier(node);
    if ((declSpecifier != null)) {
      return ASTUtils.getQualifiedName(declSpecifier);
    }
    return null;
  }

  /**
   * return the qualified name of a declaration based on its
   * declaration specifier
   */
  public static String getQualifiedName(final IASTDeclSpecifier declSpecifier) {
    if ((declSpecifier instanceof IASTNamedTypeSpecifier)) {
      return ASTUtils.getQualifiedName(((IASTNamedTypeSpecifier)declSpecifier).getName());
    } else {
      if ((declSpecifier instanceof IASTElaboratedTypeSpecifier)) {
        return ASTUtils.getQualifiedName(((IASTElaboratedTypeSpecifier)declSpecifier).getName());
      } else {
        if ((declSpecifier instanceof IASTCompositeTypeSpecifier)) {
          return ASTUtils.getQualifiedName(((IASTCompositeTypeSpecifier)declSpecifier).getName());
        } else {
          if ((declSpecifier instanceof IASTEnumerationSpecifier)) {
            return ASTUtils.getQualifiedName(((IASTEnumerationSpecifier)declSpecifier).getName());
          } else {
            return ReverseUtils.getCppTypeName(
              ASTStringUtil.getSignatureString(declSpecifier, null));
          }
        }
      }
    }
  }

  /**
   * CDT resolves bindings completely, i.e. references to "using" names are
   * resolved to their right-hand definition
   * Therefore recover the first segment
   */
  public static String getQNameOfUsing(final IASTName name) {
    if ((name instanceof ICPPASTQualifiedName)) {
      final ICPPASTNameSpecifier segment = ((ICPPASTQualifiedName)name).getAllSegments()[0];
      if ((segment instanceof IASTName)) {
        final IASTName segLastName = ((IASTName)segment).getLastName();
        return ASTUtils.getQualifiedName(segLastName);
      }
    }
    return null;
  }

  /**
   * get the qualified name from an AST name via a binding.
   * If it is a template, get binding of template itself (no arguments)
   */
  public static String getQualifiedName(final IASTName name) {
    final IASTName lastName = name.getLastName();
    IBinding binding = null;
    if ((lastName instanceof ICPPASTTemplateId)) {
      binding = ((ICPPASTTemplateId)lastName).getTemplateName().resolveBinding();
    } else {
      binding = name.resolveBinding();
    }
    if ((binding instanceof ICPPUnknownMemberClass)) {
      binding = ((ICPPUnknownMemberClass) binding).getOwner();
    }
    return ASTUtils.getQualifiedName(binding);
  }

  /**
   * Get a binding from a C++ name-specifier. If it is a template, get binding
   * of template itself (no arguments).
   * IASTName is no common supertype
   */
  public static String getQualifiedNameFromNSpec(final ICPPASTNameSpecifier nameSpecifier) {
    if ((nameSpecifier instanceof ICPPASTTemplateId)) {
      return ASTUtils.getQualifiedName(((ICPPASTTemplateId)nameSpecifier).getTemplateName().resolveBinding());
    } else {
      if ((nameSpecifier instanceof IASTName)) {
        return ASTUtils.getQualifiedName(((IASTName)nameSpecifier));
      } else {
        return ASTUtils.getQualifiedName(nameSpecifier.resolveBinding());
      }
    }
  }

  /**
   * return a qualified name from a binding from. If the binding cannot
   * be resolved, return the name as string
   */
  public static String getQualifiedName(final IBinding binding) {
    if ((binding instanceof IProblemBinding)) {
      ReverseUtils.LOGGER.log(Level.WARNING, String.format("binding <%s> is problematic", binding));
      int _size = ((List<IBinding>)Conversions.doWrapArray(((IProblemBinding)binding).getCandidateBindings())).size();
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        return ASTUtils.getQualifiedName(((IProblemBinding)binding).getCandidateBindings()[0]);
      }
    }
    if ((binding instanceof IField)) {
      return ASTUtils.getQualifiedName(((IField)binding).getCompositeTypeOwner());
    }
    if ((binding instanceof ICPPTemplateInstance)) {
      ICPPTemplateDefinition _templateDefinition = ((ICPPTemplateInstance)binding).getTemplateDefinition();
      return ASTUtils.getQualifiedName(((IBinding) _templateDefinition));
    }
    if ((binding instanceof ICPPSpecialization)) {
      return ASTUtils.getQualifiedName(((ICPPSpecialization)binding).getSpecializedBinding());
    }
    if ((binding instanceof ITypedef)) {
    }
    if ((binding instanceof ICPPBinding)) {
      return ASTUtils.stripFNHint(ASTTypeUtil.getQualifiedName(((ICPPBinding)binding)));
    } else {
      return binding.getName();
    }
  }

  /**
   * This functions removes a file-name hint that is eventually present in
   * a qualified name. The ASTTypeUtil adds it in enclosed with { .. } to detect
   * recursively calls to the same binding.
   */
  public static String stripFNHint(final String qName) {
    if ((qName != null)) {
      int idx = qName.lastIndexOf("{");
      if ((idx != (-1))) {
        if ((idx > 2)) {
          int _idx = idx;
          idx = (_idx - 2);
        }
        int endIdx = qName.indexOf("}", idx);
        if ((endIdx != (-1))) {
          int _length = qName.length();
          int _minus = (_length - 2);
          boolean _lessThan = (endIdx < _minus);
          if (_lessThan) {
            int _endIdx = endIdx;
            endIdx = (_endIdx + 2);
          }
          String _substring = qName.substring(0, idx);
          String _substring_1 = qName.substring((endIdx + 1));
          final String strippedName = (_substring + _substring_1);
          return ASTUtils.stripFNHint(strippedName);
        }
      }
    }
    return qName;
  }

  /**
   * get the target translation unit from a name
   */
  public static ITranslationUnit getTargetTU(final IASTName name) {
    return ReverseUtils.getTranslationUnitFromPath(name.getContainingFilename(), ReverseData.current.project);
  }

  /**
   * return the qualified name of a template type
   */
  public static String getTemplateType(final IASTName astName, final String name) {
    IASTNode parent = astName;
    while ((parent != null)) {
      {
        if ((parent instanceof ICPPASTTemplateDeclaration)) {
          ICPPASTTemplateParameter[] _templateParameters = ((ICPPASTTemplateDeclaration)parent).getTemplateParameters();
          for (final ICPPASTTemplateParameter parameter : _templateParameters) {
            String _string = parameter.toString();
            boolean _equals = Objects.equals(_string, name);
            if (_equals) {
              IASTDeclaration _declaration = ((ICPPASTTemplateDeclaration)parent).getDeclaration();
              final String templateQN = ASTUtils.getQualifiedName(((IASTSimpleDeclaration) _declaration).getDeclSpecifier());
              return ((templateQN + "::") + name);
            }
          }
        }
        parent = parent.getParent();
      }
    }
    return name;
  }

  /**
   * Get the C++ typename from an AST decl specifier
   * This includes all modifiers and eventual template parameters
   */
  public static String getCppTypeName(final IASTDeclSpecifier declarator) {
    String parameterTypeName = "";
    try {
      IToken token = declarator.getSyntax();
      while ((token != null)) {
        {
          String tokenStr = token.toString();
          boolean _equals = tokenStr.equals("*");
          if (_equals) {
          } else {
            boolean _equals_1 = tokenStr.equals("&");
            if (_equals_1) {
            } else {
              boolean _equals_2 = tokenStr.equals("const");
              if (_equals_2) {
              } else {
                int _length = parameterTypeName.length();
                boolean _greaterThan = (_length > 0);
                if (_greaterThan) {
                  String _parameterTypeName = parameterTypeName;
                  parameterTypeName = (_parameterTypeName + " ");
                }
                String _parameterTypeName_1 = parameterTypeName;
                parameterTypeName = (_parameterTypeName_1 + tokenStr);
              }
            }
          }
          token = token.getNext();
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof ExpansionOverlapsBoundaryException) {
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return parameterTypeName;
  }

  /**
   * Convert AST visibilities ...
   */
  public static VisibilityKind convertVisibility(final ASTAccessVisibility visibility) {
    boolean _equals = Objects.equals(visibility, ASTAccessVisibility.PRIVATE);
    if (_equals) {
      return VisibilityKind.PRIVATE_LITERAL;
    }
    boolean _equals_1 = Objects.equals(visibility, ASTAccessVisibility.PROTECTED);
    if (_equals_1) {
      return VisibilityKind.PROTECTED_LITERAL;
    }
    return VisibilityKind.PUBLIC_LITERAL;
  }

  /**
   * Yes, there are two AST ways for visibilities ...
   */
  public static VisibilityKind convertVisibility(final int visibility) {
    if ((visibility == ICPPASTVisibilityLabel.v_private)) {
      return VisibilityKind.PRIVATE_LITERAL;
    }
    if ((visibility == ICPPASTVisibilityLabel.v_protected)) {
      return VisibilityKind.PROTECTED_LITERAL;
    }
    return VisibilityKind.PUBLIC_LITERAL;
  }
}
