/**
 * Copyright (c) 2022, 2023 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Mohamed Harkat - Initial API and implementation
 *   Ansgar Radermacher - Integration and bug fixes
 */
package org.eclipse.papyrus.designer.languages.python.codegen.gen;

import java.util.Objects;
import org.eclipse.papyrus.designer.infra.base.StringConstants;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.IntegerRange;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class PyAttributes {
  /**
   * Assign the value of the attribute depending on its multiplicity. If the attribute has an infinite multiplicity
   * the function returns an empty list. If the multiplicity number is specified, a list of the same size will be
   * created containing all the attributes
   */
  public static String assignAttributeValue(final Property property) {
    boolean _isMultivalued = property.isMultivalued();
    boolean _not = (!_isMultivalued);
    if (_not) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(" ");
      _builder.append("= ");
      String _attributeValue = PyAttributes.getAttributeValue(property);
      _builder.append(_attributeValue, " ");
      return _builder.toString();
    } else {
      int _upperBound = property.upperBound();
      boolean _equals = (_upperBound == (-1));
      if (_equals) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(" ");
        _builder_1.append("= []");
        return _builder_1.toString();
      } else {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append(" ");
        _builder_2.append("= [");
        {
          int _upper = property.getUpper();
          IntegerRange _upTo = new IntegerRange(0, _upper);
          boolean _hasElements = false;
          for(final Integer i : _upTo) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder_2.appendImmediate(", ", " ");
            }
            String _attributeValue_1 = PyAttributes.getAttributeValue(property);
            _builder_2.append(_attributeValue_1, " ");
            _builder_2.append("()");
          }
        }
        _builder_2.append("]");
        return _builder_2.toString();
      }
    }
  }

  /**
   * Same as the previous method, but it treats composite attributes. For multiple instances, the constructor
   * of the attribute will be called multiple times in a list
   */
  public static String assignCompositeAttributes(final Property property) {
    boolean _isMultivalued = property.isMultivalued();
    boolean _not = (!_isMultivalued);
    if (_not) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(" ");
      _builder.append("= ");
      String _name = property.getType().getName();
      _builder.append(_name, " ");
      _builder.append("()");
      return _builder.toString();
    } else {
      int _upperBound = property.upperBound();
      boolean _equals = (_upperBound == (-1));
      if (_equals) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(" ");
        _builder_1.append("= []");
        return _builder_1.toString();
      } else {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append(" ");
        _builder_2.append("= [");
        {
          int _upper = property.getUpper();
          IntegerRange _upTo = new IntegerRange(0, _upper);
          boolean _hasElements = false;
          for(final Integer i : _upTo) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder_2.appendImmediate(", ", " ");
            }
            String _name_1 = property.getType().getName();
            _builder_2.append(_name_1, " ");
            _builder_2.append("()");
          }
        }
        _builder_2.append("]");
        return _builder_2.toString();
      }
    }
  }

  /**
   * A simple method that adds a type definition for an attribute
   */
  public static String assignType(final Property property) {
    Type _type = property.getType();
    boolean _tripleEquals = (_type == null);
    if (_tripleEquals) {
      return " : None";
    }
    String _name = property.getType().getName();
    boolean _matched = false;
    String _name_1 = property.getType().getName();
    boolean _equals = Objects.equals(_name_1, "Integer");
    if (_equals) {
      _matched=true;
      return ": int";
    }
    if (!_matched) {
      String _name_2 = property.getType().getName();
      boolean _equals_1 = Objects.equals(_name_2, "Boolean");
      if (_equals_1) {
        _matched=true;
        return ": bool";
      }
    }
    if (!_matched) {
      String _name_3 = property.getType().getName();
      boolean _equals_2 = Objects.equals(_name_3, "Real");
      if (_equals_2) {
        _matched=true;
        return ": float";
      }
    }
    if (!_matched) {
      String _name_4 = property.getType().getName();
      boolean _equals_3 = Objects.equals(_name_4, "String");
      if (_equals_3) {
        _matched=true;
        return ": str";
      }
    }
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(": ");
    String _name_5 = property.getType().getName();
    _builder.append(_name_5);
    return _builder.toString();
  }

  /**
   * A method that assigns the value of the attribute, depending on if the value is assigned in the model or not.
   */
  public static String getAttributeValue(final Property property) {
    if (((property.getDefaultValue() != null) && (property.getType() != null))) {
      String _name = property.getType().getName();
      boolean _matched = false;
      if ((Objects.equals(property.getType().getName(), "bool") || Objects.equals(property.getType().getName(), "Boolean"))) {
        _matched=true;
        return StringExtensions.toFirstUpper(property.getDefaultValue().stringValue());
      }
      if (!_matched) {
        if (Objects.equals(_name, "byte")) {
          _matched=true;
          String _stringValue = property.getDefaultValue().stringValue();
          String _plus = ("bytes(" + _stringValue);
          return (_plus + ")");
        }
      }
      if (!_matched) {
        Type _type = property.getType();
        if ((_type instanceof Enumeration)) {
          _matched=true;
          String _name_1 = property.getType().getName();
          String _plus_1 = (_name_1 + ".");
          String _stringValue_1 = property.getDefaultValue().stringValue();
          return (_plus_1 + _stringValue_1);
        }
      }
      return property.getDefaultValue().stringValue();
    } else {
      Type _type_1 = property.getType();
      boolean _tripleEquals = (_type_1 == null);
      if (_tripleEquals) {
        return "None";
      }
      String _name_2 = property.getType().getName();
      boolean _matched_1 = false;
      if ((Objects.equals(property.getType().getName(), "int") || Objects.equals(property.getType().getName(), "Integer"))) {
        _matched_1=true;
        return "0";
      }
      if (!_matched_1) {
        if ((Objects.equals(property.getType().getName(), "bool") || Objects.equals(property.getType().getName(), "Boolean"))) {
          _matched_1=true;
          return "False";
        }
      }
      if (!_matched_1) {
        if ((Objects.equals(property.getType().getName(), "float") || Objects.equals(property.getType().getName(), "Real"))) {
          _matched_1=true;
          return "0.0";
        }
      }
      if (!_matched_1) {
        if ((Objects.equals(property.getType().getName(), "str") || Objects.equals(property.getType().getName(), "String"))) {
          _matched_1=true;
          return "\'\'";
        }
      }
      if (!_matched_1) {
        if (Objects.equals(_name_2, "complex")) {
          _matched_1=true;
          return "0+0j";
        }
      }
      if (!_matched_1) {
        if (Objects.equals(_name_2, "byte")) {
          _matched_1=true;
          return "0x00";
        }
      }
      if (!_matched_1) {
        if (Objects.equals(_name_2, "tuple")) {
          _matched_1=true;
          return "()";
        }
      }
      if (!_matched_1) {
        if (Objects.equals(_name_2, "set")) {
          _matched_1=true;
          return "set()";
        }
      }
      if (!_matched_1) {
        if (Objects.equals(_name_2, "list")) {
          _matched_1=true;
          return "[]";
        }
      }
      if (!_matched_1) {
        if (Objects.equals(_name_2, "dict")) {
          _matched_1=true;
          return "{}";
        }
      }
      if (!_matched_1) {
        if (Objects.equals(_name_2, "range")) {
          _matched_1=true;
          return "rang(0)";
        }
      }
      return "None";
    }
  }

  /**
   * A method that write the visibility of the attribute.
   * __ for private, _ for protected and nothing for public
   */
  public static String writeAttributeVisibility(final Property attribute) {
    String _string = attribute.getVisibility().toString();
    if (_string != null) {
      switch (_string) {
        case "private":
          return "__";
        case "protected":
          return "_";
        case "package":
          return "_";
        default:
          return StringConstants.EMPTY;
      }
    } else {
      return StringConstants.EMPTY;
    }
  }
}
