/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Ansgar Radermacher - Initial API and implementation
 */
package org.eclipse.papyrus.designer.languages.idl.codegen;

import java.util.Objects;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.languages.common.base.GenUtils;
import org.eclipse.papyrus.designer.languages.idl.codegen.preferences.IDLCodeGenUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.DirectedRelationship;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class GenIDL2 {
  private static final String KEY_STEREOTYPE = "IDLprofile::Key";

  public static CharSequence genIDL2(final Classifier cl) {
    StringConcatenation _builder = new StringConcatenation();
    String _commentHeader = IDLCodeGenUtils.getCommentHeader();
    _builder.append(_commentHeader);
    _builder.newLineIfNotEmpty();
    {
      if ((cl instanceof Interface)) {
        CharSequence _genIDL2intf = GenIDL2.genIDL2intf(((Interface) cl));
        _builder.append(_genIDL2intf);
        _builder.append(" ");
        _builder.newLineIfNotEmpty();
      } else {
        if ((cl instanceof DataType)) {
          CharSequence _genIDL2dataType = GenIDL2.genIDL2dataType(((DataType) cl));
          _builder.append(_genIDL2dataType);
          _builder.newLineIfNotEmpty();
        }
      }
    }
    return _builder;
  }

  public static CharSequence getIncludes(final Classifier cl) {
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<DirectedRelationship> _sourceDirectedRelationships = cl.getSourceDirectedRelationships();
      for(final DirectedRelationship relShip : _sourceDirectedRelationships) {
        {
          Element _get = relShip.getTargets().get(0);
          if ((_get instanceof Classifier)) {
            _builder.append("#include \"");
            Element _get_1 = relShip.getTargets().get(0);
            String _includeFilename = GenIDL2.includeFilename(((NamedElement) _get_1));
            _builder.append(_includeFilename);
            _builder.append("\"");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    return _builder;
  }

  public static String includeFilename(final NamedElement ne) {
    String _fullPath = GenUtils.getFullPath(ne.getNearestPackage());
    String _plus = (_fullPath + "/");
    String _name = ne.getName();
    String _plus_1 = (_plus + _name);
    String _plus_2 = (_plus_1 + ".");
    String _iDLSuffix = IDLCodeGenUtils.getIDLSuffix();
    return (_plus_2 + _iDLSuffix);
  }

  public static CharSequence genIDL2intf(final Interface intf) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("// need to include referenced types (assuming a naming convention) ");
    _builder.newLine();
    _builder.append("#ifndef D_");
    String _name = intf.getName();
    _builder.append(_name);
    _builder.newLineIfNotEmpty();
    _builder.append("#define D_");
    String _name_1 = intf.getName();
    _builder.append(_name_1);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    CharSequence _includes = GenIDL2.getIncludes(intf);
    _builder.append(_includes);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    CharSequence _openNamespaceIDL = Utils.openNamespaceIDL(intf);
    _builder.append(_openNamespaceIDL);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      EList<Classifier> _generals = intf.getGenerals();
      boolean _hasElements = false;
      for(final Classifier general : _generals) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        String _qualifiedName = general.getQualifiedName();
        _builder.append(_qualifiedName);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("{");
    _builder.newLine();
    {
      EList<Operation> _ownedOperations = intf.getOwnedOperations();
      for(final Operation operation : _ownedOperations) {
        {
          EList<Type> _raisedExceptions = operation.getRaisedExceptions();
          boolean _hasElements_1 = false;
          for(final Type raisedException : _raisedExceptions) {
            if (!_hasElements_1) {
              _hasElements_1 = true;
            } else {
              _builder.appendImmediate(", ", "\t");
            }
            _builder.append("\t");
            String _qualifiedName_1 = raisedException.getQualifiedName();
            _builder.append(_qualifiedName_1, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    }
    {
      EList<Property> _attributes = intf.getAttributes();
      for(final Property attribute : _attributes) {
        _builder.append("\t");
        _builder.append("attribute ");
        String _qualifiedName_2 = attribute.getType().getQualifiedName();
        _builder.append(_qualifiedName_2, "\t");
        _builder.append(" ");
        String _name_2 = attribute.getName();
        _builder.append(_name_2, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("};");
    _builder.newLine();
    _builder.newLine();
    CharSequence _closeNamespaceIDL = Utils.closeNamespaceIDL(intf);
    _builder.append(_closeNamespaceIDL);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("#endif");
    _builder.newLine();
    return _builder;
  }

  /**
   * Generate IDL2 from a datatype. It will respect the stereotype "Key" on one of the attributes
   * and produce an associated pragma.
   */
  public static CharSequence genIDL2dataType(final DataType dataType) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _openNamespaceIDL = Utils.openNamespaceIDL(dataType);
    _builder.append(_openNamespaceIDL);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("struct ");
    String _name = dataType.getName();
    _builder.append(_name);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    {
      EList<Property> _attributes = dataType.getAttributes();
      for(final Property attribute : _attributes) {
        _builder.append("\t");
        String _name_1 = attribute.getType().getName();
        _builder.append(_name_1, "\t");
        _builder.append(" ");
        String _name_2 = attribute.getName();
        _builder.append(_name_2, "\t");
        _builder.append(";");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("};");
    _builder.newLine();
    {
      final Function1<Property, Boolean> _function = (Property it) -> {
        return Boolean.valueOf(StereotypeUtil.isApplied(it, GenIDL2.KEY_STEREOTYPE));
      };
      int _size = IterableExtensions.size(IterableExtensions.<Property>filter(dataType.getAttributes(), _function));
      boolean _greaterThan = (_size > 0);
      if (_greaterThan) {
        _builder.append("#pragma keylist ");
        String _name_3 = dataType.getName();
        _builder.append(_name_3);
        _builder.append(" ");
        {
          final Function1<Property, Boolean> _function_1 = (Property it) -> {
            return Boolean.valueOf(StereotypeUtil.isApplied(it, GenIDL2.KEY_STEREOTYPE));
          };
          Iterable<Property> _filter = IterableExtensions.<Property>filter(dataType.getAttributes(), _function_1);
          for(final Property attribute_1 : _filter) {
            String _name_4 = attribute_1.getName();
            _builder.append(_name_4);
          }
        }
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    CharSequence _closeNamespaceIDL = Utils.closeNamespaceIDL(dataType);
    _builder.append(_closeNamespaceIDL);
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  public static CharSequence IDLparameters(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    {
      final Function1<Parameter, Boolean> _function = (Parameter it) -> {
        ParameterDirectionKind _direction = it.getDirection();
        return Boolean.valueOf((!Objects.equals(_direction, ParameterDirectionKind.RETURN_LITERAL)));
      };
      Iterable<Parameter> _filter = IterableExtensions.<Parameter>filter(operation.getOwnedParameters(), _function);
      boolean _hasElements = false;
      for(final Parameter parameter : _filter) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        ParameterDirectionKind _direction = parameter.getDirection();
        _builder.append(_direction);
        _builder.append(" ");
        String _name = parameter.getType().getName();
        _builder.append(_name);
        _builder.append(" ");
        String _name_1 = parameter.getName();
        _builder.append(_name_1);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }

  public static CharSequence IDLretType(final Operation operation) {
    StringConcatenation _builder = new StringConcatenation();
    {
      Type _type = operation.getType();
      boolean _tripleEquals = (_type == null);
      if (_tripleEquals) {
        _builder.append("void");
        _builder.newLine();
      } else {
        String _name = operation.getType().getName();
        _builder.append(_name);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder;
  }
}
