/*****************************************************************************
 * Copyright (c) 2022 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 *****************************************************************************/
 
 package org.eclipse.papyrus.designer.transformation.library.xtend

import org.eclipse.papyrus.designer.languages.common.profile.Codegen.NoCodeGen
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.External
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Typedef
import org.eclipse.papyrus.designer.transformation.extensions.ITextTemplate
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil
import org.eclipse.uml2.uml.Model
import org.eclipse.uml2.uml.NamedElement
import org.eclipse.uml2.uml.Namespace
import org.eclipse.uml2.uml.Operation
import org.eclipse.uml2.uml.Parameter
import org.eclipse.uml2.uml.ParameterDirectionKind
import org.eclipse.uml2.uml.PrimitiveType
import org.eclipse.uml2.uml.Type
import org.eclipse.uml2.uml.util.UMLUtil

import static extension org.eclipse.papyrus.designer.uml.tools.utils.ElementUtils.usedNamespaces
import static extension org.eclipse.papyrus.designer.uml.tools.utils.OperationUtils.parametersNonRet

class CppUtils implements ITextTemplate {

	/**
	 * create the C++ signature for an operation (including parenthesis)
	 */
	static def cppSignature(Operation operation) '''
		«operation.name»(«FOR parameter : operation.ownedParameters SEPARATOR(', ')»
			«parameter.cppParameter»
		«ENDFOR»
	'''


	/**
	 * make a C++ call, pass all parameters except the return parameter
	 */
	 static def cppCall(Operation operation) '''
		«operation.name»(«FOR parameter : operation.parametersNonRet SEPARATOR(', ')»
			«parameter.name»
		«ENDFOR»)
	'''

	/**
	 *
	 * @param ne
	 * @return
	 */
	static def cppQName(NamedElement ne) {
		if (StereotypeUtil.isApplied(ne, External) || StereotypeUtil.isApplied(ne, NoCodeGen)) {
			return ne.name
		}
		else {
			var qName = ne.name;
			for (Namespace ns : ne.allNamespaces()) {
				if (!(ns instanceof Model)) {
					qName = '''«ns.name»::«qName»'''
				}
			}
			return qName
		}
	}
	
	/**
	 *
	 * @param type
	 *            a type
	 * @return return the definition of a typedef, if the type has been defined via
	 *         the stereotype CppType of the Cpp profile
	 */
	static def dereferenceTypedef(Type type) {
		if (type instanceof PrimitiveType) {
			val cppType = UMLUtil.getStereotypeApplication(type, Typedef)
			if (cppType !== null) {
				cppType.definition
			}
		}
		return type.qualifiedName

	}
	
	/**
	 * make a C++ call, pass all parameters except the return parameter, prefix with "return",
	 * if there is a return type in the operations declaration
	 */
	static def returnCppCall(Operation operation) '''
		«IF (operation.type !== null)»return «ENDIF»«operation.cppCall»
	'''
	
	static def cppParameter(Parameter parameter) '''
		«parameter.type»«IF (parameter.direction == ParameterDirectionKind.OUT)»_out«ENDIF» «parameter.name»
	'''

	static def cppType(Type type) '''
		«IF (type.qualifiedName == 'UMLPrimitiveTypes::Boolean')»
			bool
		«ELSEIF (type.qualifiedName == 'UMLPrimitiveTypes::Integer')»
			int
		«ELSE»
			«dereferenceTypedef(type)»
		«ENDIF»
	'''


	static def cppRetType(Operation operation) '''
		«IF (operation.type === null)»
			void
		«ELSE»
			«operation.type.cppType»
		«ENDIF»
	'''

	/**
	 * Open a set of C++ namespaces associated with the packages of of the passed named element
	 * TODO: use indentTab? => requires making this script recursive
	 * Need to include referenced types (assuming a naming convention?
	 */
	static def openNamespace(NamedElement namedElement) '''
		«FOR ns : namedElement.usedNamespaces.reverse»
			namespace «ns.name»
		«ENDFOR»
	'''

	/**
	 * Close a set of C++ namespaces associated with the packages of of the passed named element
	 */
	static def closeNamespace(NamedElement namedElement) '''
		«FOR ns : namedElement.usedNamespaces»
			}; // of namespace [ns.name/]
		«ENDFOR»
	'''
}