/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse.reverse;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.cdt.core.CCorePlugin;
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.IASTName;
import org.eclipse.cdt.core.dom.ast.IASTNode;
import org.eclipse.cdt.core.dom.ast.IASTNodeSelector;
import org.eclipse.cdt.core.dom.ast.IASTParameterDeclaration;
import org.eclipse.cdt.core.dom.ast.IASTSimpleDeclaration;
import org.eclipse.cdt.core.dom.ast.IASTTranslationUnit;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTConstructorChainInitializer;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateDeclaration;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTemplateParameter;
import org.eclipse.cdt.core.index.IIndex;
import org.eclipse.cdt.core.model.ICContainer;
import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.ICProject;
import org.eclipse.cdt.core.model.IDeclaration;
import org.eclipse.cdt.core.model.IEnumeration;
import org.eclipse.cdt.core.model.IEnumerator;
import org.eclipse.cdt.core.model.IField;
import org.eclipse.cdt.core.model.IFunction;
import org.eclipse.cdt.core.model.IInclude;
import org.eclipse.cdt.core.model.IMacro;
import org.eclipse.cdt.core.model.IMethod;
import org.eclipse.cdt.core.model.IMethodDeclaration;
import org.eclipse.cdt.core.model.IMethodTemplateDeclaration;
import org.eclipse.cdt.core.model.IParent;
import org.eclipse.cdt.core.model.ISourceRange;
import org.eclipse.cdt.core.model.ISourceReference;
import org.eclipse.cdt.core.model.ISourceRoot;
import org.eclipse.cdt.core.model.IStructure;
import org.eclipse.cdt.core.model.IStructureDeclaration;
import org.eclipse.cdt.core.model.IStructureTemplate;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.model.ITypeDef;
import org.eclipse.cdt.core.model.IUsing;
import org.eclipse.cdt.core.model.IVariableDeclaration;
import org.eclipse.cdt.core.parser.IToken;
import org.eclipse.cdt.core.parser.ast.ASTAccessVisibility;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.designer.languages.common.base.StdUriConstants;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Array;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Const;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.ConstInit;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.External;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Friend;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Include;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Inline;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Ptr;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Ref;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Template;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Typedef;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Virtual;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Visibility;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Volatile;
import org.eclipse.papyrus.designer.languages.cpp.profile.CppProfileResource;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.BatchReverseFunctionBody;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.DependencyAnalysis;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.ReverseUtils;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.change.CElementChange;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.change.ChangeMapStore;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.change.CppChangeObject;
import org.eclipse.papyrus.designer.languages.cpp.reverse.reverse.change.ModelChangeObject;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.FileWatcher;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.ModelManagement;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.RoundtripCppUtils;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.TypeOperationsEnhanced;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.AggregationKind;
import org.eclipse.uml2.uml.Association;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.ClassifierTemplateParameter;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.EnumerationLiteral;
import org.eclipse.uml2.uml.Generalization;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.OpaqueExpression;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.ParameterableElement;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.RedefinableTemplateSignature;
import org.eclipse.uml2.uml.TemplateParameter;
import org.eclipse.uml2.uml.TemplateSignature;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLFactory;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.VisibilityKind;
import org.eclipse.uml2.uml.profile.standard.Create;
import org.eclipse.uml2.uml.profile.standard.Destroy;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.ArrayExtensions;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * The main reverser class that will reverse C++ to UML
 */
@SuppressWarnings("all")
public class ReverseCpp2Uml {
  public enum ReverseMode {
    BATCH,
    
    INCREMENTAL;
  }
  
  public enum ConflictResolutionMode {
    FROM_MODEL,
    
    FROM_CODE,
    
    UI_INTERACTION;
  }
  
  private ReverseCpp2Uml.ConflictResolutionMode conflictResolveMode = ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE;
  
  public static String C_LangID = "C";
  
  public static String Cpp_LangID = "C++";
  
  private String langID = "C++";
  
  private Map<ICElement, EObject> map = new HashMap<ICElement, EObject>();
  
  private static final String TEMPLATE_PARAMETER_SIGNATURE_NAME = "template_paremeter_signature";
  
  private static final String REVERSE_FOLDER = "reversed_models";
  
  private static final String MODEL_POSTFIX = ".uml";
  
  private ITranslationUnit unit;
  
  private ICProject m_project;
  
  private List<Model> m_models;
  
  private static final Logger LOGGER = Logger.getLogger(ReverseCpp2Uml.class.getName());
  
  private IIndex index;
  
  private IProgressMonitor m_monitor;
  
  private String path;
  
  private ReverseUtils reverse_utils = ReverseUtils.getInstance();
  
  private static long timestamp = 0;
  
  private ReverseCpp2Uml.ReverseMode reverseMode = ReverseCpp2Uml.ReverseMode.BATCH;
  
  private Map<ITranslationUnit, IASTTranslationUnit> translationUnitToASTTranslationUnitMap;
  
  private Map<String, List<IInclude>> excludedIncludesMap;
  
  private List<CppChangeObject> changeList = new ArrayList<CppChangeObject>();
  
  private static List<ModelChangeObject> modelChangeList = new ArrayList<ModelChangeObject>();
  
  private List<ICContainer> containers = new UniqueEList<ICContainer>();
  
  private static final String projectPrefix = "org.eclipse.papyrus.cppgen";
  
  public static boolean addModelChange(final ModelChangeObject change) {
    return ReverseCpp2Uml.modelChangeList.add(change);
  }
  
  public static void clearModelChange() {
    ReverseCpp2Uml.modelChangeList.clear();
  }
  
  public ReverseCpp2Uml(final ITranslationUnit unit, final IProgressMonitor monitor, final String langID) {
    try {
      this.unit = unit;
      this.m_project = unit.getCProject();
      this.index = CCorePlugin.getIndexManager().getIndex(this.m_project);
      HashMap<ITranslationUnit, IASTTranslationUnit> _hashMap = new HashMap<ITranslationUnit, IASTTranslationUnit>();
      this.translationUnitToASTTranslationUnitMap = _hashMap;
      HashMap<String, List<IInclude>> _hashMap_1 = new HashMap<String, List<IInclude>>();
      this.excludedIncludesMap = _hashMap_1;
      this.m_monitor = monitor;
      this.langID = langID;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ReverseCpp2Uml(final ITranslationUnit unit, final IProgressMonitor monitor) {
    try {
      this.unit = unit;
      this.m_project = unit.getCProject();
      this.index = CCorePlugin.getIndexManager().getIndex(this.m_project);
      HashMap<ITranslationUnit, IASTTranslationUnit> _hashMap = new HashMap<ITranslationUnit, IASTTranslationUnit>();
      this.translationUnitToASTTranslationUnitMap = _hashMap;
      HashMap<String, List<IInclude>> _hashMap_1 = new HashMap<String, List<IInclude>>();
      this.excludedIncludesMap = _hashMap_1;
      this.m_monitor = monitor;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ReverseCpp2Uml(final ICProject project, final IProgressMonitor monitor, final String langID) {
    try {
      this.m_project = project;
      this.index = CCorePlugin.getIndexManager().getIndex(this.m_project);
      HashMap<ITranslationUnit, IASTTranslationUnit> _hashMap = new HashMap<ITranslationUnit, IASTTranslationUnit>();
      this.translationUnitToASTTranslationUnitMap = _hashMap;
      HashMap<String, List<IInclude>> _hashMap_1 = new HashMap<String, List<IInclude>>();
      this.excludedIncludesMap = _hashMap_1;
      this.m_monitor = monitor;
      this.langID = langID;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ReverseCpp2Uml(final ICProject project, final IProgressMonitor monitor) {
    try {
      this.m_project = project;
      this.index = CCorePlugin.getIndexManager().getIndex(this.m_project);
      HashMap<ITranslationUnit, IASTTranslationUnit> _hashMap = new HashMap<ITranslationUnit, IASTTranslationUnit>();
      this.translationUnitToASTTranslationUnitMap = _hashMap;
      HashMap<String, List<IInclude>> _hashMap_1 = new HashMap<String, List<IInclude>>();
      this.excludedIncludesMap = _hashMap_1;
      this.m_monitor = monitor;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public ReverseCpp2Uml(final ICProject project, final IProgressMonitor monitor, final String langID, final String path) {
    try {
      this.m_project = project;
      this.index = CCorePlugin.getIndexManager().getIndex(this.m_project);
      HashMap<ITranslationUnit, IASTTranslationUnit> _hashMap = new HashMap<ITranslationUnit, IASTTranslationUnit>();
      this.translationUnitToASTTranslationUnitMap = _hashMap;
      HashMap<String, List<IInclude>> _hashMap_1 = new HashMap<String, List<IInclude>>();
      this.excludedIncludesMap = _hashMap_1;
      this.m_monitor = monitor;
      this.langID = langID;
      this.path = path;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void reverse() {
    try {
      String projectName = this.unit.getCProject().getElementName();
      BatchReverseFunctionBody iSync = new BatchReverseFunctionBody(this.unit, projectName, this);
      iSync.run();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private boolean isSourceFolder(final ICContainer container) {
    try {
      ICElement[] _children = container.getChildren();
      for (final ICElement child : _children) {
        if ((child instanceof ITranslationUnit)) {
          return true;
        } else {
          if ((child instanceof ICContainer)) {
            boolean _isSourceFolder = this.isSourceFolder(((ICContainer) child));
            if (_isSourceFolder) {
              return true;
            }
          }
        }
      }
      return false;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Object reverseProject(final boolean reset) {
    try {
      final Function1<ISourceRoot, Boolean> _function = new Function1<ISourceRoot, Boolean>() {
        @Override
        public Boolean apply(final ISourceRoot it) {
          return Boolean.valueOf(ReverseCpp2Uml.this.isSourceFolder(it));
        }
      };
      Iterable<ISourceRoot> sourceRoots = IterableExtensions.<ISourceRoot>filter(((Iterable<ISourceRoot>)Conversions.doWrapArray(this.m_project.getSourceRoots())), _function);
      int _size = IterableExtensions.size(sourceRoots);
      boolean _lessThan = (_size < 1);
      if (_lessThan) {
        throw new Exception("No source folder");
      }
      this.containers.clear();
      final Function1<ISourceRoot, Boolean> _function_1 = new Function1<ISourceRoot, Boolean>() {
        @Override
        public Boolean apply(final ISourceRoot it) {
          return Boolean.valueOf(it.getElementName().contains(ReverseCpp2Uml.projectPrefix));
        }
      };
      ISourceRoot sourceRootGenerated = IterableExtensions.<ISourceRoot>head(IterableExtensions.<ISourceRoot>filter(sourceRoots, _function_1));
      if ((sourceRootGenerated != null)) {
        final Function1<ICContainer, Boolean> _function_2 = new Function1<ICContainer, Boolean>() {
          @Override
          public Boolean apply(final ICContainer it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSourceFolder(it));
          }
        };
        Iterables.<ICContainer>addAll(this.containers, IterableExtensions.<ICContainer>filter(Iterables.<ICContainer>filter(((Iterable<?>)Conversions.doWrapArray(sourceRootGenerated.getChildren())), ICContainer.class), _function_2));
      } else {
        Iterables.<ICContainer>addAll(this.containers, sourceRoots);
      }
      final ModelManagement modelManager = new ModelManagement();
      String umlFilePath = null;
      if (((this.path == null) || this.path.equals(""))) {
        IProject _project = this.m_project.getProject();
        String _elementName = this.m_project.getElementName();
        String _plus = (_elementName + ReverseCpp2Uml.MODEL_POSTFIX);
        umlFilePath = modelManager.getPath(_project, 
          ReverseCpp2Uml.REVERSE_FOLDER, _plus);
      } else {
        umlFilePath = this.path;
      }
      modelManager.createOrgetModel(this.m_project.getElementName(), umlFilePath, (!reset), reset);
      this.m_models = modelManager.getModels();
      final Consumer<Model> _function_3 = new Consumer<Model>() {
        @Override
        public void accept(final Model it) {
          RoundtripCppUtils.applyProfile(it, CppProfileResource.PROFILE_PATH);
          RoundtripCppUtils.applyProfile(it, StdUriConstants.UML_STD_PROFILE_PATH);
        }
      };
      this.m_models.forEach(_function_3);
      final Consumer<ICContainer> _function_4 = new Consumer<ICContainer>() {
        @Override
        public void accept(final ICContainer it) {
          ReverseCpp2Uml.this.reverseProject(it);
        }
      };
      this.containers.forEach(_function_4);
      modelManager.saveModel(IterableExtensions.<String>toList(Collections.<String>singleton(umlFilePath)));
      long _timeStamp = IterableExtensions.<Model>last(this.m_models).eResource().getTimeStamp();
      long _divide = (_timeStamp / 1000);
      ReverseCpp2Uml.timestamp = _divide;
      this.clearRawChangeList();
      modelManager.dispose();
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void syncIncrementalProject() {
    try {
      this.reverseMode = ReverseCpp2Uml.ReverseMode.INCREMENTAL;
      final Function1<ISourceRoot, Boolean> _function = new Function1<ISourceRoot, Boolean>() {
        @Override
        public Boolean apply(final ISourceRoot it) {
          return Boolean.valueOf(ReverseCpp2Uml.this.isSourceFolder(it));
        }
      };
      Iterable<ISourceRoot> sourceRoots = IterableExtensions.<ISourceRoot>filter(((Iterable<ISourceRoot>)Conversions.doWrapArray(this.m_project.getSourceRoots())), _function);
      int _size = IterableExtensions.size(sourceRoots);
      boolean _lessThan = (_size < 1);
      if (_lessThan) {
        throw new Exception("No source folder");
      }
      this.containers.clear();
      final Function1<ISourceRoot, Boolean> _function_1 = new Function1<ISourceRoot, Boolean>() {
        @Override
        public Boolean apply(final ISourceRoot it) {
          return Boolean.valueOf(it.getElementName().contains(ReverseCpp2Uml.projectPrefix));
        }
      };
      ISourceRoot sourceRootGenerated = IterableExtensions.<ISourceRoot>head(IterableExtensions.<ISourceRoot>filter(sourceRoots, _function_1));
      if ((sourceRootGenerated != null)) {
        final Function1<ICContainer, Boolean> _function_2 = new Function1<ICContainer, Boolean>() {
          @Override
          public Boolean apply(final ICContainer it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSourceFolder(it));
          }
        };
        Iterables.<ICContainer>addAll(this.containers, IterableExtensions.<ICContainer>filter(Iterables.<ICContainer>filter(((Iterable<?>)Conversions.doWrapArray(sourceRootGenerated.getChildren())), ICContainer.class), _function_2));
      } else {
        Iterables.<ICContainer>addAll(this.containers, sourceRoots);
      }
      final ModelManagement modelManager = new ModelManagement();
      String umlFilePath = null;
      if (((this.path == null) || this.path.equals(""))) {
        IProject _project = this.m_project.getProject();
        String _elementName = this.m_project.getElementName();
        String _plus = (_elementName + ReverseCpp2Uml.MODEL_POSTFIX);
        umlFilePath = modelManager.getPath(_project, 
          ReverseCpp2Uml.REVERSE_FOLDER, _plus);
      } else {
        umlFilePath = this.path;
      }
      modelManager.createOrgetModel(this.m_project.getElementName(), umlFilePath, false, false);
      this.m_models = modelManager.getModels();
      final Consumer<Model> _function_3 = new Consumer<Model>() {
        @Override
        public void accept(final Model it) {
          RoundtripCppUtils.applyProfile(it, CppProfileResource.PROFILE_PATH);
          RoundtripCppUtils.applyProfile(it, StdUriConstants.UML_STD_PROFILE_PATH);
        }
      };
      this.m_models.forEach(_function_3);
      if (((this.getRawChangeList() == null) || (this.getRawChangeList().size() == 0))) {
        FileWatcher fileWatcher = new FileWatcher(this.m_project, ReverseCpp2Uml.timestamp);
        List<ITranslationUnit> modifiedItus = fileWatcher.getModifiledTranslationUnits(this.m_project);
        for (final ITranslationUnit modified : modifiedItus) {
          this.syncTranslationUnit(modified);
        }
        modelManager.saveModel(IterableExtensions.<String>toList(Collections.<String>singleton(umlFilePath)));
        long _timeStamp = IterableExtensions.<Model>last(this.m_models).eResource().getTimeStamp();
        long _divide = (_timeStamp / 1000);
        ReverseCpp2Uml.timestamp = _divide;
      } else {
        this.reverseIncrementalChanges();
        modelManager.saveModel(IterableExtensions.<String>toList(Collections.<String>singleton(umlFilePath)));
      }
      modelManager.dispose();
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void reverseProject(final IParent parent) {
    try {
      if ((!(parent instanceof ITranslationUnit))) {
        final Consumer<ITranslationUnit> _function = new Consumer<ITranslationUnit>() {
          @Override
          public void accept(final ITranslationUnit it) {
            String _elementName = it.getElementName();
            String _plus = ("Parsing types in " + _elementName);
            ReverseCpp2Uml.this.m_monitor.subTask(_plus);
            ReverseCpp2Uml.this.reverseHeader(it);
          }
        };
        Iterables.<ITranslationUnit>filter(((Iterable<?>)Conversions.doWrapArray(parent.getChildren())), ITranslationUnit.class).forEach(_function);
        final Consumer<IParent> _function_1 = new Consumer<IParent>() {
          @Override
          public void accept(final IParent it) {
            ReverseCpp2Uml.this.reverseProject(it);
          }
        };
        Iterables.<IParent>filter(((Iterable<?>)Conversions.doWrapArray(parent.getChildren())), IParent.class).forEach(_function_1);
        final Function1<ITranslationUnit, Boolean> _function_2 = new Function1<ITranslationUnit, Boolean>() {
          @Override
          public Boolean apply(final ITranslationUnit it) {
            boolean _isHeaderUnit = it.isHeaderUnit();
            return Boolean.valueOf((!_isHeaderUnit));
          }
        };
        final Consumer<ITranslationUnit> _function_3 = new Consumer<ITranslationUnit>() {
          @Override
          public void accept(final ITranslationUnit it) {
            String _elementName = it.getElementName();
            String _plus = ("Parsing method implementations in " + _elementName);
            ReverseCpp2Uml.this.m_monitor.subTask(_plus);
            ReverseCpp2Uml.this.reverseSource(it);
          }
        };
        IterableExtensions.<ITranslationUnit>filter(Iterables.<ITranslationUnit>filter(((Iterable<?>)Conversions.doWrapArray(parent.getChildren())), ITranslationUnit.class), _function_2).forEach(_function_3);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Model getCorrespondingModel(final ITranslationUnit unit) {
    Model ret = null;
    for (final Model model : this.m_models) {
      boolean _contains = unit.getPath().toString().contains(model.getName());
      if (_contains) {
        ret = model;
      }
    }
    return ret;
  }
  
  private void reverseSource(final ITranslationUnit unit) {
    String _elementName = this.m_project.getElementName();
    Model _correspondingModel = this.getCorrespondingModel(unit);
    BatchReverseFunctionBody sync = new BatchReverseFunctionBody(unit, _elementName, _correspondingModel, this);
    try {
      sync.run();
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        e.printStackTrace();
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  private List<Type> reverseHeader(final ITranslationUnit headerUnit) {
    try {
      return this.getOrCreateClassifier(this.getCorrespondingModel(headerUnit), headerUnit);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private List<Type> getOrCreateClassifier(final Model model, final IParent parent) throws Exception {
    List<Type> ret = new UniqueEList<Type>();
    ICElement[] childrend = parent.getChildren();
    for (int i = 0; (i < childrend.length); i++) {
      {
        ICElement child = childrend[i];
        boolean _matched = false;
        if ((child instanceof IStructure)) {
          _matched=true;
        }
        if (!_matched) {
          if ((child instanceof IEnumeration)) {
            _matched=true;
          }
        }
        if (_matched) {
          boolean _equals = ((IDeclaration) child).getElementName().equals("");
          boolean _not = (!_equals);
          if (_not) {
            ret.addAll(this.getOrcreateTypes(((IDeclaration) child), parent, model));
          }
        }
        if (!_matched) {
          if ((child instanceof IMethod)) {
            _matched=true;
            if ((((IMethod) child).getTranslationUnit().isHeaderUnit() && (!(child.getParent() instanceof IStructure)))) {
              final IMethod method = ((IMethod) child);
              List<Classifier> types = this.getTypeByQualifiedName(method, method.getTranslationUnit(), this.getContextNamespaces(method));
              int _size = types.size();
              boolean _greaterThan = (_size > 0);
              if (_greaterThan) {
                final Classifier type = types.get(0);
                ITranslationUnit _translationUnit = method.getTranslationUnit();
                String _elementName = method.getCProject().getElementName();
                BatchReverseFunctionBody batchReverser = new BatchReverseFunctionBody(_translationUnit, _elementName, this, type);
                String name = method.getElementName();
                IASTNode node = this.findEnclosingNode(method);
                if ((node instanceof IASTFunctionDefinition)) {
                  IASTFunctionDefinition definition = ((IASTFunctionDefinition) node);
                  IASTFunctionDeclarator declarator = definition.getDeclarator();
                  String body = this.getBody(method);
                  Operation operation = batchReverser.updateMethod(method, node, 0, parent, name, body, declarator);
                  if ((operation != null)) {
                    ITranslationUnit _translationUnit_1 = method.getTranslationUnit();
                    new DependencyAnalysis(operation, definition, _translationUnit_1, this).analyzeDependencies();
                  }
                }
              }
            }
          }
        }
        if (!_matched) {
          if ((child instanceof IFunction)) {
            _matched=true;
          }
        }
        if (!_matched) {
          if ((child instanceof IParent)) {
            _matched=true;
            ret.addAll(this.getOrCreateClassifier(model, ((IParent) child)));
          }
        }
        if (!_matched) {
          if ((child instanceof ITypeDef)) {
            _matched=true;
            org.eclipse.uml2.uml.Package parentPack = this.getContainerPackage(((ITypeDef) child).getTranslationUnit());
            Classifier tempType = null;
            ICElement _parent = child.getParent();
            if ((_parent instanceof IStructure)) {
              Type _uMLType = this.getUMLType(child.getParent().getElementName(), ((ITypeDef) child).getTranslationUnit(), 
                this.getContextNamespaces(child));
              Classifier parentClass = ((Classifier) _uMLType);
              NamedElement _ownedMember = parentClass.getOwnedMember(child.getElementName());
              boolean _tripleEquals = (_ownedMember == null);
              if (_tripleEquals) {
                this.createOrgetClassifier(parentClass, child, true);
              }
              NamedElement _ownedMember_1 = parentClass.getOwnedMember(child.getElementName());
              tempType = ((Classifier) _ownedMember_1);
            } else {
              Type _ownedType = parentPack.getOwnedType(child.getElementName());
              boolean _tripleEquals_1 = (_ownedType == null);
              if (_tripleEquals_1) {
                this.createOrgetClassifier(parentPack, child, true);
              }
              NamedElement _ownedMember_2 = parentPack.getOwnedMember(child.getElementName());
              tempType = ((Classifier) _ownedMember_2);
            }
            ret.add(tempType);
          }
        }
      }
    }
    return ret;
  }
  
  private List<Type> getOrcreateTypes(final IDeclaration iStructure, final IParent parent, final Model model) {
    try {
      List<Type> ret = new UniqueEList<Type>();
      String _elementName = iStructure.getElementName();
      String _plus = ("Parsing type " + _elementName);
      this.m_monitor.subTask(_plus);
      Classifier tempType = null;
      IDeclaration istructure = ((IDeclaration) iStructure);
      String elementName = istructure.getElementName();
      org.eclipse.uml2.uml.Package parentPack = this.getContainerPackage(istructure.getTranslationUnit());
      Classifier parentClass = null;
      ICElement _parent = istructure.getParent();
      if ((_parent instanceof IStructure)) {
        Type _uMLType = this.getUMLType(istructure.getParent().getElementName(), istructure.getTranslationUnit(), 
          this.getContextNamespaces(istructure));
        parentClass = ((Classifier) _uMLType);
        NamedElement _ownedMember = parentClass.getOwnedMember(elementName);
        boolean _tripleEquals = (_ownedMember == null);
        if (_tripleEquals) {
          this.createOrgetClassifier(parentClass, istructure, true);
        }
        NamedElement _ownedMember_1 = parentClass.getOwnedMember(elementName);
        tempType = ((Classifier) _ownedMember_1);
      } else {
        Type _ownedType = parentPack.getOwnedType(elementName);
        boolean _tripleEquals_1 = (_ownedType == null);
        if (_tripleEquals_1) {
          this.createOrgetClassifier(parentPack, istructure, true);
        }
        NamedElement _ownedMember_2 = parentPack.getOwnedMember(elementName);
        tempType = ((Classifier) _ownedMember_2);
      }
      if ((tempType != null)) {
        ret.add(tempType);
        if ((istructure instanceof IParent)) {
          List<ICElement> nestedStructures = new UniqueEList<ICElement>();
          Iterables.<ICElement>addAll(nestedStructures, Iterables.<IDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(((IParent)istructure).getChildren())), IDeclaration.class));
          Iterables.<ICElement>addAll(nestedStructures, Iterables.<ITypeDef>filter(((Iterable<?>)Conversions.doWrapArray(((IParent)istructure).getChildren())), ITypeDef.class));
          int _size = nestedStructures.size();
          ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _size, true);
          for (final Integer i : _doubleDotLessThan) {
            {
              ICElement ns = nestedStructures.get((i).intValue());
              boolean _isEmpty = ns.getElementName().trim().isEmpty();
              boolean _not = (!_isEmpty);
              if (_not) {
                EObject _createOrgetClassifier = this.createOrgetClassifier(tempType, ns, true);
                Type t = ((Type) _createOrgetClassifier);
                if ((t != null)) {
                  ret.add(t);
                }
              } else {
                if ((ns instanceof IEnumeration)) {
                  boolean isAnonEnumInstance = false;
                  try {
                    int _size_1 = nestedStructures.size();
                    boolean _lessThan = (((i).intValue() + 1) < _size_1);
                    if (_lessThan) {
                      ICElement nextNs = nestedStructures.get(((i).intValue() + 1));
                      if ((nextNs instanceof IField)) {
                        String nsRaw = this.findEnclosingNode(((ISourceReference)ns)).getRawSignature();
                        String nextNsRaw = this.findEnclosingNode(((ISourceReference)nextNs)).getRawSignature();
                        boolean _contains = nextNsRaw.contains(nsRaw);
                        if (_contains) {
                          isAnonEnumInstance = true;
                        }
                      }
                    }
                  } catch (final Throwable _t) {
                    if (_t instanceof Exception) {
                      final Exception e = (Exception)_t;
                      e.printStackTrace();
                    } else {
                      throw Exceptions.sneakyThrow(_t);
                    }
                  }
                  if ((!isAnonEnumInstance)) {
                    EObject _createOrgetClassifier_1 = this.createOrgetClassifier(tempType, ns, true);
                    Type t_1 = ((Type) _createOrgetClassifier_1);
                    if ((t_1 != null)) {
                      ret.add(t_1);
                    }
                  }
                }
              }
            }
          }
        }
      }
      if ((istructure instanceof IStructureTemplate)) {
        IStructureTemplate istructureTemplate = ((IStructureTemplate) istructure);
        for (int ti = 0; (ti < ((List<String>)Conversions.doWrapArray(istructureTemplate.getTemplateParameterTypes())).size()); ti++) {
          this.createOrgetTemplateParameter(tempType, istructureTemplate.getTemplateParameterTypes()[ti], "class");
        }
      }
      int _size_1 = ret.size();
      boolean _greaterThan = (_size_1 > 0);
      if (_greaterThan) {
        if (((!(tempType instanceof Enumeration)) && (!(tempType instanceof PrimitiveType)))) {
          ITranslationUnit structureItu = istructure.getTranslationUnit();
          IParent file = parent;
          List<IInclude> excludedIncludes = null;
          if ((structureItu instanceof IParent)) {
            file = ((IParent) structureItu);
            final String key = this.getExcludedIncludesMapKey(structureItu, tempType);
            excludedIncludes = this.excludedIncludesMap.get(key);
          }
          boolean _isApplied = StereotypeUtil.isApplied(tempType, Include.class);
          boolean _not = (!_isApplied);
          if (_not) {
            StereotypeUtil.apply(tempType, Include.class);
          }
          Include _stereotypeApplication = UMLUtil.<Include>getStereotypeApplication(tempType, Include.class);
          boolean _tripleNotEquals = (_stereotypeApplication != null);
          if (_tripleNotEquals) {
            String header = UMLUtil.<Include>getStereotypeApplication(tempType, Include.class).getHeader();
            String escape = "";
            ICElement[] _children = file.getChildren();
            for (final ICElement j : _children) {
              if ((j instanceof IInclude)) {
                boolean excluded = false;
                if ((excludedIncludes != null)) {
                  excluded = excludedIncludes.contains(j);
                }
                if (((!excluded) && (!header.contains(((IInclude)j).getElementName())))) {
                  boolean _contains = ((IInclude)j).getElementName().contains("Pkg_");
                  if (_contains) {
                    try {
                      String[] includeTokens = ((IInclude)j).getElementName().split("/");
                      String includeTrimmed = "";
                      int _length = includeTokens.length;
                      int _minus = (_length - 1);
                      ExclusiveRange _doubleDotLessThan_1 = new ExclusiveRange(0, _minus, true);
                      for (final Integer i_1 : _doubleDotLessThan_1) {
                        String _includeTrimmed = includeTrimmed;
                        String _get = includeTokens[(i_1).intValue()];
                        includeTrimmed = (_includeTrimmed + _get);
                      }
                      String[] typeQualifiedNameTokens = tempType.getQualifiedName().split("::");
                      String typeQualifiedNameTrimmed = "";
                      int _length_1 = typeQualifiedNameTokens.length;
                      int _minus_1 = (_length_1 - 1);
                      ExclusiveRange _doubleDotLessThan_2 = new ExclusiveRange(1, _minus_1, true);
                      for (final Integer i_2 : _doubleDotLessThan_2) {
                        String _typeQualifiedNameTrimmed = typeQualifiedNameTrimmed;
                        String _get_1 = typeQualifiedNameTokens[(i_2).intValue()];
                        typeQualifiedNameTrimmed = (_typeQualifiedNameTrimmed + _get_1);
                      }
                      boolean _equals = includeTrimmed.equals(typeQualifiedNameTrimmed);
                      boolean _not_1 = (!_equals);
                      if (_not_1) {
                        String _elementName_1 = ((IInclude)j).getElementName();
                        String _plus_1 = ((((header + escape) + "#include ") + "\"") + _elementName_1);
                        String _plus_2 = (_plus_1 + "\"");
                        header = _plus_2;
                        escape = "\n";
                      }
                    } catch (final Throwable _t) {
                      if (_t instanceof Exception) {
                        final Exception e = (Exception)_t;
                        e.printStackTrace();
                        String _elementName_2 = ((IInclude)j).getElementName();
                        String _plus_3 = ((((header + escape) + "#include ") + "\"") + _elementName_2);
                        String _plus_4 = (_plus_3 + "\"");
                        header = _plus_4;
                        escape = "\n";
                      } else {
                        throw Exceptions.sneakyThrow(_t);
                      }
                    }
                  } else {
                    String _elementName_1 = ((IInclude)j).getElementName();
                    String _plus_1 = ((((header + escape) + "#include ") + "\"") + _elementName_1);
                    String _plus_2 = (_plus_1 + "\"");
                    header = _plus_2;
                    escape = "\n";
                  }
                }
              } else {
                if ((j instanceof IMacro)) {
                  boolean _contains_1 = header.contains(((IMacro)j).getElementName());
                  boolean _not_1 = (!_contains_1);
                  if (_not_1) {
                    IASTNode node = this.findEnclosingNode(((ISourceReference)j));
                    if ((node != null)) {
                      final String nodeString = node.toString();
                      int _indexOf = nodeString.indexOf("=");
                      int _plus_3 = (_indexOf + 1);
                      String value = nodeString.substring(_plus_3);
                      if (((value == null) || value.equals(""))) {
                        boolean _contains_2 = ((IMacro)j).getElementName().contains(tempType.getName().replaceAll(" ", "").toUpperCase());
                        boolean _not_2 = (!_contains_2);
                        if (_not_2) {
                          String _elementName_2 = ((IMacro)j).getElementName();
                          String _plus_4 = (((header + escape) + "#define ") + _elementName_2);
                          String _plus_5 = (_plus_4 + " ");
                          String _plus_6 = (_plus_5 + value);
                          header = _plus_6;
                          escape = "\n";
                        }
                      } else {
                        String _elementName_3 = ((IMacro)j).getElementName();
                        String _plus_7 = (((header + escape) + "#define ") + _elementName_3);
                        String _plus_8 = (_plus_7 + " ");
                        String _plus_9 = (_plus_8 + value);
                        header = _plus_9;
                        escape = "\n";
                      }
                    }
                  }
                }
              }
            }
            Include _stereotypeApplication_1 = UMLUtil.<Include>getStereotypeApplication(tempType, Include.class);
            _stereotypeApplication_1.setHeader(header);
            if ((file instanceof ITranslationUnit)) {
              try {
                IPath sourcePath = ((ITranslationUnit)file).getPath().removeFileExtension().addFileExtension("cpp");
                ICElement sourceFile = ((ITranslationUnit) file).getCProject().findElement(sourcePath);
                if ((sourceFile instanceof ITranslationUnit)) {
                  BatchReverseFunctionBody.updateCppInclude(((ITranslationUnit)sourceFile), tempType);
                }
              } catch (final Throwable _t) {
                if (_t instanceof Exception) {
                } else {
                  throw Exceptions.sneakyThrow(_t);
                }
              }
            }
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void syncTranslationUnit(final ITranslationUnit itu) {
    try {
      Iterable<IStructure> structures = Iterables.<IStructure>filter(this.reverse_utils.getAllIStructures(itu, false, true, this.m_project), IStructure.class);
      for (final IStructure structure : structures) {
        {
          Type classifier = this.getClassifier(this.getCorrespondingModel(itu), structure, structure.getElementName(), itu);
          if ((classifier instanceof Classifier)) {
            this.syncIStructureToModel(((Classifier)classifier), structure);
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Object syncIStructureToModel(final Classifier classifier, final IStructure istructure) {
    try {
      Object _xblockexpression = null;
      {
        List<IField> fields = IterableExtensions.<IField>toList(Iterables.<IField>filter(((Iterable<?>)Conversions.doWrapArray(istructure.getChildren())), IField.class));
        List<Property> attributes = IterableExtensions.<Property>toList(Iterables.<Property>filter(classifier.getOwnedElements(), Property.class));
        this.mergeAttributes(classifier, attributes, fields);
        List<IMethodDeclaration> methods = IterableExtensions.<IMethodDeclaration>toList(Iterables.<IMethodDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(istructure.getChildren())), IMethodDeclaration.class));
        List<Operation> operations = IterableExtensions.<Operation>toList(Iterables.<Operation>filter(classifier.getOwnedElements(), Operation.class));
        this.mergeOperations(classifier, operations, methods);
        List<IDeclaration> declarations = IterableExtensions.<IDeclaration>toList(Iterables.<IDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(istructure.getChildren())), IDeclaration.class));
        List<Type> nestedTypes = IterableExtensions.<Type>toList(Iterables.<Type>filter(classifier.getOwnedElements(), Type.class));
        _xblockexpression = this.mergeNestedTypes(classifier, nestedTypes, declarations);
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void mergeAttributes(final Classifier parent, final List<Property> attributes, final List<IField> fields) {
    try {
      UniqueEList<IField> foundFieldList = new UniqueEList<IField>();
      UniqueEList<IField> notFoundFieldList = new UniqueEList<IField>();
      UniqueEList<Property> foundAttrList = new UniqueEList<Property>();
      UniqueEList<Property> notFoundAttrList = new UniqueEList<Property>();
      for (final IField field : fields) {
        {
          final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
            @Override
            public Boolean apply(final Property it) {
              return Boolean.valueOf(it.getName().equals(field.getElementName()));
            }
          };
          final Property attrFound = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(attributes, _function));
          if ((attrFound == null)) {
            notFoundFieldList.add(field);
          } else {
            foundAttrList.add(attrFound);
            foundFieldList.add(field);
            final Function1<ModelChangeObject, Boolean> _function_1 = new Function1<ModelChangeObject, Boolean>() {
              @Override
              public Boolean apply(final ModelChangeObject it) {
                return Boolean.valueOf(Objects.equal(it.eObject, attrFound));
              }
            };
            ModelChangeObject _head = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_1));
            boolean isModelObjectChanged = (_head != null);
            if (isModelObjectChanged) {
              this.syncAttributeWithMode(parent, attrFound, field, this.conflictResolveMode);
            } else {
              this.syncAttributeWithMode(parent, attrFound, field, ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
            }
          }
        }
      }
      for (final Property attr : attributes) {
        boolean _contains = foundAttrList.contains(attr);
        boolean _not = (!_contains);
        if (_not) {
          notFoundAttrList.add(attr);
        }
      }
      final Function1<ModelChangeObject, Boolean> _function = new Function1<ModelChangeObject, Boolean>() {
        @Override
        public Boolean apply(final ModelChangeObject it) {
          return Boolean.valueOf((it.eventType == Notification.ADD));
        }
      };
      List<ModelChangeObject> addChanges = IterableExtensions.<ModelChangeObject>toList(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function));
      UniqueEList<Property> tobeRemovedsInNotFound = new UniqueEList<Property>();
      for (final Property attr_1 : notFoundAttrList) {
        {
          final Function1<ModelChangeObject, Boolean> _function_1 = new Function1<ModelChangeObject, Boolean>() {
            @Override
            public Boolean apply(final ModelChangeObject it) {
              return Boolean.valueOf(Objects.equal(it.eObject, attr_1));
            }
          };
          ModelChangeObject modelChangeObj = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(addChanges, _function_1));
          if ((modelChangeObj != null)) {
            tobeRemovedsInNotFound.add(attr_1);
          } else {
          }
        }
      }
      for (final Property i : tobeRemovedsInNotFound) {
        notFoundAttrList.remove(i);
      }
      UniqueEList<IField> processedFields = new UniqueEList<IField>();
      UniqueEList<IField> remainingFields = new UniqueEList<IField>();
      UniqueEList<Property> remainingAttributes = new UniqueEList<Property>();
      for (final IField notFoundField : notFoundFieldList) {
        boolean _contains_1 = processedFields.contains(notFoundField);
        boolean _not_1 = (!_contains_1);
        if (_not_1) {
          final Type umlType = this.getUMLType(notFoundField, this.getContextNamespaces(notFoundField));
          final String typeName = this.getCppTypeName(notFoundField.getTypeName());
          final Function1<IField, Boolean> _function_1 = new Function1<IField, Boolean>() {
            @Override
            public Boolean apply(final IField it) {
              try {
                return Boolean.valueOf(ReverseCpp2Uml.this.getCppTypeName(it.getTypeName()).equals(typeName));
              } catch (Throwable _e) {
                throw Exceptions.sneakyThrow(_e);
              }
            }
          };
          List<IField> sameTypeNameFields = IterableExtensions.<IField>toList(IterableExtensions.<IField>filter(notFoundFieldList, _function_1));
          final Function1<Property, Boolean> _function_2 = new Function1<Property, Boolean>() {
            @Override
            public Boolean apply(final Property it) {
              return Boolean.valueOf(it.getType().getName().equals(umlType.getName()));
            }
          };
          List<Property> sameTypeAttrs = IterableExtensions.<Property>toList(IterableExtensions.<Property>filter(notFoundAttrList, _function_2));
          int i_1 = 0;
          for (i_1 = 0; (i_1 < sameTypeNameFields.size()); i_1++) {
            int _size = sameTypeAttrs.size();
            boolean _greaterEqualsThan = (i_1 >= _size);
            if (_greaterEqualsThan) {
              remainingFields.add(sameTypeNameFields.get(i_1));
            } else {
              final IField tobeProcessedField = sameTypeNameFields.get(i_1);
              final Property tobeProcessedAttr = sameTypeAttrs.get(i_1);
              final Function1<ModelChangeObject, Boolean> _function_3 = new Function1<ModelChangeObject, Boolean>() {
                @Override
                public Boolean apply(final ModelChangeObject it) {
                  return Boolean.valueOf(Objects.equal(it.eObject, tobeProcessedAttr));
                }
              };
              ModelChangeObject _head = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_3));
              boolean isModelObjectChanged = (_head != null);
              if (isModelObjectChanged) {
                this.syncAttributeWithMode(parent, tobeProcessedAttr, tobeProcessedField, ReverseCpp2Uml.ConflictResolutionMode.FROM_MODEL);
              } else {
                this.syncAttributeWithMode(parent, tobeProcessedAttr, tobeProcessedField, ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
              }
              processedFields.add(tobeProcessedField);
              notFoundAttrList.remove(tobeProcessedAttr);
            }
          }
          final List<Property> _converted_sameTypeAttrs = (List<Property>)sameTypeAttrs;
          int _length = ((Object[])Conversions.unwrapArray(_converted_sameTypeAttrs, Object.class)).length;
          boolean _lessThan = (i_1 < _length);
          if (_lessThan) {
            for (int j = i_1; (j < ((Object[])Conversions.unwrapArray(sameTypeAttrs, Object.class)).length); j++) {
              remainingAttributes.add(sameTypeAttrs.get(j));
            }
          }
        }
      }
      for (final Property notFoundAttr : notFoundAttrList) {
        boolean _contains_2 = remainingAttributes.contains(notFoundAttr);
        boolean _not_2 = (!_contains_2);
        if (_not_2) {
          remainingAttributes.add(notFoundAttr);
        }
      }
      for (final IField remaining : remainingFields) {
        this.createProperty(remaining, parent);
      }
      final Function1<ModelChangeObject, Boolean> _function_3 = new Function1<ModelChangeObject, Boolean>() {
        @Override
        public Boolean apply(final ModelChangeObject it) {
          return Boolean.valueOf(((it.eventType == Notification.ADD) || (it.eventType == Notification.SET)));
        }
      };
      List<ModelChangeObject> remainingChanges = IterableExtensions.<ModelChangeObject>toList(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_3));
      for (final Property remaining_1 : remainingAttributes) {
        final Function1<ModelChangeObject, Boolean> _function_4 = new Function1<ModelChangeObject, Boolean>() {
          @Override
          public Boolean apply(final ModelChangeObject it) {
            return Boolean.valueOf(Objects.equal(it.eObject, remaining_1));
          }
        };
        boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<ModelChangeObject>filter(remainingChanges, _function_4));
        if (_isEmpty) {
          if ((parent instanceof org.eclipse.uml2.uml.Class)) {
            EList<Property> _ownedAttributes = ((org.eclipse.uml2.uml.Class) parent).getOwnedAttributes();
            _ownedAttributes.remove(remaining_1);
          } else {
            if ((parent instanceof DataType)) {
              EList<Property> _ownedAttributes_1 = ((DataType) parent).getOwnedAttributes();
              _ownedAttributes_1.remove(remaining_1);
            }
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Boolean syncAttributeWithMode(final Classifier parent, final Property attribute, final IField field, final ReverseCpp2Uml.ConflictResolutionMode resolveMode) {
    Boolean _switchResult = null;
    if (resolveMode != null) {
      switch (resolveMode) {
        case FROM_MODEL:
          _switchResult = null;
          break;
        case FROM_CODE:
          boolean _xblockexpression = false;
          {
            Type type = this.getUMLType(field, this.getContextNamespaces(field));
            attribute.setType(type);
            attribute.setName(field.getElementName());
            this.updateProperty(field, attribute);
            final Function1<ModelChangeObject, Boolean> _function = new Function1<ModelChangeObject, Boolean>() {
              @Override
              public Boolean apply(final ModelChangeObject it) {
                return Boolean.valueOf(Objects.equal(it.eObject, attribute));
              }
            };
            ModelChangeObject changeObj = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function));
            boolean _xifexpression = false;
            if ((changeObj != null)) {
              _xifexpression = ReverseCpp2Uml.modelChangeList.remove(changeObj);
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = Boolean.valueOf(_xblockexpression);
          break;
        case UI_INTERACTION:
          _switchResult = null;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  private Boolean syncOperationWithMode(final Classifier parent, final Operation op, final IMethodDeclaration method, final ReverseCpp2Uml.ConflictResolutionMode resolveMode) {
    Boolean _switchResult = null;
    if (resolveMode != null) {
      switch (resolveMode) {
        case FROM_MODEL:
          _switchResult = null;
          break;
        case FROM_CODE:
          boolean _xblockexpression = false;
          {
            op.setName(method.getElementName());
            this.updateMethod(((org.eclipse.uml2.uml.Class) parent), op, method);
            final Function1<ModelChangeObject, Boolean> _function = new Function1<ModelChangeObject, Boolean>() {
              @Override
              public Boolean apply(final ModelChangeObject it) {
                return Boolean.valueOf(Objects.equal(it.eObject, op));
              }
            };
            ModelChangeObject changeObj = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function));
            boolean _xifexpression = false;
            if ((changeObj != null)) {
              _xifexpression = ReverseCpp2Uml.modelChangeList.remove(changeObj);
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = Boolean.valueOf(_xblockexpression);
          break;
        case UI_INTERACTION:
          _switchResult = null;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }
  
  private boolean isSameOperation(final Operation op, final IMethodDeclaration method) {
    boolean ret = true;
    boolean _equals = op.getName().equals(method.getElementName());
    boolean _not = (!_equals);
    if (_not) {
      ret = false;
    } else {
      ret = this.isSameSignature(op, method);
    }
    return ret;
  }
  
  private boolean isSameSignature(final Operation op, final IMethodDeclaration method) {
    boolean ret = true;
    final Function1<Parameter, Boolean> _function = new Function1<Parameter, Boolean>() {
      @Override
      public Boolean apply(final Parameter it) {
        ParameterDirectionKind _direction = it.getDirection();
        return Boolean.valueOf((!Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL)));
      }
    };
    List<Parameter> params = IterableExtensions.<Parameter>toList(IterableExtensions.<Parameter>filter(op.getOwnedParameters(), _function));
    int _numberOfParameters = method.getNumberOfParameters();
    int _size = params.size();
    boolean _notEquals = (_numberOfParameters != _size);
    if (_notEquals) {
      ret = false;
    } else {
      for (int i = 0; (i < method.getNumberOfParameters()); i++) {
        boolean _equals = this.getCppTypeName(method.getParameterTypes()[i]).equals(params.get(i).getType().getName());
        boolean _not = (!_equals);
        if (_not) {
          ret = false;
        }
      }
      if ((ret != false)) {
        final Function1<Parameter, Boolean> _function_1 = new Function1<Parameter, Boolean>() {
          @Override
          public Boolean apply(final Parameter it) {
            ParameterDirectionKind _direction = it.getDirection();
            return Boolean.valueOf(Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL));
          }
        };
        Parameter returnParam = IterableExtensions.<Parameter>head(IterableExtensions.<Parameter>filter(op.getOwnedParameters(), _function_1));
        if ((returnParam == null)) {
          boolean _equals = this.getCppTypeName(method.getReturnType()).equals("void");
          boolean _not = (!_equals);
          if (_not) {
            ret = false;
          }
        } else {
          if (((returnParam.getType() != null) && (!this.getCppTypeName(method.getReturnType()).equals(returnParam.getType().getName())))) {
            ret = false;
          }
        }
      }
    }
    return ret;
  }
  
  private boolean isSameMethodDeclaration(final IMethodDeclaration method1, final IMethodDeclaration method2) {
    try {
      boolean result = true;
      result = (result && method1.getSignature().equals(method2.getSignature()));
      result = (result && Objects.equal(this.convertVisibility(method1.getVisibility()), this.convertVisibility(method2.getVisibility())));
      result = (result && (method1.isStatic() == method2.isStatic()));
      result = (result && (method1.isVirtual() == method2.isVirtual()));
      result = (result && (method1.isVirtual() == method2.isVirtual()));
      result = (result && (method1.isInline() == method2.isInline()));
      result = (result && (method1.isFriend() == method2.isFriend()));
      result = (result && (method1.isVolatile() == method2.isVolatile()));
      result = (result && (method1.isConstructor() == method2.isConstructor()));
      result = (result && (method1.isDestructor() == method2.isDestructor()));
      result = (result && ((method1.getReturnType() != null) && (method2.getReturnType() != null)));
      if (((method1.getReturnType() != null) && (method2.getReturnType() != null))) {
        result = (result && method1.getReturnType().equals(method2.getReturnType()));
      }
      result = (result && (method1.getNumberOfParameters() == method2.getNumberOfParameters()));
      if ((result == false)) {
        return result;
      }
      try {
        IASTFunctionDeclarator declarator1 = this.getDeclarator(method1);
        IASTFunctionDeclarator declarator2 = this.getDeclarator(method2);
        result = (result && declarator1.getRawSignature().equals(declarator2.getRawSignature()));
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          e.printStackTrace();
          return result;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      return result;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private int getNumberOfSameParameters(final Operation op, final IMethodDeclaration method) {
    int ret = 0;
    final Function1<Parameter, Boolean> _function = new Function1<Parameter, Boolean>() {
      @Override
      public Boolean apply(final Parameter it) {
        ParameterDirectionKind _direction = it.getDirection();
        return Boolean.valueOf((!Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL)));
      }
    };
    List<Parameter> params = IterableExtensions.<Parameter>toList(IterableExtensions.<Parameter>filter(op.getOwnedParameters(), _function));
    for (int i = 0; (i < method.getNumberOfParameters()); i++) {
      int _size = params.size();
      boolean _lessEqualsThan = (i <= _size);
      if (_lessEqualsThan) {
        if (((params.get(i).getType() != null) && this.getCppTypeName(method.getParameterTypes()[i]).equals(params.get(i).getType().getName()))) {
          ret++;
        }
      }
    }
    return ret;
  }
  
  private Operation findMostSameOperation(final List<Operation> ops, final IMethodDeclaration method) {
    boolean _isEmpty = ops.isEmpty();
    if (_isEmpty) {
      return null;
    }
    Operation ret = IterableExtensions.<Operation>head(ops);
    for (final Operation op : ops) {
      boolean _notEquals = (!Objects.equal(ret, op));
      if (_notEquals) {
        int _numberOfSameParameters = this.getNumberOfSameParameters(op, method);
        int _numberOfSameParameters_1 = this.getNumberOfSameParameters(ret, method);
        boolean _greaterThan = (_numberOfSameParameters > _numberOfSameParameters_1);
        if (_greaterThan) {
          ret = op;
        }
      }
    }
    return ret;
  }
  
  private void mergeOperations(final Classifier parent, final List<Operation> operations, final List<IMethodDeclaration> methods) {
    UniqueEList<IMethodDeclaration> foundMethodList = new UniqueEList<IMethodDeclaration>();
    UniqueEList<IMethodDeclaration> notFoundMethodList = new UniqueEList<IMethodDeclaration>();
    UniqueEList<Operation> foundOperationList = new UniqueEList<Operation>();
    UniqueEList<Operation> notFoundOperationList = new UniqueEList<Operation>();
    for (final IMethodDeclaration method : methods) {
      {
        final Function1<Operation, Boolean> _function = new Function1<Operation, Boolean>() {
          @Override
          public Boolean apply(final Operation it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSameOperation(it, method));
          }
        };
        final Operation opFound = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(operations, _function));
        if ((opFound == null)) {
          notFoundMethodList.add(method);
        } else {
          foundOperationList.add(opFound);
          foundMethodList.add(method);
          final Function1<ModelChangeObject, Boolean> _function_1 = new Function1<ModelChangeObject, Boolean>() {
            @Override
            public Boolean apply(final ModelChangeObject it) {
              return Boolean.valueOf(Objects.equal(it.eObject, opFound));
            }
          };
          ModelChangeObject _head = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_1));
          boolean isModelObjectChanged = (_head != null);
          if (isModelObjectChanged) {
          } else {
          }
        }
      }
    }
    for (final Operation op : operations) {
      boolean _contains = foundOperationList.contains(op);
      boolean _not = (!_contains);
      if (_not) {
        notFoundOperationList.add(op);
      }
    }
    final Function1<ModelChangeObject, Boolean> _function = new Function1<ModelChangeObject, Boolean>() {
      @Override
      public Boolean apply(final ModelChangeObject it) {
        return Boolean.valueOf((it.eventType == Notification.ADD));
      }
    };
    List<ModelChangeObject> addChanges = IterableExtensions.<ModelChangeObject>toList(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function));
    UniqueEList<Operation> tobeRemovedsInNotFound = new UniqueEList<Operation>();
    for (final Operation op_1 : notFoundOperationList) {
      {
        final Function1<ModelChangeObject, Boolean> _function_1 = new Function1<ModelChangeObject, Boolean>() {
          @Override
          public Boolean apply(final ModelChangeObject it) {
            return Boolean.valueOf(Objects.equal(it.eObject, op_1));
          }
        };
        ModelChangeObject modelChangeObj = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(addChanges, _function_1));
        if ((modelChangeObj != null)) {
          tobeRemovedsInNotFound.add(op_1);
        } else {
        }
      }
    }
    for (final Operation i : tobeRemovedsInNotFound) {
      notFoundOperationList.remove(i);
    }
    UniqueEList<IMethodDeclaration> processedMethods = new UniqueEList<IMethodDeclaration>();
    UniqueEList<IMethodDeclaration> remainingMethods = new UniqueEList<IMethodDeclaration>();
    UniqueEList<Operation> remainingOperations = new UniqueEList<Operation>();
    for (final IMethodDeclaration notFoundMethod : notFoundMethodList) {
      boolean _contains_1 = processedMethods.contains(notFoundMethod);
      boolean _not_1 = (!_contains_1);
      if (_not_1) {
        final String functionName = IterableExtensions.<String>last(((Iterable<String>)Conversions.doWrapArray(notFoundMethod.getElementName().split("::"))));
        final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
          @Override
          public Boolean apply(final Operation it) {
            return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, notFoundMethod));
          }
        };
        final Operation sameSignatureOperation = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(notFoundOperationList, _function_1));
        if ((sameSignatureOperation != null)) {
          final Function1<ModelChangeObject, Boolean> _function_2 = new Function1<ModelChangeObject, Boolean>() {
            @Override
            public Boolean apply(final ModelChangeObject it) {
              return Boolean.valueOf(Objects.equal(it.eObject, sameSignatureOperation));
            }
          };
          ModelChangeObject _head = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_2));
          boolean isModelObjectChanged = (_head != null);
          if (isModelObjectChanged) {
            this.syncOperationWithMode(parent, sameSignatureOperation, notFoundMethod, ReverseCpp2Uml.ConflictResolutionMode.FROM_MODEL);
          } else {
            this.syncOperationWithMode(parent, sameSignatureOperation, notFoundMethod, ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
          }
          notFoundOperationList.remove(sameSignatureOperation);
          processedMethods.add(notFoundMethod);
        } else {
          final Function1<IMethodDeclaration, Boolean> _function_3 = new Function1<IMethodDeclaration, Boolean>() {
            @Override
            public Boolean apply(final IMethodDeclaration it) {
              return Boolean.valueOf(it.getElementName().equals(functionName));
            }
          };
          final List<IMethodDeclaration> sameNameMethods = IterableExtensions.<IMethodDeclaration>toList(IterableExtensions.<IMethodDeclaration>filter(notFoundMethodList, _function_3));
          final Function1<Operation, Boolean> _function_4 = new Function1<Operation, Boolean>() {
            @Override
            public Boolean apply(final Operation it) {
              return Boolean.valueOf(it.getName().equals(functionName));
            }
          };
          final List<Operation> sameNameOperations = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(notFoundOperationList, _function_4));
          int _size = sameNameOperations.size();
          boolean _greaterThan = (_size > 0);
          if (_greaterThan) {
            int i_1 = 0;
            for (i_1 = 0; (i_1 < sameNameMethods.size()); i_1++) {
              int _size_1 = sameNameOperations.size();
              boolean _greaterEqualsThan = (i_1 >= _size_1);
              if (_greaterEqualsThan) {
                remainingMethods.add(sameNameMethods.get(i_1));
              } else {
                final IMethodDeclaration tobeProcessedMethod = sameNameMethods.get(i_1);
                final Operation tobeProcessedOp = sameNameOperations.get(i_1);
                final Function1<ModelChangeObject, Boolean> _function_5 = new Function1<ModelChangeObject, Boolean>() {
                  @Override
                  public Boolean apply(final ModelChangeObject it) {
                    return Boolean.valueOf(Objects.equal(it.eObject, tobeProcessedOp));
                  }
                };
                ModelChangeObject _head_1 = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_5));
                boolean isModelObjectChanged_1 = (_head_1 != null);
                if (isModelObjectChanged_1) {
                  this.syncOperationWithMode(parent, tobeProcessedOp, tobeProcessedMethod, ReverseCpp2Uml.ConflictResolutionMode.FROM_MODEL);
                } else {
                  this.syncOperationWithMode(parent, tobeProcessedOp, tobeProcessedMethod, ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
                }
                processedMethods.add(tobeProcessedMethod);
                notFoundOperationList.remove(tobeProcessedOp);
              }
            }
            int _length = ((Object[])Conversions.unwrapArray(sameNameOperations, Object.class)).length;
            boolean _lessThan = (i_1 < _length);
            if (_lessThan) {
              for (int j = i_1; (j < ((Object[])Conversions.unwrapArray(sameNameOperations, Object.class)).length); j++) {
                remainingOperations.add(sameNameOperations.get(j));
              }
            }
          } else {
            final Operation mostSameOp = this.findMostSameOperation(notFoundOperationList, notFoundMethod);
            if ((mostSameOp != null)) {
              final Function1<ModelChangeObject, Boolean> _function_5 = new Function1<ModelChangeObject, Boolean>() {
                @Override
                public Boolean apply(final ModelChangeObject it) {
                  return Boolean.valueOf(Objects.equal(it.eObject, mostSameOp));
                }
              };
              ModelChangeObject _head_1 = IterableExtensions.<ModelChangeObject>head(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_5));
              boolean isModelObjectChanged_1 = (_head_1 != null);
              if (isModelObjectChanged_1) {
                this.syncOperationWithMode(parent, mostSameOp, notFoundMethod, ReverseCpp2Uml.ConflictResolutionMode.FROM_MODEL);
              } else {
                this.syncOperationWithMode(parent, mostSameOp, notFoundMethod, ReverseCpp2Uml.ConflictResolutionMode.FROM_CODE);
              }
              processedMethods.add(notFoundMethod);
              notFoundOperationList.remove(mostSameOp);
            }
          }
        }
      }
    }
    for (final Operation notFoundOp : notFoundOperationList) {
      boolean _contains_2 = remainingOperations.contains(notFoundOp);
      boolean _not_2 = (!_contains_2);
      if (_not_2) {
        remainingOperations.add(notFoundOp);
      }
    }
    for (final IMethodDeclaration remaining : remainingMethods) {
      this.createMethod(remaining, ((org.eclipse.uml2.uml.Class) parent));
    }
    final Function1<ModelChangeObject, Boolean> _function_6 = new Function1<ModelChangeObject, Boolean>() {
      @Override
      public Boolean apply(final ModelChangeObject it) {
        return Boolean.valueOf(((it.eventType == Notification.ADD) || (it.eventType == Notification.SET)));
      }
    };
    List<ModelChangeObject> remainingChanges = IterableExtensions.<ModelChangeObject>toList(IterableExtensions.<ModelChangeObject>filter(ReverseCpp2Uml.modelChangeList, _function_6));
    for (final Operation remaining_1 : remainingOperations) {
      final Function1<ModelChangeObject, Boolean> _function_7 = new Function1<ModelChangeObject, Boolean>() {
        @Override
        public Boolean apply(final ModelChangeObject it) {
          return Boolean.valueOf(Objects.equal(it.eObject, remaining_1));
        }
      };
      boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<ModelChangeObject>filter(remainingChanges, _function_7));
      if (_isEmpty) {
        if ((parent instanceof org.eclipse.uml2.uml.Class)) {
          EList<Operation> _ownedOperations = ((org.eclipse.uml2.uml.Class) parent).getOwnedOperations();
          _ownedOperations.remove(remaining_1);
        } else {
          if ((parent instanceof DataType)) {
            EList<Operation> _ownedOperations_1 = ((DataType) parent).getOwnedOperations();
            _ownedOperations_1.remove(remaining_1);
          }
        }
      }
    }
  }
  
  private Object mergeNestedTypes(final Classifier parent, final List<Type> nestedTypes, final List<IDeclaration> declarations) {
    return null;
  }
  
  private void reverseIncrementalChanges() {
    this.changeList.clear();
    this.optimizeChangeList();
    for (final CppChangeObject change : this.changeList) {
      final int _switchValue = change.changeKind;
      switch (_switchValue) {
        case IResourceDelta.ADDED:
          this.addToModel(change);
          break;
        case IResourceDelta.CHANGED:
          this.updateToModel(change);
          break;
        case IResourceDelta.REMOVED:
          this.removeFromModel(change);
          break;
      }
    }
    this.changeList.clear();
  }
  
  private List<CElementChange> getRawChangeList() {
    if ((((ChangeMapStore.changesMap != null) && (ChangeMapStore.changesMap.get(this.m_project.getElementName()) != null)) && 
      (ChangeMapStore.changesMap.get(this.m_project.getElementName()).size() > 0))) {
      return ChangeMapStore.changesMap.get(this.m_project.getElementName());
    }
    return null;
  }
  
  private void clearRawChangeList() {
    if ((((ChangeMapStore.changesMap != null) && (this.m_project != null)) && (ChangeMapStore.changesMap.get(this.m_project.getElementName()) != null))) {
      ChangeMapStore.changesMap.get(this.m_project.getElementName()).clear();
    }
  }
  
  private void optimizeChangeList() {
    List<CElementChange> rawChangeList = this.getRawChangeList();
    this.changeList.clear();
    List<CElementChange> processedList = new UniqueEList<CElementChange>();
    List<CElementChange> remainList = new UniqueEList<CElementChange>();
    for (int i = 0; (i < rawChangeList.size()); i++) {
      {
        CElementChange change = rawChangeList.get(i);
        boolean _isInSourceContainers = this.isInSourceContainers(change.getElement());
        if (_isInSourceContainers) {
          boolean _contains = processedList.contains(change);
          boolean _not = (!_contains);
          if (_not) {
            int _changeKind = change.getChangeKind();
            switch (_changeKind) {
              case IResourceDelta.ADDED:
                List<CElementChange> inter = this.findIntermediateEvent(rawChangeList, i, change);
                int _size = inter.size();
                boolean _greaterThan = (_size > 0);
                if (_greaterThan) {
                  processedList.add(change);
                  processedList.addAll(inter);
                }
                break;
              case IResourceDelta.CHANGED:
                List<CElementChange> inter_1 = this.findIntermediateEvent(rawChangeList, i, change);
                int _size_1 = inter_1.size();
                boolean _greaterThan_1 = (_size_1 > 0);
                if (_greaterThan_1) {
                  processedList.addAll(inter_1);
                }
                break;
              case IResourceDelta.REMOVED:
                break;
              default:
                processedList.add(change);
                break;
            }
          }
          boolean _contains_1 = processedList.contains(change);
          boolean _not_1 = (!_contains_1);
          if (_not_1) {
            remainList.add(change);
          }
        }
      }
    }
    final Map<ICElement, List<CppChangeObject>> doubtChangeListMap = new HashMap<ICElement, List<CppChangeObject>>();
    for (int i = 0; (i < remainList.size()); i++) {
      {
        CElementChange change = remainList.get(i);
        CppChangeObject cppChangeObject = null;
        int _changeKind = change.getChangeKind();
        switch (_changeKind) {
          case IResourceDelta.ADDED:
            boolean found = false;
            int j = (i + 1);
            while (((j < remainList.size()) && (!found))) {
              {
                if ((((remainList.get(j).getChangeKind() == IResourceDelta.REMOVED) && (change.getElement().getElementType() == remainList.get(j).getElement().getElementType())) && Objects.equal(remainList.get(j).getParent(), change.getParent()))) {
                  ICElement _element = remainList.get(j).getElement();
                  ICElement _element_1 = change.getElement();
                  ITranslationUnit _translationUnitFromElement = this.reverse_utils.getTranslationUnitFromElement(change.getElement());
                  ICElement _parent = change.getParent();
                  CppChangeObject _cppChangeObject = new CppChangeObject(_element, _element_1, _translationUnitFromElement, _parent, IResourceDelta.CHANGED);
                  cppChangeObject = _cppChangeObject;
                  List<CppChangeObject> _get = doubtChangeListMap.get(change.getParent());
                  boolean _tripleEquals = (_get == null);
                  if (_tripleEquals) {
                    ICElement _parent_1 = change.getParent();
                    ArrayList<CppChangeObject> _arrayList = new ArrayList<CppChangeObject>();
                    doubtChangeListMap.put(_parent_1, _arrayList);
                  }
                  if ((((change.getElement() instanceof IField) || (change.getElement() instanceof IMethodDeclaration)) || (change.getElement() instanceof IEnumerator))) {
                    doubtChangeListMap.get(change.getParent()).add(cppChangeObject);
                  }
                  remainList.remove(j);
                  j--;
                  found = true;
                }
                j++;
              }
            }
            if ((!found)) {
              ICElement _element = change.getElement();
              ITranslationUnit _translationUnitFromElement = this.reverse_utils.getTranslationUnitFromElement(change.getElement());
              ICElement _parent = change.getParent();
              CppChangeObject _cppChangeObject = new CppChangeObject(null, _element, _translationUnitFromElement, _parent, IResourceDelta.ADDED);
              cppChangeObject = _cppChangeObject;
            }
            break;
          case IResourceDelta.CHANGED:
            ICElement _element_1 = remainList.get(i).getElement();
            ICElement _element_2 = change.getElement();
            ITranslationUnit _translationUnitFromElement_1 = this.reverse_utils.getTranslationUnitFromElement(change.getElement());
            ICElement _parent_1 = change.getParent();
            CppChangeObject _cppChangeObject_1 = new CppChangeObject(_element_1, _element_2, _translationUnitFromElement_1, _parent_1, IResourceDelta.CHANGED);
            cppChangeObject = _cppChangeObject_1;
            break;
          case IResourceDelta.REMOVED:
            boolean found_1 = false;
            int j_1 = (i + 1);
            while (((j_1 < remainList.size()) && (!found_1))) {
              {
                if (((remainList.get(j_1).getChangeKind() == IResourceDelta.ADDED) && (change.getElement().getElementType() == remainList.get(j_1).getElement().getElementType()))) {
                  ICElement _element_3 = change.getElement();
                  ICElement _element_4 = remainList.get(j_1).getElement();
                  ITranslationUnit _translationUnitFromElement_2 = this.reverse_utils.getTranslationUnitFromElement(change.getElement());
                  ICElement _parent_2 = change.getParent();
                  CppChangeObject _cppChangeObject_2 = new CppChangeObject(_element_3, _element_4, _translationUnitFromElement_2, _parent_2, IResourceDelta.CHANGED);
                  cppChangeObject = _cppChangeObject_2;
                  List<CppChangeObject> _get = doubtChangeListMap.get(change.getParent());
                  boolean _tripleEquals = (_get == null);
                  if (_tripleEquals) {
                    ICElement _parent_3 = change.getParent();
                    ArrayList<CppChangeObject> _arrayList = new ArrayList<CppChangeObject>();
                    doubtChangeListMap.put(_parent_3, _arrayList);
                  }
                  if ((((change.getElement() instanceof IField) || (change.getElement() instanceof IMethodDeclaration)) || (change.getElement() instanceof IEnumerator))) {
                    doubtChangeListMap.get(change.getParent()).add(cppChangeObject);
                  }
                  remainList.remove(j_1);
                  j_1--;
                  found_1 = true;
                }
                j_1++;
              }
            }
            if ((!found_1)) {
              ICElement _element_3 = change.getElement();
              ITranslationUnit _translationUnitFromElement_2 = this.reverse_utils.getTranslationUnitFromElement(change.getElement());
              ICElement _parent_2 = change.getParent();
              CppChangeObject _cppChangeObject_2 = new CppChangeObject(null, _element_3, _translationUnitFromElement_2, _parent_2, IResourceDelta.REMOVED);
              cppChangeObject = _cppChangeObject_2;
            }
            break;
        }
        if ((cppChangeObject != null)) {
          this.changeList.add(cppChangeObject);
        }
      }
    }
    Set<Map.Entry<ICElement, List<CppChangeObject>>> _entrySet = doubtChangeListMap.entrySet();
    for (final Map.Entry<ICElement, List<CppChangeObject>> doubtMap : _entrySet) {
      int _size = doubtMap.getValue().size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        this.alignChangedElements(doubtMap.getKey(), doubtMap.getValue());
      }
    }
    this.clearRawChangeList();
  }
  
  private void alignChangedElements(final ICElement parent, final List<CppChangeObject> changes) {
    try {
      final ArrayList<NamedElement> oldNamedElementsList = new ArrayList<NamedElement>();
      final Map<NamedElement, CppChangeObject> oldNamedElementToChangeMap = new HashMap<NamedElement, CppChangeObject>();
      final Consumer<CppChangeObject> _function = new Consumer<CppChangeObject>() {
        @Override
        public void accept(final CppChangeObject it) {
          NamedElement oldNamedElement = ReverseCpp2Uml.this.getNamedElement(it.parent, it.oldElement);
          oldNamedElementToChangeMap.put(oldNamedElement, it);
          oldNamedElementsList.add(oldNamedElement);
        }
      };
      changes.forEach(_function);
      final EObject namedElementParent = IterableExtensions.<NamedElement>head(oldNamedElementsList).eContainer();
      final ArrayList<NamedElement> alignedOldNamedElements = new ArrayList<NamedElement>();
      final Consumer<NamedElement> _function_1 = new Consumer<NamedElement>() {
        @Override
        public void accept(final NamedElement it) {
          boolean _contains = oldNamedElementsList.contains(it);
          if (_contains) {
            alignedOldNamedElements.add(it);
          }
        }
      };
      Iterables.<NamedElement>filter(namedElementParent.eContents(), NamedElement.class).forEach(_function_1);
      final ArrayList<ICElement> newICElements = new ArrayList<ICElement>();
      if ((parent instanceof IParent)) {
        final Consumer<ICElement> _function_2 = new Consumer<ICElement>() {
          @Override
          public void accept(final ICElement it) {
            final ICElement childInParent = it;
            final Function1<CppChangeObject, Boolean> _function = new Function1<CppChangeObject, Boolean>() {
              @Override
              public Boolean apply(final CppChangeObject it) {
                return Boolean.valueOf(Objects.equal(it.newElement, childInParent));
              }
            };
            final CppChangeObject founded = IterableExtensions.<CppChangeObject>head(IterableExtensions.<CppChangeObject>filter(changes, _function));
            if ((founded != null)) {
              newICElements.add(founded.newElement);
            }
          }
        };
        ((List<ICElement>)Conversions.doWrapArray(((IParent)parent).getChildren())).forEach(_function_2);
      }
      int _size = alignedOldNamedElements.size();
      int _size_1 = newICElements.size();
      boolean _equals = (_size == _size_1);
      if (_equals) {
        for (int i = 0; (i < alignedOldNamedElements.size()); i++) {
          {
            CppChangeObject change = oldNamedElementToChangeMap.get(alignedOldNamedElements.get(i));
            ICElement _get = newICElements.get(i);
            boolean _notEquals = (!Objects.equal(change.newElement, _get));
            if (_notEquals) {
              boolean sameElementType = ((change.newElement instanceof IField) && (newICElements.get(i) instanceof IField));
              sameElementType = (sameElementType || ((change.newElement instanceof IMethodDeclaration) && (newICElements.get(i) instanceof IMethodDeclaration)));
              sameElementType = (sameElementType || ((change.newElement instanceof IEnumerator) && (newICElements.get(i) instanceof IEnumerator)));
              if (sameElementType) {
                change.newElement = newICElements.get(i);
              }
            }
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private NamedElement getNamedElement(final ICElement parent, final ICElement child) {
    try {
      NamedElement ret = null;
      boolean _matched = false;
      if ((child instanceof IField)) {
        _matched=true;
        if ((parent instanceof IStructure)) {
          IStructure parentStructure = ((IStructure) parent);
          Type classifier = this.getClassifier(this.getCorrespondingModel(parentStructure.getTranslationUnit()), parentStructure, parentStructure.getElementName(), parentStructure.getTranslationUnit());
          if ((classifier != null)) {
            boolean _isExist = this.isExist(classifier, child.getElementName());
            if (_isExist) {
              final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
                @Override
                public Boolean apply(final Property it) {
                  return Boolean.valueOf(it.getName().equals(child.getElementName()));
                }
              };
              Property prop = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(Iterables.<Property>filter(classifier.getOwnedElements(), Property.class), _function));
              ret = prop;
            }
          }
        }
      }
      if (!_matched) {
        if ((child instanceof IMethodDeclaration)) {
          _matched=true;
          if ((parent instanceof IStructure)) {
            IStructure parentStructure_1 = ((IStructure) parent);
            Type classifier_1 = this.getClassifier(this.getCorrespondingModel(parentStructure_1.getTranslationUnit()), parentStructure_1, parentStructure_1.getElementName(), parentStructure_1.getTranslationUnit());
            if ((classifier_1 != null)) {
              boolean _isExist_1 = this.isExist(classifier_1, child.getElementName());
              if (_isExist_1) {
                final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
                  @Override
                  public Boolean apply(final Operation it) {
                    return Boolean.valueOf(it.getName().equals(child.getElementName()));
                  }
                };
                List<Operation> samenames = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(Iterables.<Operation>filter(classifier_1.getOwnedElements(), Operation.class), _function_1));
                final IMethodDeclaration oldMethod = ((IMethodDeclaration) child);
                final Function1<Operation, Boolean> _function_2 = new Function1<Operation, Boolean>() {
                  @Override
                  public Boolean apply(final Operation it) {
                    return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, oldMethod));
                  }
                };
                Operation op = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(samenames, _function_2));
                if ((op != null)) {
                  ret = op;
                }
              }
            }
          }
        }
      }
      if (!_matched) {
        if ((child instanceof IEnumerator)) {
          _matched=true;
          IEnumeration cppEnumeration = ((IEnumeration) parent);
          Type enumeration = this.getClassifier(this.getCorrespondingModel(cppEnumeration.getTranslationUnit()), cppEnumeration, cppEnumeration.getElementName(), cppEnumeration.getTranslationUnit());
          if (((enumeration != null) && (enumeration instanceof Enumeration))) {
            final Function1<EnumerationLiteral, Boolean> _function_3 = new Function1<EnumerationLiteral, Boolean>() {
              @Override
              public Boolean apply(final EnumerationLiteral it) {
                String _name = it.getName();
                String _elementName = child.getElementName();
                return Boolean.valueOf(Objects.equal(_name, _elementName));
              }
            };
            EnumerationLiteral enumerator = IterableExtensions.<EnumerationLiteral>head(IterableExtensions.<EnumerationLiteral>filter(Iterables.<EnumerationLiteral>filter(((Enumeration) enumeration).getOwnedLiterals(), EnumerationLiteral.class), _function_3));
            if ((enumerator != null)) {
              ret = enumerator;
            }
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private boolean isInSourceContainers(final ICElement element) {
    if ((element == null)) {
      return false;
    }
    if (((element instanceof ICContainer) && this.containers.contains(element))) {
      return true;
    }
    return this.isInSourceContainers(element.getParent());
  }
  
  private List<CElementChange> findIntermediateEvent(final List<CElementChange> list, final int position, final CElementChange change) {
    List<CElementChange> ret = new UniqueEList<CElementChange>();
    for (int i = (position + 1); (i < list.size()); i++) {
      if ((((change.getChangeKind() == IResourceDelta.ADDED) && (list.get(i).getChangeKind() == IResourceDelta.REMOVED)) && 
        this.isSameICElement(change.getElement(), list.get(i).getElement()))) {
        if (((change.getElement() instanceof IMethodDeclaration) && (list.get(i).getElement() instanceof IMethodDeclaration))) {
          ICElement _element = change.getElement();
          final IMethodDeclaration method = ((IMethodDeclaration) _element);
          ICElement _element_1 = list.get(i).getElement();
          boolean _isSameMethodDeclaration = this.isSameMethodDeclaration(method, ((IMethodDeclaration) _element_1));
          if (_isSameMethodDeclaration) {
            ret.add(list.get(i));
          }
        } else {
          ret.add(list.get(i));
        }
      } else {
        if (((change.getChangeKind() == IResourceDelta.CHANGED) && this.isSameICElement(change.getElement(), list.get(i).getElement()))) {
          ret.add(list.get(i));
        }
      }
    }
    final Function1<CElementChange, Boolean> _function = new Function1<CElementChange, Boolean>() {
      @Override
      public Boolean apply(final CElementChange it) {
        int _changeKind = it.getChangeKind();
        return Boolean.valueOf((_changeKind == IResourceDelta.CHANGED));
      }
    };
    List<CElementChange> changed = IterableExtensions.<CElementChange>toList(IterableExtensions.<CElementChange>filter(ret, _function));
    int _size = changed.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      ret.remove(IterableExtensions.<CElementChange>last(changed));
    }
    return ret;
  }
  
  private ICElement getSameICelement(final ICElement element, final int fromPos, final int changeKind) {
    int _size = this.getRawChangeList().size();
    boolean _greaterEqualsThan = (fromPos >= _size);
    if (_greaterEqualsThan) {
      return null;
    }
    ICElement ret = null;
    for (int i = fromPos; (i < this.getRawChangeList().size()); i++) {
      {
        CElementChange change = this.getRawChangeList().get(i);
        if ((ret == null)) {
          if ((changeKind == IResourceDelta.ADDED)) {
            if (((change.getChangeKind() == IResourceDelta.REMOVED) && this.isSameICElement(element, change.getElement()))) {
              ret = change.getElement();
            }
          } else {
            if ((changeKind == IResourceDelta.REMOVED)) {
              if (((change.getChangeKind() == IResourceDelta.ADDED) && this.isSameICElement(element, change.getElement()))) {
                ret = change.getElement();
              }
            } else {
              if ((changeKind == IResourceDelta.CHANGED)) {
                if (((change.getChangeKind() == IResourceDelta.CHANGED) && this.isSameICElement(element, change.getElement()))) {
                  ret = change.getElement();
                }
              }
            }
          }
        }
      }
    }
    return ret;
  }
  
  private boolean isSameICElement(final ICElement e1, final ICElement e2) {
    boolean ret = false;
    if (((e1.getElementName().equals(e2.getElementName()) && (e1.getElementType() == e2.getElementType())) && e1.getParent().getElementName().equals(e2.getParent().getElementName()))) {
      ret = true;
    }
    return ret;
  }
  
  private boolean isExist(final NamedElement parent, final String childName) {
    final Function1<NamedElement, Boolean> _function = new Function1<NamedElement, Boolean>() {
      @Override
      public Boolean apply(final NamedElement it) {
        return Boolean.valueOf(it.getName().equals(childName));
      }
    };
    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<NamedElement>filter(Iterables.<NamedElement>filter(parent.getOwnedElements(), NamedElement.class), _function));
    return (!_isEmpty);
  }
  
  private Object addToModel(final CppChangeObject change) {
    try {
      Object _switchResult = null;
      final ICElement _switchValue = change.newElement;
      boolean _matched = false;
      if ((change.newElement instanceof IField)) {
        _matched=true;
        final IField field = ((IField) change.newElement);
        ICElement _parent = field.getParent();
        if ((_parent instanceof IStructure)) {
          ICElement _parent_1 = field.getParent();
          IStructure parentStructure = ((IStructure) _parent_1);
          Type classifier = this.getClassifier(this.getCorrespondingModel(parentStructure.getTranslationUnit()), parentStructure, parentStructure.getElementName(), parentStructure.getTranslationUnit());
          if ((classifier != null)) {
            boolean _isExist = this.isExist(classifier, field.getElementName());
            boolean _not = (!_isExist);
            if (_not) {
              this.createProperty(field, ((Classifier) classifier));
            }
          }
        }
      }
      if (!_matched) {
        if ((change.newElement instanceof IMethodDeclaration)) {
          _matched=true;
          Object _xblockexpression = null;
          {
            final IMethodDeclaration method = ((IMethodDeclaration) change.newElement);
            Object _xifexpression = null;
            ICElement _parent_2 = method.getParent();
            if ((_parent_2 instanceof IStructure)) {
              ICElement _parent_3 = method.getParent();
              IStructure parentStructure_1 = ((IStructure) _parent_3);
              Type classifier_1 = this.getClassifier(this.getCorrespondingModel(parentStructure_1.getTranslationUnit()), parentStructure_1, parentStructure_1.getElementName(), parentStructure_1.getTranslationUnit());
              if ((classifier_1 != null)) {
                final Function1<Operation, Boolean> _function = new Function1<Operation, Boolean>() {
                  @Override
                  public Boolean apply(final Operation it) {
                    return Boolean.valueOf(it.getName().equals(method.getElementName()));
                  }
                };
                List<Operation> samenames = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(Iterables.<Operation>filter(classifier_1.getOwnedElements(), Operation.class), _function));
                final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
                  @Override
                  public Boolean apply(final Operation it) {
                    return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, method));
                  }
                };
                boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<Operation>filter(samenames, _function_1));
                if (_isEmpty) {
                  this.createMethod(method, ((org.eclipse.uml2.uml.Class) classifier_1));
                }
              }
            } else {
              Object _xifexpression_1 = null;
              ICElement _parent_4 = method.getParent();
              if ((_parent_4 instanceof ITranslationUnit)) {
                _xifexpression_1 = null;
              }
              _xifexpression = _xifexpression_1;
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = _xblockexpression;
        }
      }
      if (!_matched) {
        if ((change.newElement instanceof IStructure)) {
          _matched=true;
        }
        if (!_matched) {
          if ((change.newElement instanceof IEnumeration)) {
            _matched=true;
          }
        }
        if (!_matched) {
          if ((change.newElement instanceof ITypeDef)) {
            _matched=true;
          }
        }
        if (_matched) {
          EObject _xblockexpression_1 = null;
          {
            IDeclaration declaration = ((IDeclaration) change.newElement);
            EObject _xifexpression = null;
            ICElement _parent_2 = declaration.getParent();
            if ((_parent_2 instanceof IStructure)) {
              EObject _xblockexpression_2 = null;
              {
                ICElement _parent_3 = declaration.getParent();
                IStructure parentStructure_1 = ((IStructure) _parent_3);
                Type parentClassifier = this.getClassifier(this.getCorrespondingModel(parentStructure_1.getTranslationUnit()), parentStructure_1, parentStructure_1.getElementName(), parentStructure_1.getTranslationUnit());
                _xblockexpression_2 = this.createOrgetClassifier(parentClassifier, declaration, true);
              }
              _xifexpression = _xblockexpression_2;
            } else {
              EObject _xblockexpression_3 = null;
              {
                org.eclipse.uml2.uml.Package containerPackage = this.getContainerPackage(declaration.getTranslationUnit());
                _xblockexpression_3 = this.createOrgetClassifier(containerPackage, declaration, true);
              }
              _xifexpression = _xblockexpression_3;
            }
            _xblockexpression_1 = _xifexpression;
          }
          _switchResult = _xblockexpression_1;
        }
      }
      if (!_matched) {
        if ((change.newElement instanceof IEnumerator)) {
          _matched=true;
          EnumerationLiteral _xblockexpression_2 = null;
          {
            IEnumerator enumerator = ((IEnumerator) change.newElement);
            IPath _path = enumerator.getPath();
            IEnumeration cppEnumeration = ((IEnumeration) _path);
            Type enumeration = this.getClassifier(this.getCorrespondingModel(cppEnumeration.getTranslationUnit()), cppEnumeration, cppEnumeration.getElementName(), cppEnumeration.getTranslationUnit());
            EnumerationLiteral _xifexpression = null;
            if ((enumeration instanceof Enumeration)) {
              _xifexpression = ((Enumeration)enumeration).createOwnedLiteral(enumerator.getElementName());
            }
            _xblockexpression_2 = _xifexpression;
          }
          _switchResult = _xblockexpression_2;
        }
      }
      if (!_matched) {
        if ((change.newElement instanceof ITranslationUnit)) {
          _matched=true;
          List<Type> _xblockexpression_3 = null;
          {
            ITranslationUnit itu = ((ITranslationUnit) change.newElement);
            _xblockexpression_3 = this.getOrCreateClassifier(this.getCorrespondingModel(itu), itu);
          }
          _switchResult = _xblockexpression_3;
        }
      }
      return _switchResult;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private List<Type> updateToModel(final CppChangeObject change) {
    try {
      List<Type> _switchResult = null;
      final ICElement _switchValue = change.newElement;
      boolean _matched = false;
      if ((change.oldElement instanceof IField)) {
        _matched=true;
        final IField field = ((IField) change.newElement);
        ICElement _parent = field.getParent();
        if ((_parent instanceof IStructure)) {
          ICElement _parent_1 = field.getParent();
          IStructure parentStructure = ((IStructure) _parent_1);
          Type classifier = this.getClassifier(this.getCorrespondingModel(parentStructure.getTranslationUnit()), parentStructure, parentStructure.getElementName(), parentStructure.getTranslationUnit());
          if ((classifier != null)) {
            boolean _isExist = this.isExist(classifier, change.oldElement.getElementName());
            if (_isExist) {
              final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
                @Override
                public Boolean apply(final Property it) {
                  return Boolean.valueOf(it.getName().equals(change.oldElement.getElementName()));
                }
              };
              Property prop = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(Iterables.<Property>filter(classifier.getOwnedElements(), Property.class), _function));
              if ((prop != null)) {
                this.updateProperty(((IField) change.newElement), prop);
              }
            }
          }
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IMethodDeclaration)) {
          _matched=true;
          final IMethodDeclaration method = ((IMethodDeclaration) change.newElement);
          ICElement _parent_2 = method.getParent();
          if ((_parent_2 instanceof IStructure)) {
            ICElement _parent_3 = method.getParent();
            IStructure parentStructure_1 = ((IStructure) _parent_3);
            Type classifier_1 = this.getClassifier(this.getCorrespondingModel(parentStructure_1.getTranslationUnit()), parentStructure_1, parentStructure_1.getElementName(), parentStructure_1.getTranslationUnit());
            if ((classifier_1 != null)) {
              boolean _isExist_1 = this.isExist(classifier_1, change.oldElement.getElementName());
              if (_isExist_1) {
                final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
                  @Override
                  public Boolean apply(final Operation it) {
                    return Boolean.valueOf(it.getName().equals(change.oldElement.getElementName()));
                  }
                };
                List<Operation> samenames = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(Iterables.<Operation>filter(classifier_1.getOwnedElements(), Operation.class), _function_1));
                final IMethodDeclaration oldMethod = ((IMethodDeclaration) change.oldElement);
                final Function1<Operation, Boolean> _function_2 = new Function1<Operation, Boolean>() {
                  @Override
                  public Boolean apply(final Operation it) {
                    return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, oldMethod));
                  }
                };
                Operation op = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(samenames, _function_2));
                if ((op != null)) {
                  this.updateMethod(((org.eclipse.uml2.uml.Class) classifier_1), op, ((IMethodDeclaration) change.newElement));
                }
              }
            }
          } else {
            ICElement _parent_4 = method.getParent();
            if ((_parent_4 instanceof ITranslationUnit)) {
              ICElement _parent_5 = method.getParent();
              this.reverseSource(((ITranslationUnit) _parent_5));
            }
          }
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IStructure)) {
          _matched=true;
        }
        if (!_matched) {
          if ((change.oldElement instanceof IEnumeration)) {
            _matched=true;
          }
        }
        if (!_matched) {
          if ((change.oldElement instanceof ITypeDef)) {
            _matched=true;
          }
        }
        if (_matched) {
          IDeclaration declaration = ((IDeclaration) change.newElement);
          Type umlClassifier = this.getClassifier(this.getCorrespondingModel(declaration.getTranslationUnit()), 
            ((IDeclaration) change.oldElement), change.oldElement.getElementName(), declaration.getTranslationUnit());
          if ((umlClassifier != null)) {
            umlClassifier.setName(declaration.getElementName());
          }
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IEnumerator)) {
          _matched=true;
          final IEnumerator cppEnumerator = ((IEnumerator) change.newElement);
          ICElement _parent_6 = cppEnumerator.getParent();
          IEnumeration cppEnumeration = ((IEnumeration) _parent_6);
          Type enumeration = this.getClassifier(this.getCorrespondingModel(cppEnumeration.getTranslationUnit()), cppEnumeration, cppEnumeration.getElementName(), cppEnumeration.getTranslationUnit());
          if (((enumeration != null) && (enumeration instanceof Enumeration))) {
            final Function1<EnumerationLiteral, Boolean> _function_3 = new Function1<EnumerationLiteral, Boolean>() {
              @Override
              public Boolean apply(final EnumerationLiteral it) {
                String _name = it.getName();
                String _elementName = change.oldElement.getElementName();
                return Boolean.valueOf(Objects.equal(_name, _elementName));
              }
            };
            EnumerationLiteral enumerator = IterableExtensions.<EnumerationLiteral>head(IterableExtensions.<EnumerationLiteral>filter(Iterables.<EnumerationLiteral>filter(((Enumeration) enumeration).getOwnedLiterals(), EnumerationLiteral.class), _function_3));
            if ((enumerator != null)) {
              enumerator.setName(cppEnumerator.getElementName());
            }
          }
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof ITranslationUnit)) {
          _matched=true;
          List<Type> _xblockexpression = null;
          {
            ITranslationUnit itu = ((ITranslationUnit) change.newElement);
            List<Type> _xifexpression = null;
            boolean _isSourceUnit = itu.isSourceUnit();
            if (_isSourceUnit) {
              this.reverseSource(itu);
            } else {
              List<Type> _xifexpression_1 = null;
              boolean _isHeaderUnit = itu.isHeaderUnit();
              if (_isHeaderUnit) {
                _xifexpression_1 = this.reverseHeader(itu);
              }
              _xifexpression = _xifexpression_1;
            }
            _xblockexpression = _xifexpression;
          }
          _switchResult = _xblockexpression;
        }
      }
      return _switchResult;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Object removeFromModel(final CppChangeObject change) {
    try {
      Object _switchResult = null;
      final ICElement _switchValue = change.oldElement;
      boolean _matched = false;
      if ((change.oldElement instanceof IField)) {
        _matched=true;
        boolean _xblockexpression = false;
        {
          final IField field = ((IField) change.oldElement);
          boolean _xifexpression = false;
          ICElement _parent = field.getParent();
          if ((_parent instanceof IStructure)) {
            boolean _xblockexpression_1 = false;
            {
              ICElement _parent_1 = field.getParent();
              IStructure parentStructure = ((IStructure) _parent_1);
              Type classifier = this.getClassifier(this.getCorrespondingModel(parentStructure.getTranslationUnit()), parentStructure, parentStructure.getElementName(), parentStructure.getTranslationUnit());
              boolean _xifexpression_1 = false;
              if ((classifier != null)) {
                boolean _xifexpression_2 = false;
                boolean _isExist = this.isExist(classifier, change.oldElement.getElementName());
                if (_isExist) {
                  boolean _xblockexpression_2 = false;
                  {
                    final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
                      @Override
                      public Boolean apply(final Property it) {
                        return Boolean.valueOf(it.getName().equals(change.oldElement.getElementName()));
                      }
                    };
                    Property prop = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(Iterables.<Property>filter(classifier.getOwnedElements(), Property.class), _function));
                    boolean _xifexpression_3 = false;
                    if ((prop != null)) {
                      boolean _xifexpression_4 = false;
                      if ((classifier instanceof org.eclipse.uml2.uml.Class)) {
                        EList<Property> _ownedAttributes = ((org.eclipse.uml2.uml.Class)classifier).getOwnedAttributes();
                        _xifexpression_4 = _ownedAttributes.remove(prop);
                      } else {
                        boolean _xifexpression_5 = false;
                        if ((classifier instanceof DataType)) {
                          EList<Property> _ownedAttributes_1 = ((DataType)classifier).getOwnedAttributes();
                          _xifexpression_5 = _ownedAttributes_1.remove(prop);
                        }
                        _xifexpression_4 = _xifexpression_5;
                      }
                      _xifexpression_3 = _xifexpression_4;
                    }
                    _xblockexpression_2 = _xifexpression_3;
                  }
                  _xifexpression_2 = _xblockexpression_2;
                }
                _xifexpression_1 = _xifexpression_2;
              }
              _xblockexpression_1 = _xifexpression_1;
            }
            _xifexpression = _xblockexpression_1;
          }
          _xblockexpression = _xifexpression;
        }
        _switchResult = Boolean.valueOf(_xblockexpression);
      }
      if (!_matched) {
        if ((change.oldElement instanceof IMethodDeclaration)) {
          _matched=true;
          boolean _xblockexpression_1 = false;
          {
            final IMethodDeclaration method = ((IMethodDeclaration) change.oldElement);
            boolean _xifexpression = false;
            ICElement _parent = method.getParent();
            if ((_parent instanceof IStructure)) {
              boolean _xblockexpression_2 = false;
              {
                ICElement _parent_1 = method.getParent();
                IStructure parentStructure = ((IStructure) _parent_1);
                Type classifier = this.getClassifier(this.getCorrespondingModel(parentStructure.getTranslationUnit()), parentStructure, parentStructure.getElementName(), parentStructure.getTranslationUnit());
                boolean _xifexpression_1 = false;
                if ((classifier != null)) {
                  boolean _xifexpression_2 = false;
                  boolean _isExist = this.isExist(classifier, change.oldElement.getElementName());
                  if (_isExist) {
                    boolean _xblockexpression_3 = false;
                    {
                      final Function1<Operation, Boolean> _function = new Function1<Operation, Boolean>() {
                        @Override
                        public Boolean apply(final Operation it) {
                          return Boolean.valueOf(it.getName().equals(method.getElementName()));
                        }
                      };
                      List<Operation> samenames = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(Iterables.<Operation>filter(classifier.getOwnedElements(), Operation.class), _function));
                      final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
                        @Override
                        public Boolean apply(final Operation it) {
                          return Boolean.valueOf(ReverseCpp2Uml.this.isSameSignature(it, method));
                        }
                      };
                      Operation op = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(samenames, _function_1));
                      boolean _xifexpression_3 = false;
                      if ((op != null)) {
                        boolean _xifexpression_4 = false;
                        if ((classifier instanceof org.eclipse.uml2.uml.Class)) {
                          EList<Operation> _ownedOperations = ((org.eclipse.uml2.uml.Class)classifier).getOwnedOperations();
                          _xifexpression_4 = _ownedOperations.remove(op);
                        }
                        _xifexpression_3 = _xifexpression_4;
                      }
                      _xblockexpression_3 = _xifexpression_3;
                    }
                    _xifexpression_2 = _xblockexpression_3;
                  }
                  _xifexpression_1 = _xifexpression_2;
                }
                _xblockexpression_2 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_2;
            }
            _xblockexpression_1 = _xifexpression;
          }
          _switchResult = Boolean.valueOf(_xblockexpression_1);
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IStructure)) {
          _matched=true;
        }
        if (!_matched) {
          if ((change.oldElement instanceof IEnumeration)) {
            _matched=true;
          }
        }
        if (!_matched) {
          if ((change.oldElement instanceof ITypeDef)) {
            _matched=true;
          }
        }
        if (_matched) {
          boolean _xblockexpression_2 = false;
          {
            final IDeclaration declaration = ((IDeclaration) change.oldElement);
            boolean _xifexpression = false;
            if ((change.parent instanceof IStructure)) {
              boolean _xblockexpression_3 = false;
              {
                IStructure parentStructure = ((IStructure) change.parent);
                Type parentClassifier = this.getClassifier(this.getCorrespondingModel(parentStructure.getTranslationUnit()), parentStructure, parentStructure.getElementName(), parentStructure.getTranslationUnit());
                final Function1<Type, Boolean> _function = new Function1<Type, Boolean>() {
                  @Override
                  public Boolean apply(final Type it) {
                    String _name = it.getName();
                    String _elementName = declaration.getElementName();
                    return Boolean.valueOf(Objects.equal(_name, _elementName));
                  }
                };
                Type childClassifier = IterableExtensions.<Type>head(IterableExtensions.<Type>filter(Iterables.<Type>filter(parentClassifier.getOwnedElements(), Type.class), _function));
                boolean _xifexpression_1 = false;
                if (((childClassifier != null) && (parentClassifier instanceof org.eclipse.uml2.uml.Class))) {
                  EList<Classifier> _nestedClassifiers = ((org.eclipse.uml2.uml.Class) parentClassifier).getNestedClassifiers();
                  _xifexpression_1 = _nestedClassifiers.remove(((Classifier) childClassifier));
                }
                _xblockexpression_3 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_3;
            } else {
              boolean _xblockexpression_4 = false;
              {
                org.eclipse.uml2.uml.Package containerPackage = this.getContainerPackage(declaration.getTranslationUnit());
                final Function1<Type, Boolean> _function = new Function1<Type, Boolean>() {
                  @Override
                  public Boolean apply(final Type it) {
                    String _name = it.getName();
                    String _elementName = declaration.getElementName();
                    return Boolean.valueOf(Objects.equal(_name, _elementName));
                  }
                };
                Type childClassifier = IterableExtensions.<Type>head(IterableExtensions.<Type>filter(Iterables.<Type>filter(containerPackage.getOwnedElements(), Type.class), _function));
                EList<Type> _ownedTypes = containerPackage.getOwnedTypes();
                _xblockexpression_4 = _ownedTypes.remove(childClassifier);
              }
              _xifexpression = _xblockexpression_4;
            }
            _xblockexpression_2 = _xifexpression;
          }
          _switchResult = Boolean.valueOf(_xblockexpression_2);
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof IEnumerator)) {
          _matched=true;
          boolean _xblockexpression_3 = false;
          {
            IEnumerator cppEnumerator = ((IEnumerator) change.oldElement);
            IPath _path = cppEnumerator.getPath();
            IEnumeration cppEnumeration = ((IEnumeration) _path);
            Type enumeration = this.getClassifier(this.getCorrespondingModel(cppEnumeration.getTranslationUnit()), cppEnumeration, cppEnumeration.getElementName(), cppEnumeration.getTranslationUnit());
            boolean _xifexpression = false;
            if (((enumeration != null) && (enumeration instanceof Enumeration))) {
              boolean _xblockexpression_4 = false;
              {
                EnumerationLiteral enumerator = ((Enumeration) enumeration).getOwnedLiteral(change.oldElement.getElementName());
                boolean _xifexpression_1 = false;
                if ((enumerator != null)) {
                  EList<EnumerationLiteral> _ownedLiterals = ((Enumeration) enumeration).getOwnedLiterals();
                  _xifexpression_1 = _ownedLiterals.remove(enumerator);
                }
                _xblockexpression_4 = _xifexpression_1;
              }
              _xifexpression = _xblockexpression_4;
            }
            _xblockexpression_3 = _xifexpression;
          }
          _switchResult = Boolean.valueOf(_xblockexpression_3);
        }
      }
      if (!_matched) {
        if ((change.oldElement instanceof ITranslationUnit)) {
          _matched=true;
          List<Type> _xblockexpression_4 = null;
          {
            ITranslationUnit itu = ((ITranslationUnit) change.newElement);
            _xblockexpression_4 = this.getOrCreateClassifier(this.getCorrespondingModel(itu), itu);
          }
          _switchResult = _xblockexpression_4;
        }
      }
      return _switchResult;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Object compareUml2Cpp(final Element umlElement, final ICElement icElement) {
    Object _xifexpression = null;
    if (((umlElement instanceof org.eclipse.uml2.uml.Package) && (icElement instanceof ICContainer))) {
      _xifexpression = null;
    }
    return _xifexpression;
  }
  
  private Object comparePackage(final org.eclipse.uml2.uml.Package pack, final ICContainer container) {
    try {
      Object _xblockexpression = null;
      {
        List<ICContainer> containerChildren = IterableExtensions.<ICContainer>toList(Iterables.<ICContainer>filter(((Iterable<?>)Conversions.doWrapArray(container.getChildren())), ICContainer.class));
        Object _xifexpression = null;
        int _size = pack.getNestedPackages().size();
        int _size_1 = containerChildren.size();
        boolean _notEquals = (_size != _size_1);
        if (_notEquals) {
          _xifexpression = null;
        } else {
          Iterator<ICContainer> iterator = containerChildren.iterator();
          EList<org.eclipse.uml2.uml.Package> _nestedPackages = pack.getNestedPackages();
          for (final org.eclipse.uml2.uml.Package child : _nestedPackages) {
            this.syncName(child, iterator.next());
          }
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void syncName(final NamedElement umlElement, final ICElement icElement) {
    boolean _equals = icElement.getElementName().equals(umlElement.getName());
    boolean _not = (!_equals);
    if (_not) {
      umlElement.setName(icElement.getElementName());
    }
  }
  
  private void comparePackageableElement(final PackageableElement umlElement, final IParent parent) {
    if (((!(umlElement instanceof org.eclipse.uml2.uml.Package)) || (!(umlElement instanceof Classifier)))) {
      return;
    }
    Iterable<Type> typeChildren = Iterables.<Type>filter(umlElement.getOwnedElements(), Type.class);
    List<ICElement> structureChildren = this.reverse_utils.getAllIStructures(parent, false, false, this.m_project);
    int _size = IterableExtensions.size(typeChildren);
    int _size_1 = structureChildren.size();
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
    } else {
    }
  }
  
  private org.eclipse.uml2.uml.Package getContainerPackage(final ITranslationUnit unit) {
    org.eclipse.uml2.uml.Package _xblockexpression = null;
    {
      Model model = this.getCorrespondingModel(unit);
      String unitPath = unit.getFile().getProjectRelativePath().toString();
      if (((unitPath == null) || unitPath.equals(""))) {
        unitPath = unit.getLocation().toString();
      }
      String _name = model.getName();
      String _plus = ("/" + _name);
      String _plus_1 = (_plus + "/");
      final String[] arrayNames = IterableExtensions.<String>last(((Iterable<String>)Conversions.doWrapArray(unitPath.split(_plus_1)))).split("/");
      final ArrayList<String> names = new ArrayList<String>();
      final Consumer<String> _function = new Consumer<String>() {
        @Override
        public void accept(final String it) {
          names.add(it);
        }
      };
      ((List<String>)Conversions.doWrapArray(arrayNames)).forEach(_function);
      org.eclipse.uml2.uml.Package parentPack = model;
      for (final String name : names) {
        boolean _equals = name.equals(unit.getElementName());
        boolean _not = (!_equals);
        if (_not) {
          org.eclipse.uml2.uml.Package _nestedPackage = parentPack.getNestedPackage(name);
          boolean _tripleEquals = (_nestedPackage == null);
          if (_tripleEquals) {
            parentPack.createNestedPackage(name);
          }
          parentPack = parentPack.getNestedPackage(name);
        }
      }
      _xblockexpression = parentPack;
    }
    return _xblockexpression;
  }
  
  private Type getClassifier(final Model model, final ICElement declaration, final String name, final ITranslationUnit unit) throws Exception {
    if (((this.map.get(declaration) != null) && (this.map.get(declaration) instanceof Type))) {
      EObject _get = this.map.get(declaration);
      return ((Type) _get);
    }
    Type ret = null;
    org.eclipse.uml2.uml.Package parentPack = this.getContainerPackage(unit);
    Classifier parentClass = null;
    ICElement _parent = declaration.getParent();
    if ((_parent instanceof IStructure)) {
      Type _uMLType = this.getUMLType(declaration.getParent().getElementName(), unit, this.getContextNamespaces(declaration));
      parentClass = ((Classifier) _uMLType);
      EObject _createOrgetClassifier = this.createOrgetClassifier(parentClass, declaration, false);
      ret = ((Type) _createOrgetClassifier);
    } else {
      EObject _createOrgetClassifier_1 = this.createOrgetClassifier(parentPack, declaration, false);
      ret = ((Type) _createOrgetClassifier_1);
    }
    return ret;
  }
  
  private EObject createOrgetClassifier(final PackageableElement packagebleElement, final ICElement ideclaration, final boolean createContent) {
    try {
      EObject _xblockexpression = null;
      {
        PackageableElement packageble = packagebleElement;
        if (((((((ideclaration.getElementType() != ICElement.C_CLASS) && (ideclaration.getElementType() != ICElement.C_TEMPLATE_CLASS)) && 
          (ideclaration.getElementType() != ICElement.C_ENUMERATION)) && (ideclaration.getElementType() != ICElement.C_STRUCT)) && 
          (ideclaration.getElementType() != ICElement.C_UNION)) && (!(ideclaration instanceof ITypeDef)))) {
          return null;
        }
        NamedElement existing = null;
        if (((!(packageble instanceof org.eclipse.uml2.uml.Package)) && (!(packageble instanceof org.eclipse.uml2.uml.Class)))) {
          if (((packageble instanceof DataType) && (ideclaration instanceof ITypeDef))) {
            packageble = packagebleElement.getNearestPackage();
          } else {
            return null;
          }
        }
        if (((this.map.get(ideclaration) != null) && (this.map.get(ideclaration) instanceof Type))) {
          EObject _get = this.map.get(ideclaration);
          existing = ((Type) _get);
        }
        if ((existing == null)) {
          final Function1<Type, Boolean> _function = new Function1<Type, Boolean>() {
            @Override
            public Boolean apply(final Type it) {
              return Boolean.valueOf(it.getName().equals(ideclaration.getElementName()));
            }
          };
          Iterable<Type> namedElements = IterableExtensions.<Type>filter(Iterables.<Type>filter(packageble.getOwnedElements(), Type.class), _function);
          boolean _isEmpty = IterableExtensions.isEmpty(namedElements);
          boolean _not = (!_isEmpty);
          if (_not) {
            existing = IterableExtensions.<Type>head(namedElements);
          }
        }
        if ((existing != null)) {
          return existing;
        }
        String _elementName = ideclaration.getElementName();
        String _plus = ("Creating type " + _elementName);
        this.m_monitor.subTask(_plus);
        if ((ideclaration instanceof ITypeDef)) {
          if ((existing != null)) {
            return existing;
          }
          final ITypeDef typedef = ((ITypeDef) ideclaration);
          PrimitiveType _createPrimitiveType = UMLFactory.eINSTANCE.createPrimitiveType();
          final Procedure1<PrimitiveType> _function_1 = new Procedure1<PrimitiveType>() {
            @Override
            public void apply(final PrimitiveType it) {
              it.setName(typedef.getElementName());
            }
          };
          PrimitiveType prinmitiveType = ObjectExtensions.<PrimitiveType>operator_doubleArrow(_createPrimitiveType, _function_1);
          if ((packageble instanceof org.eclipse.uml2.uml.Package)) {
            EList<Type> _ownedTypes = ((org.eclipse.uml2.uml.Package) packageble).getOwnedTypes();
            _ownedTypes.add(prinmitiveType);
          } else {
            EList<Classifier> _nestedClassifiers = ((org.eclipse.uml2.uml.Class) packageble).getNestedClassifiers();
            _nestedClassifiers.add(prinmitiveType);
          }
          StereotypeUtil.apply(prinmitiveType, Typedef.class);
          if (((Objects.equal(typedef.getTypeName(), "struct") || Objects.equal(typedef.getTypeName(), "class")) || Objects.equal(typedef.getTypeName(), "enum"))) {
            ICElement _parent = typedef.getParent();
            int pos = ((List<ICElement>)Conversions.doWrapArray(((IParent) _parent).getChildren())).indexOf(typedef);
            if ((pos > 0)) {
              ICElement _parent_1 = typedef.getParent();
              ICElement childStruct = ((IParent) _parent_1).getChildren()[(pos - 1)];
              if ((((childStruct instanceof IStructure) || (childStruct instanceof IEnumeration)) && Objects.equal(childStruct.getElementName(), ""))) {
                String source = ((ISourceReference) childStruct).getSource();
                Typedef _stereotypeApplication = UMLUtil.<Typedef>getStereotypeApplication(prinmitiveType, Typedef.class);
                String _replace = source.replace("typedef", "");
                String _plus_1 = (_replace + " typeName");
                _stereotypeApplication.setDefinition(_plus_1);
              }
            }
          } else {
            IASTNode typedefNode = this.findEnclosingNode(typedef);
            String rawSignature = typedefNode.getRawSignature().replaceAll("\\n", "").replaceAll("\\r", "").replaceAll(";", "").replaceAll("\\s+", " ").trim();
            Pattern pattern = Pattern.compile("(\\()(\\s*)(\\*)(.*)(\\))(\\s*)(\\()(.*)(\\))");
            Matcher matcher = pattern.matcher(rawSignature);
            boolean _find = matcher.find();
            if (_find) {
              final String typeName = rawSignature.replaceFirst(Pattern.quote(typedef.getElementName()), "typeName");
              Typedef _stereotypeApplication_1 = UMLUtil.<Typedef>getStereotypeApplication(prinmitiveType, Typedef.class);
              _stereotypeApplication_1.setDefinition(typeName);
            } else {
              Typedef _stereotypeApplication_2 = UMLUtil.<Typedef>getStereotypeApplication(prinmitiveType, Typedef.class);
              _stereotypeApplication_2.setDefinition(typedef.getTypeName());
            }
          }
          this.map.put(ideclaration, prinmitiveType);
          return prinmitiveType;
        }
        boolean structIsClassLike = false;
        if (((ideclaration.getElementType() == ICElement.C_STRUCT) || (ideclaration.getElementType() == ICElement.C_UNION))) {
          final IStructure iStructure = ((IStructure) ideclaration);
          int i = 0;
          while (((!structIsClassLike) && (i < ((List<ICElement>)Conversions.doWrapArray(iStructure.getChildren())).size()))) {
            {
              ICElement child = iStructure.getChildren()[i];
              if (((child instanceof IMethodDeclaration) || (child instanceof IStructure))) {
                structIsClassLike = true;
              }
              i++;
            }
          }
        }
        EObject _xifexpression = null;
        if ((((ideclaration.getElementType() == ICElement.C_CLASS) || (ideclaration.getElementType() == ICElement.C_TEMPLATE_CLASS)) || structIsClassLike)) {
          final IStructure iStructure_1 = ((IStructure) ideclaration);
          org.eclipse.uml2.uml.Class temp = null;
          if ((existing != null)) {
            temp = ((org.eclipse.uml2.uml.Class) existing);
          } else {
            if ((packageble instanceof org.eclipse.uml2.uml.Package)) {
              temp = ((org.eclipse.uml2.uml.Package) packageble).createOwnedClass(iStructure_1.getElementName(), iStructure_1.isAbstract());
            } else {
              Classifier _createNestedClassifier = ((org.eclipse.uml2.uml.Class) packageble).createNestedClassifier(iStructure_1.getElementName(), 
                UMLPackage.Literals.CLASS);
              temp = ((org.eclipse.uml2.uml.Class) _createNestedClassifier);
              temp.setIsAbstract(iStructure_1.isAbstract());
            }
          }
          final org.eclipse.uml2.uml.Class classifier = temp;
          final Consumer<ICElement> _function_2 = new Consumer<ICElement>() {
            @Override
            public void accept(final ICElement it) {
              int _elementType = it.getElementType();
              boolean _equals = (_elementType == ICElement.C_FIELD);
              if (_equals) {
                ReverseCpp2Uml.this.createProperty(((IField) it), classifier);
              } else {
                if ((it instanceof IMethodDeclaration)) {
                  ReverseCpp2Uml.this.createMethod(((IMethodDeclaration)it), classifier);
                }
              }
            }
          };
          ((List<ICElement>)Conversions.doWrapArray(iStructure_1.getChildren())).forEach(_function_2);
          final Function1<String, Type> _function_3 = new Function1<String, Type>() {
            @Override
            public Type apply(final String it) {
              return ReverseCpp2Uml.this.getUMLType(it, iStructure_1.getTranslationUnit(), ReverseCpp2Uml.this.getContextNamespaces(iStructure_1));
            }
          };
          List<Type> superTypes = ListExtensions.<String, Type>map(((List<String>)Conversions.doWrapArray(iStructure_1.getSuperClassesNames())), _function_3);
          final Consumer<Type> _function_4 = new Consumer<Type>() {
            @Override
            public void accept(final Type it) {
              if (((it != null) && (it instanceof Classifier))) {
                Generalization generalization = classifier.createGeneralization(((Classifier) it));
                Visibility visibilitySt = UMLUtil.<Visibility>getStereotypeApplication(generalization, Visibility.class);
                if ((visibilitySt == null)) {
                  StereotypeUtil.apply(generalization, Visibility.class);
                  visibilitySt = UMLUtil.<Visibility>getStereotypeApplication(generalization, Visibility.class);
                }
                final ASTAccessVisibility visibility = iStructure_1.getSuperClassAccess(it.getName());
                if ((visibility != null)) {
                  visibilitySt.setValue(visibility.name().toLowerCase());
                }
              }
            }
          };
          superTypes.forEach(_function_4);
          this.map.put(ideclaration, classifier);
          int _elementType = iStructure_1.getElementType();
          boolean _equals = (_elementType == ICElement.C_TEMPLATE_CLASS);
          if (_equals) {
            IStructureTemplate istructureTemplate = ((IStructureTemplate) iStructure_1);
            StereotypeUtil.apply(classifier, Template.class);
            Template _stereotypeApplication_3 = UMLUtil.<Template>getStereotypeApplication(classifier, Template.class);
            _stereotypeApplication_3.setDeclaration(istructureTemplate.getTemplateSignature());
          }
          return classifier;
        } else {
          EObject _xifexpression_1 = null;
          int _elementType_1 = ideclaration.getElementType();
          boolean _equals_1 = (_elementType_1 == ICElement.C_ENUMERATION);
          if (_equals_1) {
            EObject _xblockexpression_1 = null;
            {
              Enumeration temp_1 = null;
              final IEnumeration iStructure_2 = ((IEnumeration) ideclaration);
              if ((existing != null)) {
                temp_1 = ((Enumeration) existing);
              } else {
                if ((packageble instanceof org.eclipse.uml2.uml.Package)) {
                  temp_1 = ((org.eclipse.uml2.uml.Package) packageble).createOwnedEnumeration(iStructure_2.getElementName());
                } else {
                  Classifier _createNestedClassifier_1 = ((org.eclipse.uml2.uml.Class) packageble).createNestedClassifier(iStructure_2.getElementName(), 
                    UMLPackage.Literals.ENUMERATION);
                  temp_1 = ((Enumeration) _createNestedClassifier_1);
                }
              }
              final Enumeration enumeration = temp_1;
              final Consumer<ICElement> _function_5 = new Consumer<ICElement>() {
                @Override
                public void accept(final ICElement it) {
                  if ((it instanceof IEnumerator)) {
                    EnumerationLiteral literal = enumeration.createOwnedLiteral(((IEnumerator)it).getElementName());
                    String _constantExpression = ((IEnumerator) it).getConstantExpression();
                    boolean _tripleNotEquals = (_constantExpression != null);
                    if (_tripleNotEquals) {
                      EList<Element> _ownedElements = literal.getOwnedElements();
                      for (final Element element : _ownedElements) {
                        if ((element instanceof ValueSpecification)) {
                          boolean _equals = ((ValueSpecification) element).getName().equals("defaultValue");
                          if (_equals) {
                            ((ValueSpecification)element).destroy();
                          }
                        }
                      }
                      if (((((IEnumerator) it).getConstantExpression() != null) && (!((IEnumerator) it).getConstantExpression().equals("")))) {
                        ValueSpecification valueSpecification = literal.createSpecification("defaultValue", literal.getEnumeration(), UMLPackage.Literals.OPAQUE_EXPRESSION);
                        if ((valueSpecification instanceof OpaqueExpression)) {
                          OpaqueExpression opaqueExpression = ((OpaqueExpression) valueSpecification);
                          opaqueExpression.getLanguages().add(ReverseCpp2Uml.Cpp_LangID);
                          opaqueExpression.getBodies().add(((IEnumerator) it).getConstantExpression());
                        }
                      }
                    }
                  }
                }
              };
              ((List<ICElement>)Conversions.doWrapArray(iStructure_2.getChildren())).forEach(_function_5);
              _xblockexpression_1 = this.map.put(ideclaration, enumeration);
            }
            _xifexpression_1 = _xblockexpression_1;
          } else {
            EObject _xifexpression_2 = null;
            if (((!structIsClassLike) && ((ideclaration.getElementType() == ICElement.C_STRUCT) || (ideclaration.getElementType() == ICElement.C_UNION)))) {
              EObject _xblockexpression_2 = null;
              {
                DataType temp_1 = null;
                final IStructure iStructure_2 = ((IStructure) ideclaration);
                if ((existing != null)) {
                  temp_1 = ((DataType) existing);
                } else {
                  if ((packageble instanceof org.eclipse.uml2.uml.Package)) {
                    Type _createOwnedType = ((org.eclipse.uml2.uml.Package) packageble).createOwnedType(iStructure_2.getElementName(), 
                      UMLPackage.Literals.DATA_TYPE);
                    temp_1 = ((DataType) _createOwnedType);
                  } else {
                    Classifier _createNestedClassifier_1 = ((org.eclipse.uml2.uml.Class) packageble).createNestedClassifier(iStructure_2.getElementName(), 
                      UMLPackage.Literals.DATA_TYPE);
                    temp_1 = ((DataType) _createNestedClassifier_1);
                  }
                }
                final DataType dataType = temp_1;
                final Consumer<ICElement> _function_5 = new Consumer<ICElement>() {
                  @Override
                  public void accept(final ICElement it) {
                    int _elementType = it.getElementType();
                    boolean _equals = (_elementType == ICElement.C_FIELD);
                    if (_equals) {
                      ReverseCpp2Uml.this.createProperty(((IField) it), dataType);
                    }
                  }
                };
                ((List<ICElement>)Conversions.doWrapArray(iStructure_2.getChildren())).forEach(_function_5);
                final Function1<String, Type> _function_6 = new Function1<String, Type>() {
                  @Override
                  public Type apply(final String it) {
                    return ReverseCpp2Uml.this.getUMLType(it, iStructure_2.getTranslationUnit(), ReverseCpp2Uml.this.getContextNamespaces(iStructure_2));
                  }
                };
                List<Type> superTypes_1 = ListExtensions.<String, Type>map(((List<String>)Conversions.doWrapArray(iStructure_2.getSuperClassesNames())), _function_6);
                final Consumer<Type> _function_7 = new Consumer<Type>() {
                  @Override
                  public void accept(final Type it) {
                    if (((it != null) && (it instanceof Classifier))) {
                      Generalization generalization = dataType.createGeneralization(((Classifier) it));
                      Visibility visibilitySt = UMLUtil.<Visibility>getStereotypeApplication(generalization, Visibility.class);
                      if ((visibilitySt == null)) {
                        StereotypeUtil.apply(generalization, Visibility.class);
                        visibilitySt = UMLUtil.<Visibility>getStereotypeApplication(generalization, Visibility.class);
                      }
                      final ASTAccessVisibility visibility = iStructure_2.getSuperClassAccess(it.getName());
                      if ((visibility != null)) {
                        visibilitySt.setValue(visibility.name().toLowerCase());
                      }
                    }
                  }
                };
                superTypes_1.forEach(_function_7);
                _xblockexpression_2 = this.map.put(ideclaration, dataType);
              }
              _xifexpression_2 = _xblockexpression_2;
            }
            _xifexpression_1 = _xifexpression_2;
          }
          _xifexpression = _xifexpression_1;
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Classifier createOrgetTemplateParameter(final NamedElement element, final String parameterTypeName, final String keyWord) {
    Classifier ret = null;
    TemplateSignature templateSignature = null;
    if ((element instanceof Classifier)) {
      Classifier classifier = ((Classifier) element);
      if (((classifier.getOwnedTemplateSignature() == null) || 
        (!(classifier.getOwnedTemplateSignature() instanceof RedefinableTemplateSignature)))) {
        TemplateSignature _createOwnedTemplateSignature = classifier.createOwnedTemplateSignature(
          UMLPackage.Literals.REDEFINABLE_TEMPLATE_SIGNATURE);
        templateSignature = ((RedefinableTemplateSignature) _createOwnedTemplateSignature);
        ((RedefinableTemplateSignature) templateSignature).setName(ReverseCpp2Uml.TEMPLATE_PARAMETER_SIGNATURE_NAME);
      }
      TemplateSignature _ownedTemplateSignature = classifier.getOwnedTemplateSignature();
      templateSignature = ((RedefinableTemplateSignature) _ownedTemplateSignature);
    } else {
      if ((element instanceof Operation)) {
        Operation operation = ((Operation) element);
        if (((operation.getOwnedTemplateSignature() == null) || 
          (!(operation.getOwnedTemplateSignature() instanceof TemplateSignature)))) {
          TemplateSignature _createOwnedTemplateSignature_1 = operation.createOwnedTemplateSignature(
            UMLPackage.Literals.TEMPLATE_SIGNATURE);
          templateSignature = ((TemplateSignature) _createOwnedTemplateSignature_1);
        }
        TemplateSignature _ownedTemplateSignature_1 = operation.getOwnedTemplateSignature();
        templateSignature = ((TemplateSignature) _ownedTemplateSignature_1);
      } else {
        return null;
      }
    }
    Iterable<ClassifierTemplateParameter> classifierTemplates = Iterables.<ClassifierTemplateParameter>filter(templateSignature.getOwnedParameters(), ClassifierTemplateParameter.class);
    final Function1<ClassifierTemplateParameter, Boolean> _function = new Function1<ClassifierTemplateParameter, Boolean>() {
      @Override
      public Boolean apply(final ClassifierTemplateParameter it) {
        ParameterableElement _ownedParameteredElement = it.getOwnedParameteredElement();
        return Boolean.valueOf((_ownedParameteredElement instanceof Classifier));
      }
    };
    Iterable<ClassifierTemplateParameter> classifierTemplatesContainClassifier = IterableExtensions.<ClassifierTemplateParameter>filter(classifierTemplates, _function);
    final Function1<ClassifierTemplateParameter, ParameterableElement> _function_1 = new Function1<ClassifierTemplateParameter, ParameterableElement>() {
      @Override
      public ParameterableElement apply(final ClassifierTemplateParameter it) {
        return it.getOwnedParameteredElement();
      }
    };
    Iterable<ParameterableElement> containedClassifiers = IterableExtensions.<ClassifierTemplateParameter, ParameterableElement>map(classifierTemplatesContainClassifier, _function_1);
    final Function1<Classifier, Boolean> _function_2 = new Function1<Classifier, Boolean>() {
      @Override
      public Boolean apply(final Classifier it) {
        return Boolean.valueOf(it.getName().equals(parameterTypeName));
      }
    };
    ret = IterableExtensions.<Classifier>head(IterableExtensions.<Classifier>filter(Iterables.<Classifier>filter(containedClassifiers, Classifier.class), _function_2));
    if ((ret == null)) {
      TemplateParameter _createOwnedParameter = templateSignature.createOwnedParameter(
        UMLPackage.Literals.CLASSIFIER_TEMPLATE_PARAMETER);
      ClassifierTemplateParameter classifierTemplate = ((ClassifierTemplateParameter) _createOwnedParameter);
      ParameterableElement _createOwnedParameteredElement = classifierTemplate.createOwnedParameteredElement(UMLPackage.Literals.CLASS);
      ret = ((Classifier) _createOwnedParameteredElement);
      ret.setName(parameterTypeName);
      classifierTemplate.addKeyword(keyWord);
    }
    return ret;
  }
  
  private IASTNodeSelector getSelector(final ITranslationUnit unit) {
    try {
      IASTTranslationUnit ast = null;
      if ((unit != null)) {
        IASTTranslationUnit _get = this.translationUnitToASTTranslationUnitMap.get(unit);
        boolean _tripleNotEquals = (_get != null);
        if (_tripleNotEquals) {
          return this.translationUnitToASTTranslationUnitMap.get(unit).getNodeSelector(null);
        } else {
          ast = unit.getAST(this.index, ITranslationUnit.AST_CONFIGURE_USING_SOURCE_CONTEXT);
          if ((ast != null)) {
            this.translationUnitToASTTranslationUnitMap.put(unit, ast);
            return ast.getNodeSelector(null);
          }
        }
      }
      return null;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public void createMethod(final IMethodDeclaration method, final org.eclipse.uml2.uml.Class classifier) {
    IASTFunctionDeclarator declarator = this.getDeclarator(method);
    if ((declarator != null)) {
      final Operation op = classifier.createOwnedOperation(method.getElementName(), null, null);
      this.updateMethod(classifier, op, method);
    }
  }
  
  private IASTNode findEnclosingNode(final ISourceReference source) {
    try {
      IASTNode _xblockexpression = null;
      {
        IASTNodeSelector selector = this.getSelector(source.getTranslationUnit());
        ISourceRange range = source.getSourceRange();
        _xblockexpression = selector.findEnclosingNode(range.getStartPos(), range.getLength());
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private IASTFunctionDeclarator getDeclarator(final IMethodDeclaration method) {
    IASTFunctionDeclarator declarator = null;
    IASTNode node = this.findEnclosingNode(method);
    if ((node instanceof ICPPASTFunctionDefinition)) {
      declarator = ((ICPPASTFunctionDefinition) node).getDeclarator();
    } else {
      if ((node instanceof IASTSimpleDeclaration)) {
        IASTSimpleDeclaration declaration = ((IASTSimpleDeclaration) node);
        declarator = IterableExtensions.<IASTFunctionDeclarator>head(Iterables.<IASTFunctionDeclarator>filter(((Iterable<?>)Conversions.doWrapArray(declaration.getDeclarators())), IASTFunctionDeclarator.class));
      } else {
        if ((node instanceof ICPPASTTemplateDeclaration)) {
          ICPPASTTemplateDeclaration declaration_1 = ((ICPPASTTemplateDeclaration) node);
          IASTNode[] chidren = declaration_1.getChildren();
          for (final IASTNode child : chidren) {
            if ((child instanceof IASTSimpleDeclaration)) {
              declarator = IterableExtensions.<IASTFunctionDeclarator>head(Iterables.<IASTFunctionDeclarator>filter(((Iterable<?>)Conversions.doWrapArray(((IASTSimpleDeclaration)child).getDeclarators())), IASTFunctionDeclarator.class));
            }
          }
        }
      }
    }
    return declarator;
  }
  
  private List<String> getKeywords(final IMethodDeclaration method) {
    try {
      IASTNode node = this.findEnclosingNode(method);
      List<String> keywords = new ArrayList<String>();
      if ((node instanceof ICPPASTTemplateDeclaration)) {
        ICPPASTTemplateDeclaration declaration = ((ICPPASTTemplateDeclaration) node);
        IASTNode[] chidren = declaration.getChildren();
        for (final IASTNode child : chidren) {
          if ((child instanceof ICPPASTTemplateParameter)) {
            IToken token = ((ICPPASTTemplateParameter)child).getSyntax();
            String keyword = "class";
            while ((token != null)) {
              {
                int _type = token.getType();
                boolean _equals = (_type == IToken.t_typename);
                if (_equals) {
                  keyword = "typename";
                }
                token = token.getNext();
              }
            }
            keywords.add(keyword);
          }
        }
      }
      return keywords;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private String getBody(final IMethodDeclaration method) {
    IASTFunctionDeclarator declarator = null;
    IASTNode node = this.findEnclosingNode(method);
    String body = null;
    if ((node instanceof ICPPASTFunctionDefinition)) {
      declarator = ((ICPPASTFunctionDefinition) node).getDeclarator();
      body = BatchReverseFunctionBody.getBody(method.getTranslationUnit(), ((ICPPASTFunctionDefinition) node));
    } else {
      if ((node instanceof IASTFunctionDeclarator)) {
      }
    }
    return body;
  }
  
  private IASTNode getRootNode(final IASTNode fromNode) {
    IASTNode _parent = fromNode.getParent();
    boolean _tripleEquals = (_parent == null);
    if (_tripleEquals) {
      return fromNode;
    }
    return this.getRootNode(fromNode.getParent());
  }
  
  private String getMemberInit(final IMethodDeclaration method) {
    return ReverseCpp2Uml.getMemberInit(this.findEnclosingNode(method));
  }
  
  public static String getMemberInit(final IASTNode methodNode) {
    String ret = "";
    if ((methodNode instanceof ICPPASTFunctionDefinition)) {
      final ICPPASTConstructorChainInitializer[] inits = ((ICPPASTFunctionDefinition)methodNode).getMemberInitializers();
      if ((inits != null)) {
        StringConcatenation _builder = new StringConcatenation();
        {
          boolean _hasElements = false;
          for(final ICPPASTConstructorChainInitializer i : inits) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "");
            }
            IASTName _memberInitializerId = i.getMemberInitializerId();
            _builder.append(_memberInitializerId);
            String _rawSignature = i.getInitializer().getRawSignature();
            _builder.append(_rawSignature);
          }
        }
        ret = _builder.toString();
      }
    }
    return ret.toString();
  }
  
  private void updateMethod(final org.eclipse.uml2.uml.Class classifier, final Operation op, final IMethodDeclaration method) {
    try {
      op.setName(method.getElementName());
      StereotypeUtil.unapply(op, Inline.class);
      StereotypeUtil.unapply(op, Friend.class);
      StereotypeUtil.unapply(op, Virtual.class);
      StereotypeUtil.unapply(op, Volatile.class);
      StereotypeUtil.unapply(op, Create.class);
      StereotypeUtil.unapply(op, Destroy.class);
      StereotypeUtil.unapply(op, Const.class);
      op.getOwnedParameters().clear();
      IASTFunctionDeclarator declarator = this.getDeclarator(method);
      List<String> keywords = this.getKeywords(method);
      String body = this.getBody(method);
      op.setVisibility(this.convertVisibility(method.getVisibility()));
      op.setIsStatic(method.isStatic());
      this.reverse_utils.applyStereotype(op, method.isInline(), ReverseUtils.StereotypeType.INLINE, "");
      this.reverse_utils.applyStereotype(op, method.isFriend(), ReverseUtils.StereotypeType.FRIENDLINE, "");
      this.reverse_utils.applyStereotype(op, method.isVirtual(), ReverseUtils.StereotypeType.VIRTUAL, "");
      this.reverse_utils.applyStereotype(op, method.isVolatile(), ReverseUtils.StereotypeType.VOLATILE, "");
      this.reverse_utils.applyStereotype(op, method.isConst(), ReverseUtils.StereotypeType.CONST, "");
      boolean _isVirtual = method.isVirtual();
      if (_isVirtual) {
        this.reverse_utils.applyStereotype(op, method.isVirtual(), ReverseUtils.StereotypeType.VIRTUAL, "");
        boolean _isPureVirtual = method.isPureVirtual();
        if (_isPureVirtual) {
          op.setIsAbstract(true);
        }
      }
      this.reverse_utils.applyStereotype(op, method.isConstructor(), ReverseUtils.StereotypeType.CREATE, "");
      this.reverse_utils.applyStereotype(op, method.isDestructor(), ReverseUtils.StereotypeType.DESTROY, "");
      try {
        String signature = method.getSignature();
        Pattern pattern = Pattern.compile("([\\s]*)(\\.\\.\\.)([\\s]*)(\\))");
        Matcher matcher = pattern.matcher(signature);
        this.reverse_utils.applyStereotype(op, matcher.find(), ReverseUtils.StereotypeType.VARIADIC, "");
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          e.printStackTrace();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      if ((method instanceof IMethodTemplateDeclaration)) {
        for (int i = 0; (i < ((List<String>)Conversions.doWrapArray(((IMethodTemplateDeclaration)method).getTemplateParameterTypes())).size()); i++) {
          this.createOrgetTemplateParameter(op, ((IMethodTemplateDeclaration)method).getTemplateParameterTypes()[i], keywords.get(i));
        }
      }
      if (((!method.isConstructor()) && (!method.isDestructor()))) {
        String _returnType = method.getReturnType();
        boolean _tripleNotEquals = (_returnType != null);
        if (_tripleNotEquals) {
          Type parameterType = this.getParameterTemplateType(op, method, this.getCppTypeName(method.getReturnType()));
          if ((parameterType == null)) {
            parameterType = this.getUMLType(this.getCppTypeName(method.getReturnType()), method.getTranslationUnit(), 
              this.getContextNamespaces(method));
          }
          Parameter ret = op.createOwnedParameter("ret", parameterType);
          ret.setDirection(ParameterDirectionKind.RETURN_LITERAL);
          this.reverse_utils.analyzeDeclaration(declarator, ret.getType(), ret, this.langID);
          if ((parameterType.getName().equals("void") && ret.getAppliedStereotypes().isEmpty())) {
            ret.destroy();
          }
          this.reverse_utils.applyStereotype(ret, method.getReturnType().contains("const "), ReverseUtils.StereotypeType.CONST, "");
          this.reverse_utils.applyStereotype(ret, method.getReturnType().contains("volatile "), ReverseUtils.StereotypeType.VOLATILE, "");
        } else {
          Parameter ret_1 = op.createOwnedParameter("ret", 
            this.getUMLType("void", method.getTranslationUnit(), this.getContextNamespaces(method)));
          ret_1.setDirection(ParameterDirectionKind.RETURN_LITERAL);
          this.reverse_utils.analyzeDeclaration(declarator, ret_1.getType(), ret_1, this.langID);
          boolean _isEmpty = ret_1.getAppliedStereotypes().isEmpty();
          if (_isEmpty) {
            ret_1.destroy();
          }
        }
      }
      boolean noArguments = false;
      int _size = IterableExtensions.size(Iterables.<IASTParameterDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(declarator.getChildren())), IASTParameterDeclaration.class));
      boolean _equals = (_size == 1);
      if (_equals) {
        IASTParameterDeclaration parameterNode = ((IASTParameterDeclaration[])Conversions.unwrapArray((Iterables.<IASTParameterDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(declarator.getChildren())), IASTParameterDeclaration.class)), IASTParameterDeclaration.class))[0];
        IASTDeclSpecifier _declSpecifier = parameterNode.getDeclSpecifier();
        boolean _tripleNotEquals_1 = (_declSpecifier != null);
        if (_tripleNotEquals_1) {
          String type = "";
          IToken tokens = parameterNode.getDeclSpecifier().getSyntax();
          while ((tokens != null)) {
            {
              String _type = type;
              String _string = tokens.toString();
              type = (_type + _string);
              tokens = tokens.getNext();
            }
          }
          boolean _equals_1 = type.trim().equals("void");
          if (_equals_1) {
            noArguments = true;
          }
        }
      }
      if ((!noArguments)) {
        Iterable<IASTParameterDeclaration> _filter = Iterables.<IASTParameterDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(declarator.getChildren())), IASTParameterDeclaration.class);
        for (final IASTParameterDeclaration param : _filter) {
          {
            this.excludeIncludFromTranslationUnit(param, method, op);
            Type parameterType_1 = this.getParameterTemplateType(op, method, this.reverse_utils.getCppTypeName(param.getDeclSpecifier()));
            if ((parameterType_1 == null)) {
              parameterType_1 = this.getUMLType(param.getDeclSpecifier(), method.getTranslationUnit(), this.getContextNamespaces(method));
            }
            final Parameter opParam = op.createOwnedParameter(param.getDeclarator().getName().toString(), parameterType_1);
            this.reverse_utils.applyStereotype(opParam, param.getDeclSpecifier().isConst(), 
              ReverseUtils.StereotypeType.CONST, "");
            this.reverse_utils.analyzeDeclaration(param.getDeclarator(), opParam.getType(), opParam, this.langID);
            this.reverse_utils.applyStereotype(opParam, param.getDeclSpecifier().isVolatile(), 
              ReverseUtils.StereotypeType.VOLATILE, "");
          }
        }
      }
      if ((body != null)) {
        boolean _isConstructor = method.isConstructor();
        if (_isConstructor) {
          String initStr = this.getMemberInit(method);
          boolean _isEmpty_1 = initStr.isEmpty();
          boolean _not = (!_isEmpty_1);
          if (_not) {
            StereotypeUtil.apply(op, ConstInit.class);
            ConstInit _stereotypeApplication = UMLUtil.<ConstInit>getStereotypeApplication(op, ConstInit.class);
            _stereotypeApplication.setInitialisation(initStr);
          }
        }
        OpaqueBehavior ob = null;
        int _size_1 = op.getMethods().size();
        boolean _equals_2 = (_size_1 == 0);
        if (_equals_2) {
          Behavior _createOwnedBehavior = classifier.createOwnedBehavior(op.getName(), UMLPackage.Literals.OPAQUE_BEHAVIOR);
          ob = ((OpaqueBehavior) _createOwnedBehavior);
          ob.setSpecification(op);
          ob.setIsReentrant(false);
        } else {
          Behavior _get = op.getMethods().get(0);
          ob = ((OpaqueBehavior) _get);
          boolean _equals_3 = ob.getName().equals(op.getName());
          boolean _not_1 = (!_equals_3);
          if (_not_1) {
            ob.setName(op.getName());
          }
        }
        int _size_2 = ob.getBodies().size();
        boolean _equals_4 = (_size_2 == 0);
        if (_equals_4) {
          ob.getLanguages().add(this.langID);
          ob.getBodies().add("");
        }
        for (int i = 0; (i < ob.getLanguages().size()); i++) {
          boolean _equals_5 = ob.getLanguages().get(i).equals(this.langID);
          if (_equals_5) {
            int _size_3 = ob.getBodies().size();
            boolean _lessThan = (i < _size_3);
            if (_lessThan) {
              ob.getBodies().set(i, body);
            }
          }
        }
        final IASTNode node = this.findEnclosingNode(method);
        if ((node instanceof IASTFunctionDefinition)) {
          ITranslationUnit _translationUnit = method.getTranslationUnit();
          new DependencyAnalysis(op, ((IASTFunctionDefinition)node), _translationUnit, this).analyzeDependencies();
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Type getParameterTemplateType(final Operation op, final IMethodDeclaration imethod, final String typeName) {
    Type ret = null;
    if ((imethod instanceof IMethodTemplateDeclaration)) {
      final Function1<String, Boolean> _function = new Function1<String, Boolean>() {
        @Override
        public Boolean apply(final String it) {
          return Boolean.valueOf(it.equals(typeName));
        }
      };
      boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<String>filter(((Iterable<String>)Conversions.doWrapArray(((IMethodTemplateDeclaration)imethod).getTemplateParameterTypes())), _function));
      boolean _not = (!_isEmpty);
      if (_not) {
        ret = this.createOrgetTemplateParameter(op, typeName, "class");
      }
    }
    return ret;
  }
  
  private boolean isSimpleAssociation(final Property prop) {
    if (((StereotypeUtil.isApplied(prop, Ptr.class) || StereotypeUtil.isApplied(prop, Ref.class)) && (!StereotypeUtil.isApplied(prop, Array.class)))) {
      return true;
    }
    return false;
  }
  
  private boolean isAggregation(final Property prop) {
    if (((StereotypeUtil.isApplied(prop, Ptr.class) || StereotypeUtil.isApplied(prop, Ref.class)) && StereotypeUtil.isApplied(prop, Array.class))) {
      return true;
    }
    return false;
  }
  
  private boolean isComposition(final Property prop) {
    if (((!StereotypeUtil.isApplied(prop, Ptr.class)) && (!StereotypeUtil.isApplied(prop, Ref.class)))) {
      return true;
    }
    return false;
  }
  
  private void createProperty(final IField field, final Classifier classifier) {
    Property _createProperty = UMLFactory.eINSTANCE.createProperty();
    final Procedure1<Property> _function = new Procedure1<Property>() {
      @Override
      public void apply(final Property it) {
        it.setName(field.getElementName());
      }
    };
    Property prop = ObjectExtensions.<Property>operator_doubleArrow(_createProperty, _function);
    if ((classifier instanceof DataType)) {
      EList<Property> _ownedAttributes = ((DataType) classifier).getOwnedAttributes();
      _ownedAttributes.add(prop);
    } else {
      if ((classifier instanceof org.eclipse.uml2.uml.Class)) {
        EList<Property> _ownedAttributes_1 = ((org.eclipse.uml2.uml.Class) classifier).getOwnedAttributes();
        _ownedAttributes_1.add(prop);
      }
    }
    this.updateProperty(field, prop);
  }
  
  private void updateProperty(final IField field, final Property prop) {
    try {
      this.reverse_utils.unapplyAllStereotypes(prop);
      Type type = null;
      boolean doNotAnalyzeDeclaration = false;
      try {
        IASTNode fieldNode = this.findEnclosingNode(field);
        if ((fieldNode != null)) {
          String rawSignature = fieldNode.getRawSignature().replaceAll("\\n", "").replaceAll("\\r", "").replaceAll(";", "").replaceAll("\\s+", " ").trim();
          Pattern pattern = Pattern.compile("(\\()(\\s*)(\\*)(.*)(\\))(\\s*)(\\()(.*)(\\))");
          Matcher matcher = pattern.matcher(rawSignature);
          boolean _find = matcher.find();
          if (_find) {
            String typeName = rawSignature.replaceFirst(Pattern.quote(field.getElementName()), "typeName").replaceFirst("typedef", "");
            PackageableElement packageable = null;
            Element _owner = prop.getOwner();
            if ((_owner instanceof org.eclipse.uml2.uml.Class)) {
              Element _owner_1 = prop.getOwner();
              packageable = ((org.eclipse.uml2.uml.Class) _owner_1);
            } else {
              packageable = prop.getNearestPackage();
            }
            final String ownerName = packageable.getName();
            PrimitiveType _createPrimitiveType = UMLFactory.eINSTANCE.createPrimitiveType();
            final Procedure1<PrimitiveType> _function = new Procedure1<PrimitiveType>() {
              @Override
              public void apply(final PrimitiveType it) {
                String _elementName = field.getElementName();
                String _plus = ((ownerName + "_") + _elementName);
                String _plus_1 = (_plus + "_funcptr");
                it.setName(_plus_1);
              }
            };
            PrimitiveType primitiveType = ObjectExtensions.<PrimitiveType>operator_doubleArrow(_createPrimitiveType, _function);
            if ((packageable instanceof org.eclipse.uml2.uml.Package)) {
              EList<Type> _ownedTypes = ((org.eclipse.uml2.uml.Package) packageable).getOwnedTypes();
              _ownedTypes.add(primitiveType);
            } else {
              EList<Classifier> _nestedClassifiers = ((org.eclipse.uml2.uml.Class) packageable).getNestedClassifiers();
              _nestedClassifiers.add(primitiveType);
            }
            StereotypeUtil.apply(primitiveType, Typedef.class);
            Typedef _stereotypeApplication = UMLUtil.<Typedef>getStereotypeApplication(primitiveType, Typedef.class);
            _stereotypeApplication.setDefinition(typeName);
            type = primitiveType;
            doNotAnalyzeDeclaration = true;
          }
        }
        if ((type == null)) {
          if (((field.getTypeName().equals("enum") || field.getTypeName().equals("struct")) || field.getTypeName().equals("class"))) {
            String rawSignature_1 = fieldNode.getRawSignature();
            String trimmedRawSignature = rawSignature_1.replaceAll("\\n", "").replaceAll("\\r", "").replaceAll(";", "").replaceAll("\\s+", " ").trim();
            Pattern pattern_1 = Pattern.compile("(\\{)(.*)(\\})");
            Matcher matcher_1 = pattern_1.matcher(trimmedRawSignature);
            boolean _find_1 = matcher_1.find();
            if (_find_1) {
              String[] tokens = rawSignature_1.split("}");
              final String[] _converted_tokens = (String[])tokens;
              int _size = ((List<String>)Conversions.doWrapArray(_converted_tokens)).size();
              boolean _greaterThan = (_size > 0);
              if (_greaterThan) {
                final String[] _converted_tokens_1 = (String[])tokens;
                int _size_1 = ((List<String>)Conversions.doWrapArray(_converted_tokens_1)).size();
                int _minus = (_size_1 - 1);
                String lastToken = tokens[_minus];
                boolean _contains = lastToken.contains(field.getElementName());
                if (_contains) {
                  lastToken = lastToken.replaceFirst(Pattern.quote(field.getElementName()), "typeName");
                }
                String typeName_1 = "";
                final String[] _converted_tokens_2 = (String[])tokens;
                int _size_2 = ((List<String>)Conversions.doWrapArray(_converted_tokens_2)).size();
                int _minus_1 = (_size_2 - 1);
                ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _minus_1, true);
                for (final Integer i : _doubleDotLessThan) {
                  String _typeName = typeName_1;
                  String _get = tokens[(i).intValue()];
                  typeName_1 = (_typeName + _get);
                }
                String _typeName_1 = typeName_1;
                typeName_1 = (_typeName_1 + lastToken);
                PackageableElement packageable_1 = null;
                Element _owner_2 = prop.getOwner();
                if ((_owner_2 instanceof org.eclipse.uml2.uml.Class)) {
                  Element _owner_3 = prop.getOwner();
                  packageable_1 = ((org.eclipse.uml2.uml.Class) _owner_3);
                } else {
                  packageable_1 = prop.getNearestPackage();
                }
                final String ownerName_1 = packageable_1.getName();
                PrimitiveType _createPrimitiveType_1 = UMLFactory.eINSTANCE.createPrimitiveType();
                final Procedure1<PrimitiveType> _function_1 = new Procedure1<PrimitiveType>() {
                  @Override
                  public void apply(final PrimitiveType it) {
                    try {
                      String _elementName = field.getElementName();
                      String _plus = ((ownerName_1 + "_") + _elementName);
                      String _plus_1 = (_plus + "_anon_");
                      String _typeName = field.getTypeName();
                      String _plus_2 = (_plus_1 + _typeName);
                      it.setName(_plus_2);
                    } catch (Throwable _e) {
                      throw Exceptions.sneakyThrow(_e);
                    }
                  }
                };
                PrimitiveType primitiveType_1 = ObjectExtensions.<PrimitiveType>operator_doubleArrow(_createPrimitiveType_1, _function_1);
                if ((packageable_1 instanceof org.eclipse.uml2.uml.Package)) {
                  EList<Type> _ownedTypes_1 = ((org.eclipse.uml2.uml.Package) packageable_1).getOwnedTypes();
                  _ownedTypes_1.add(primitiveType_1);
                } else {
                  EList<Classifier> _nestedClassifiers_1 = ((org.eclipse.uml2.uml.Class) packageable_1).getNestedClassifiers();
                  _nestedClassifiers_1.add(primitiveType_1);
                }
                StereotypeUtil.apply(primitiveType_1, Typedef.class);
                Typedef _stereotypeApplication_1 = UMLUtil.<Typedef>getStereotypeApplication(primitiveType_1, Typedef.class);
                _stereotypeApplication_1.setDefinition(typeName_1);
                type = primitiveType_1;
                doNotAnalyzeDeclaration = true;
              }
            }
          }
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          e.printStackTrace();
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      this.excludeIncludFromTranslationUnit(field, prop);
      if ((type == null)) {
        type = this.getUMLType(field, this.reverse_utils.getContextNamespaces(field));
      }
      prop.setType(type);
      prop.setName(field.getElementName());
      if ((!doNotAnalyzeDeclaration)) {
        IASTNode node = this.getSelector(field.getTranslationUnit()).findEnclosingNode(field.getSourceRange().getStartPos(), 
          field.getSourceRange().getLength());
        if ((node instanceof IASTSimpleDeclaration)) {
          this.reverse_utils.analyzeDeclaration(((List<IASTDeclarator>)Conversions.doWrapArray(((IASTSimpleDeclaration)node).getDeclarators())), prop.getType(), prop, this.langID);
        }
      }
      prop.setIsStatic(field.isStatic());
      prop.setVisibility(this.convertVisibility(field.getVisibility()));
      this.reverse_utils.applyStereotype(prop, field.isConst(), ReverseUtils.StereotypeType.CONST, "");
      this.reverse_utils.applyStereotype(prop, field.isVolatile(), ReverseUtils.StereotypeType.VOLATILE, "");
      this.reverse_utils.applyStereotype(prop, field.isMutable(), ReverseUtils.StereotypeType.MUTABLE, "");
      EObject _eContainer = prop.eContainer();
      Association asso = TypeOperationsEnhanced.createAssociationFromProperty(prop, true, AggregationKind.NONE_LITERAL, false, 
        AggregationKind.NONE_LITERAL, StringExtensions.toFirstLower(((Classifier) _eContainer).getName()), 1, 1);
      String _name = prop.getName();
      String _plus = ("A_" + _name);
      String _plus_1 = (_plus + "_");
      EObject _eContainer_1 = prop.eContainer();
      String _firstLower = StringExtensions.toFirstLower(((Classifier) _eContainer_1).getName());
      String _plus_2 = (_plus_1 + _firstLower);
      asso.setName(_plus_2);
      boolean _isSimpleAssociation = this.isSimpleAssociation(prop);
      if (_isSimpleAssociation) {
        prop.setAggregation(AggregationKind.NONE_LITERAL);
      } else {
        boolean _isAggregation = this.isAggregation(prop);
        if (_isAggregation) {
          prop.setAggregation(AggregationKind.SHARED_LITERAL);
        } else {
          boolean _isComposition = this.isComposition(prop);
          if (_isComposition) {
            prop.setAggregation(AggregationKind.COMPOSITE_LITERAL);
          }
        }
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private VisibilityKind convertVisibility(final ASTAccessVisibility visibility) {
    boolean _equals = Objects.equal(visibility, ASTAccessVisibility.PRIVATE);
    if (_equals) {
      return VisibilityKind.PRIVATE_LITERAL;
    }
    boolean _equals_1 = Objects.equal(visibility, ASTAccessVisibility.PUBLIC);
    if (_equals_1) {
      return VisibilityKind.PUBLIC_LITERAL;
    }
    boolean _equals_2 = Objects.equal(visibility, ASTAccessVisibility.PROTECTED);
    if (_equals_2) {
      return VisibilityKind.PROTECTED_LITERAL;
    }
    return VisibilityKind.PUBLIC_LITERAL;
  }
  
  public List<String> getContextNamespaces(final ICElement element) {
    return this.reverse_utils.getContextNamespaces(element);
  }
  
  private Type getUMLType(final IVariableDeclaration varDecl, final List<String> contextNamespaces) {
    try {
      Type ret = null;
      ret = this.getUMLType(this.getCppTypeName(varDecl.getTypeName()), varDecl.getTranslationUnit(), contextNamespaces);
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private String getCppTypeName(final String name) {
    String trimName = name.trim();
    trimName = trimName.replace("*", "");
    trimName = trimName.replace("&", "");
    trimName = trimName.replace("[", "");
    trimName = trimName.replace("]", "");
    trimName = trimName.replace("const ", "");
    trimName = trimName.replace(" const", "");
    trimName = trimName.replace("volatile", "");
    trimName = trimName.replace(" volatile", "");
    trimName = trimName.trim();
    return trimName;
  }
  
  private Type getUMLType(final IASTDeclSpecifier declarator, final ITranslationUnit itu, final List<String> contextNamespaces) {
    Type ret = null;
    ret = this.getUMLType(this.reverse_utils.getCppTypeName(declarator), itu, contextNamespaces);
    return ret;
  }
  
  private List<Map.Entry<ICElement, EObject>> getElementsSameName(final String name) {
    final Function1<Map.Entry<ICElement, EObject>, Boolean> _function = new Function1<Map.Entry<ICElement, EObject>, Boolean>() {
      @Override
      public Boolean apply(final Map.Entry<ICElement, EObject> it) {
        return Boolean.valueOf(it.getKey().getElementName().equals(name));
      }
    };
    return IterableExtensions.<Map.Entry<ICElement, EObject>>toList(IterableExtensions.<Map.Entry<ICElement, EObject>>filter(this.map.entrySet(), _function));
  }
  
  private ICElement lookTypeInContainer(final ICContainer container, final String typeName) {
    try {
      ICElement ret = null;
      ICElement[] _children = container.getChildren();
      for (final ICElement child : _children) {
        {
          if ((child instanceof ITranslationUnit)) {
            List<ICElement> nesteds = this.reverse_utils.getAllIStructures(((IParent)child), false, true, this.m_project);
            final Function1<ICElement, Boolean> _function = new Function1<ICElement, Boolean>() {
              @Override
              public Boolean apply(final ICElement it) {
                return Boolean.valueOf((((it instanceof IStructure) || (it instanceof IEnumeration)) || (it instanceof ITypeDef)));
              }
            };
            final Function1<ICElement, Boolean> _function_1 = new Function1<ICElement, Boolean>() {
              @Override
              public Boolean apply(final ICElement it) {
                return Boolean.valueOf(it.getElementName().equals(typeName));
              }
            };
            ret = IterableExtensions.<ICElement>head(IterableExtensions.<ICElement>filter(IterableExtensions.<ICElement>filter(nesteds, _function), _function_1));
          } else {
            if ((child instanceof ICContainer)) {
              ret = this.lookTypeInContainer(((ICContainer)child), typeName);
            }
          }
          if ((ret != null)) {
            return ret;
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Type lookForATypeByName(final String typeName) {
    try {
      Type ret = null;
      ICElement cType = null;
      for (final ICContainer container : this.containers) {
        if ((cType == null)) {
          cType = this.lookTypeInContainer(container, typeName);
        }
      }
      if ((cType != null)) {
        ITranslationUnit itu = this.reverse_utils.getTranslationUnitFromElement(cType);
        if ((itu != null)) {
          ret = this.getClassifier(this.getCorrespondingModel(itu), cType, cType.getElementName(), itu);
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  public Type getUMLType(final String typeName, final ITranslationUnit itu, final List<String> contextNamespaces) {
    try {
      Type ret = null;
      boolean _isPrimitiveCppType = RoundtripCppUtils.isPrimitiveCppType(typeName);
      if (_isPrimitiveCppType) {
        return RoundtripCppUtils.getPrimitiveType(typeName, this.getCorrespondingModel(itu));
      }
      String[] token = typeName.split("::");
      int _length = token.length;
      int _minus = (_length - 1);
      final String simpleTypeName = (token[_minus]).trim();
      ArrayList<String> iUsings = new ArrayList<String>();
      iUsings.addAll(contextNamespaces);
      final Function1<IUsing, String> _function = new Function1<IUsing, String>() {
        @Override
        public String apply(final IUsing it) {
          return it.getElementName();
        }
      };
      Iterables.<String>addAll(iUsings, IterableExtensions.<IUsing, String>map(Iterables.<IUsing>filter(((Iterable<?>)Conversions.doWrapArray(itu.getChildren())), IUsing.class), _function));
      final List<Map.Entry<ICElement, EObject>> sameNames = this.getElementsSameName(simpleTypeName);
      if (((sameNames.size() == 1) && (IterableExtensions.<Map.Entry<ICElement, EObject>>head(sameNames).getValue() instanceof Type))) {
        EObject _value = IterableExtensions.<Map.Entry<ICElement, EObject>>head(sameNames).getValue();
        return ((Type) _value);
      }
      List<ICElement> nestedStructures = this.reverse_utils.getAllIStructures(itu, false, true, this.m_project);
      int _size = sameNames.size();
      boolean _greaterThan = (_size > 1);
      if (_greaterThan) {
        for (final Map.Entry<ICElement, EObject> sameName : sameNames) {
          {
            final Function1<ICElement, Boolean> _function_1 = new Function1<ICElement, Boolean>() {
              @Override
              public Boolean apply(final ICElement it) {
                return Boolean.valueOf(it.getElementName().equals(sameName.getKey().getElementName()));
              }
            };
            final Iterable<ICElement> withinITU = IterableExtensions.<ICElement>filter(nestedStructures, _function_1);
            int _size_1 = IterableExtensions.size(withinITU);
            boolean _greaterThan_1 = (_size_1 > 0);
            if (_greaterThan_1) {
              EObject _value_1 = sameName.getValue();
              return ((Type) _value_1);
            } else {
              boolean _isSatisfyNamespace = this.reverse_utils.isSatisfyNamespace(iUsings, sameName.getKey());
              if (_isSatisfyNamespace) {
                EObject _value_2 = sameName.getValue();
                return ((Type) _value_2);
              }
            }
          }
        }
      }
      final Function1<ICElement, Boolean> _function_1 = new Function1<ICElement, Boolean>() {
        @Override
        public Boolean apply(final ICElement it) {
          return Boolean.valueOf(it.getElementName().trim().equals(simpleTypeName.trim()));
        }
      };
      ICElement sameNameType = IterableExtensions.<ICElement>head(IterableExtensions.<ICElement>filter(nestedStructures, _function_1));
      if ((sameNameType == null)) {
        Iterable<IStructureTemplate> structureTemplates = Iterables.<IStructureTemplate>filter(nestedStructures, IStructureTemplate.class);
        final Function1<IStructureTemplate, Boolean> _function_2 = new Function1<IStructureTemplate, Boolean>() {
          @Override
          public Boolean apply(final IStructureTemplate it) {
            final Function1<String, Boolean> _function = new Function1<String, Boolean>() {
              @Override
              public Boolean apply(final String it) {
                return Boolean.valueOf(it.equals(simpleTypeName));
              }
            };
            int _size = IterableExtensions.size(IterableExtensions.<String>filter(((Iterable<String>)Conversions.doWrapArray(it.getTemplateParameterTypes())), _function));
            return Boolean.valueOf((_size > 0));
          }
        };
        final IStructureTemplate match = IterableExtensions.<IStructureTemplate>head(IterableExtensions.<IStructureTemplate>filter(structureTemplates, _function_2));
        if ((match != null)) {
          ICElement _parent = match.getParent();
          final Function1<Type, Boolean> _function_3 = new Function1<Type, Boolean>() {
            @Override
            public Boolean apply(final Type it) {
              return Boolean.valueOf(it.getName().equals(match.getElementName()));
            }
          };
          Type templatClassifier = IterableExtensions.<Type>head(IterableExtensions.<Type>filter(this.getOrCreateClassifier(this.getCorrespondingModel(itu), ((IParent) _parent)), _function_3));
          ret = this.createOrgetTemplateParameter(((Classifier) templatClassifier), simpleTypeName, "class");
        }
      } else {
        if (((sameNameType instanceof IStructureDeclaration) && (!(sameNameType instanceof IStructure)))) {
          ret = this.lookForATypeByName(sameNameType.getElementName());
        } else {
          ret = this.getClassifier(this.getCorrespondingModel(itu), sameNameType, sameNameType.getElementName(), itu);
        }
      }
      org.eclipse.uml2.uml.Package externalPackage = RoundtripCppUtils.getOrcreateExternalPackage(this.getCorrespondingModel(itu), false);
      if ((((externalPackage != null) && (externalPackage.getOwnedMember(simpleTypeName.trim()) != null)) && 
        (externalPackage.getOwnedMember(simpleTypeName.trim()) instanceof Type))) {
        NamedElement _ownedMember = externalPackage.getOwnedMember(simpleTypeName.trim());
        ret = ((Type) _ownedMember);
      }
      if ((ret == null)) {
        final String[] _converted_token = (String[])token;
        int _size_1 = ((List<String>)Conversions.doWrapArray(_converted_token)).size();
        boolean _greaterThan_1 = (_size_1 > 1);
        if (_greaterThan_1) {
          for (int i = 0; (i < (((List<String>)Conversions.doWrapArray(token)).size() - 1)); i++) {
            {
              iUsings.add((token[i]).trim());
              if ((i > 0)) {
                final String[] _converted_token_1 = (String[])token;
                String n = IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(_converted_token_1))).trim();
                for (int j = 1; (j <= i); j++) {
                  String _trim = (token[j]).trim();
                  String _plus = ((n + "::") + _trim);
                  n = _plus;
                }
                iUsings.add(n);
              }
            }
          }
        }
        final List<IInclude> includes = new UniqueEList<IInclude>();
        final UniqueEList<ICElement> istructures = new UniqueEList<ICElement>();
        ICElement _parent_1 = itu.getParent();
        if ((_parent_1 instanceof ICContainer)) {
          ICElement _parent_2 = itu.getParent();
          ICContainer container = ((ICContainer) _parent_2);
          final Consumer<ITranslationUnit> _function_4 = new Consumer<ITranslationUnit>() {
            @Override
            public void accept(final ITranslationUnit it) {
              try {
                final Function1<ICElement, Boolean> _function = new Function1<ICElement, Boolean>() {
                  @Override
                  public Boolean apply(final ICElement it) {
                    return Boolean.valueOf(it.getElementName().trim().equals(simpleTypeName.trim()));
                  }
                };
                Iterables.<ICElement>addAll(istructures, IterableExtensions.<ICElement>filter(ReverseCpp2Uml.this.reverse_utils.getAllIStructures(it, false, true, ReverseCpp2Uml.this.m_project), _function));
                Iterable<IInclude> includesOfItu = Iterables.<IInclude>filter(((Iterable<?>)Conversions.doWrapArray(it.getChildren())), IInclude.class);
                final Consumer<IInclude> _function_1 = new Consumer<IInclude>() {
                  @Override
                  public void accept(final IInclude it) {
                    final String nameOfIncludeItu = it.getElementName();
                    final Function1<IInclude, Boolean> _function = new Function1<IInclude, Boolean>() {
                      @Override
                      public Boolean apply(final IInclude it) {
                        String _elementName = it.getElementName();
                        return Boolean.valueOf(Objects.equal(_elementName, nameOfIncludeItu));
                      }
                    };
                    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<IInclude>filter(includes, _function));
                    if (_isEmpty) {
                      includes.add(it);
                    }
                  }
                };
                includesOfItu.forEach(_function_1);
              } catch (Throwable _e) {
                throw Exceptions.sneakyThrow(_e);
              }
            }
          };
          Iterables.<ITranslationUnit>filter(((Iterable<?>)Conversions.doWrapArray(container.getChildren())), ITranslationUnit.class).forEach(_function_4);
          for (final ICElement structure : istructures) {
            if ((ret == null)) {
              boolean _isSatisfyNamespace = this.reverse_utils.isSatisfyNamespace(iUsings, structure);
              if (_isSatisfyNamespace) {
                List<Type> tmp = this.getOrCreateClassifier(this.getCorrespondingModel(this.reverse_utils.getTranslationUnitFromElement(structure)), 
                  this.reverse_utils.getTranslationUnitFromElement(structure));
                final Function1<Type, Boolean> _function_5 = new Function1<Type, Boolean>() {
                  @Override
                  public Boolean apply(final Type it) {
                    return Boolean.valueOf(((it != null) && it.getName().trim().equals(simpleTypeName)));
                  }
                };
                Iterable<Type> sames = IterableExtensions.<Type>filter(tmp, _function_5);
                int _size_2 = IterableExtensions.size(sames);
                boolean _equals = (_size_2 == 1);
                if (_equals) {
                  ret = IterableExtensions.<Type>head(sames);
                } else {
                  boolean _isSatisfyNamespace_1 = this.reverse_utils.isSatisfyNamespace(iUsings, structure);
                  if (_isSatisfyNamespace_1) {
                  }
                }
              }
            }
          }
        }
        Iterables.<IInclude>addAll(includes, Iterables.<IInclude>filter(((Iterable<?>)Conversions.doWrapArray(itu.getChildren())), IInclude.class));
        List<IInclude> includes_level2 = new UniqueEList<IInclude>();
        while (((ret == null) && (!includes.isEmpty()))) {
          {
            for (final IInclude include : includes) {
              if ((ret == null)) {
                boolean _contains = include.getFullFileName().contains(this.m_project.getElementName());
                if (_contains) {
                  ret = this.lookupTypeInInclude(simpleTypeName, include, iUsings);
                }
              } else {
                ITranslationUnit translationUnit = this.reverse_utils.getTranslationUnitFromInclude(include, this.m_project);
                if ((translationUnit != null)) {
                  Iterables.<IInclude>addAll(includes_level2, Iterables.<IInclude>filter(((Iterable<?>)Conversions.doWrapArray(translationUnit.getChildren())), IInclude.class));
                }
              }
            }
            if ((ret == null)) {
              includes.clear();
              includes.addAll(includes_level2);
              includes_level2.clear();
            }
          }
        }
      }
      if ((ret == null)) {
        ReverseCpp2Uml.LOGGER.log(Level.WARNING, (typeName + " is not found, it will be created dynamically"));
        externalPackage = RoundtripCppUtils.getOrcreateExternalPackage(this.getCorrespondingModel(itu), true);
        ret = externalPackage.getOwnedType(typeName.trim());
        if ((ret == null)) {
          ret = externalPackage.createOwnedType(typeName.trim(), UMLPackage.Literals.DATA_TYPE);
          StereotypeUtil.apply(ret, External.class);
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void excludeIncludFromTranslationUnit(final IField field, final Property property) {
    try {
      if ((field != null)) {
        final String type = field.getTypeName();
        final List<String> context = this.reverse_utils.getContextNamespaces(field);
        final ITranslationUnit translationUnit = field.getTranslationUnit();
        Classifier structure = null;
        Element owner = property.getOwner();
        while (((owner != null) && (!(owner instanceof Classifier)))) {
          owner = owner.getOwner();
        }
        if ((owner instanceof Classifier)) {
          structure = ((Classifier) owner);
        }
        this.excludeIncludFromTranslationUnit(type, structure, translationUnit, context);
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void excludeIncludFromTranslationUnit(final IASTParameterDeclaration param, final IMethodDeclaration method, final Operation operation) {
    if (((param != null) && (method != null))) {
      final ITranslationUnit translationUnit = method.getTranslationUnit();
      final String type = this.reverse_utils.getCppTypeName(param.getDeclSpecifier());
      final List<String> context = this.reverse_utils.getContextNamespaces(method);
      Classifier structure = null;
      Element owner = operation.getOwner();
      while (((owner != null) && (!(owner instanceof Classifier)))) {
        owner = owner.getOwner();
      }
      if ((owner instanceof Classifier)) {
        structure = ((Classifier) owner);
      }
      this.excludeIncludFromTranslationUnit(type, structure, translationUnit, context);
    }
  }
  
  private void excludeIncludFromTranslationUnit(final String excludedType, final Element fromStructure, final ITranslationUnit translationUnit, final List<String> context) {
    if (((((excludedType != null) && (fromStructure != null)) && (translationUnit != null)) && (context != null))) {
      List<IInclude> includes = this.getIncludesFromType(excludedType, ((ITranslationUnit) translationUnit), context);
      if (((includes != null) && (!includes.isEmpty()))) {
        final String key = this.getExcludedIncludesMapKey(translationUnit, fromStructure);
        List<IInclude> includesInTranslationUnit = this.excludedIncludesMap.get(key);
        if ((includesInTranslationUnit == null)) {
          UniqueEList<IInclude> _uniqueEList = new UniqueEList<IInclude>();
          includesInTranslationUnit = _uniqueEList;
          this.excludedIncludesMap.put(key, includesInTranslationUnit);
        }
        includesInTranslationUnit.addAll(includes);
      }
    }
  }
  
  private String getExcludedIncludesMapKey(final ITranslationUnit translationUnit, final Element umlElement) {
    int _hashCode = translationUnit.hashCode();
    String _plus = ("" + Integer.valueOf(_hashCode));
    int _hashCode_1 = umlElement.hashCode();
    return (_plus + Integer.valueOf(_hashCode_1));
  }
  
  private List<IInclude> getIncludesFromType(final String typeName, final ITranslationUnit itu, final List<String> iUsings) {
    try {
      final List<IInclude> includes = new UniqueEList<IInclude>();
      if ((((typeName == null) || (itu == null)) || (iUsings == null))) {
        return includes;
      }
      boolean _contains = ArrayExtensions.contains(BatchReverseFunctionBody.ansiTypes, typeName);
      if (_contains) {
        return includes;
      }
      String[] token = typeName.split("::");
      int _length = token.length;
      int _minus = (_length - 1);
      final String simpleTypeName = (token[_minus]).trim();
      Iterable<IInclude> _filter = Iterables.<IInclude>filter(((Iterable<?>)Conversions.doWrapArray(itu.getChildren())), IInclude.class);
      for (final IInclude include : _filter) {
        boolean _contains_1 = include.getFullFileName().contains(this.m_project.getElementName());
        if (_contains_1) {
          includes.add(include);
        }
      }
      return includes;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private Type lookupTypeInInclude(final String simpleTypeName, final IInclude include, final List<String> iUsings) {
    try {
      boolean _contains = include.getFullFileName().contains(this.m_project.getElementName());
      boolean _not = (!_contains);
      if (_not) {
        return null;
      }
      Type ret = null;
      final UniqueEList<ICElement> istructures = new UniqueEList<ICElement>();
      if ((ret == null)) {
        istructures.clear();
        ITranslationUnit unit = this.reverse_utils.getTranslationUnitFromInclude(include, this.m_project);
        if ((unit == null)) {
          return null;
        }
        final Function1<ICElement, Boolean> _function = new Function1<ICElement, Boolean>() {
          @Override
          public Boolean apply(final ICElement it) {
            return Boolean.valueOf(it.getElementName().trim().equals(simpleTypeName.trim()));
          }
        };
        Iterables.<ICElement>addAll(istructures, IterableExtensions.<ICElement>filter(this.reverse_utils.getAllIStructures(unit, false, true, this.m_project), _function));
        for (final ICElement structure : istructures) {
          if ((ret == null)) {
            List<Type> tmp = this.getOrCreateClassifier(this.getCorrespondingModel(this.reverse_utils.getTranslationUnitFromElement(structure)), 
              this.reverse_utils.getTranslationUnitFromElement(structure));
            final Function1<Type, Boolean> _function_1 = new Function1<Type, Boolean>() {
              @Override
              public Boolean apply(final Type it) {
                return Boolean.valueOf(((it != null) && it.getName().trim().equals(simpleTypeName)));
              }
            };
            Iterable<Type> sameNames = IterableExtensions.<Type>filter(tmp, _function_1);
            int _size = IterableExtensions.size(sameNames);
            boolean _equals = (_size == 1);
            if (_equals) {
              ret = IterableExtensions.<Type>head(sameNames);
            } else {
              boolean _isSatisfyNamespace = this.reverse_utils.isSatisfyNamespace(iUsings, structure);
              if (_isSatisfyNamespace) {
              }
            }
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private UniqueEList<Classifier> getTypeByQualifiedName(final ICElement element, final ITranslationUnit translationUnit, final List<String> contextNames) {
    UniqueEList<Classifier> results = new UniqueEList<Classifier>();
    if (((element instanceof IVariableDeclaration) || (element instanceof IMethod))) {
      boolean _contains = element.getElementName().contains("::");
      if (_contains) {
        String[] classifierNameTokens = element.getElementName().split("::");
        String classifierName = "";
        int _length = classifierNameTokens.length;
        boolean _greaterThan = (_length > 1);
        if (_greaterThan) {
          int _length_1 = classifierNameTokens.length;
          int _minus = (_length_1 - 2);
          classifierName = classifierNameTokens[_minus];
        } else {
          classifierName = classifierNameTokens[0];
        }
        boolean isFound = false;
        int i = 0;
        while (((i < results.size()) && (!isFound))) {
          {
            Classifier classifier = results.get(i);
            if ((classifier != null)) {
              boolean _equals = classifier.getName().equals(classifierName);
              if (_equals) {
                isFound = true;
              }
            }
            i++;
          }
        }
        if ((!isFound)) {
          Type typeRet = this.getUMLType(classifierName, translationUnit, contextNames);
          if ((typeRet instanceof Classifier)) {
            results.add(((Classifier) typeRet));
          }
        }
      }
    }
    return results;
  }
}
