/*
 * Copyright (c) 2010-2013 Eike Stepper (Loehne, Germany) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Eike Stepper - initial API and implementation
 */
package org.eclipse.emf.cdo.spi.common.revision;

import org.eclipse.emf.cdo.common.branch.CDOBranch;
import org.eclipse.emf.cdo.common.branch.CDOBranchVersion;
import org.eclipse.emf.cdo.common.id.CDOID;
import org.eclipse.emf.cdo.common.revision.CDORevisionCache;
import org.eclipse.emf.cdo.common.revision.CDORevisionManager;
import org.eclipse.emf.cdo.spi.common.branch.InternalCDOBranch;

import org.eclipse.emf.ecore.EClass;

/**
 * A synthetic revision that represents the initial period of an object in a {@link CDOBranch branch} when the object is
 * still associated with a revision from one of the baseline branches. It always has {@link #getVersion() version}
 * {@link CDOBranchVersion#UNSPECIFIED_VERSION zero} and can only appear in branches below the
 * {@link CDOBranch#isMainBranch() main} branch.
 * <p>
 * Synthetic revisions are used for two slightly different purposes:
 * <ol>
 * <li>For {@link CDORevisionCache cache} optimization.
 * <li>As a persistent "detach marker" indicating that the first modification of an object in a branch is its deletion.
 * </ol>
 * <p>
 * Instances of this marker revision are not supposed to be exposed outside of a revision {@link CDORevisionManager
 * manager}. They are mainly used in the communication between a revision manager and its associated revision
 * {@link InternalCDORevisionManager.RevisionLoader loader}.
 *
 * @author Eike Stepper
 * @since 3.0
 */
public abstract class SyntheticCDORevision extends StubCDORevision
{
  private CDOID id;

  private InternalCDOBranch branch;

  public SyntheticCDORevision(EClass eClass, CDOID id, CDOBranch branch)
  {
    super(eClass);
    this.id = id;
    this.branch = (InternalCDOBranch)branch;
  }

  @Override
  public CDOID getID()
  {
    return id;
  }

  @Override
  public InternalCDOBranch getBranch()
  {
    return branch;
  }

  @Override
  public void freeze()
  {
    // Do nothing
  }
}
