/**
 * Copyright (c) 2021, 2025 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
/*
 * generated by Xtext
 */
package org.eclipse.lsat.activity.teditor.validation

import activity.Action
import activity.Activity
import activity.ActivityPackage
import activity.ActivitySet
import activity.Claim
import activity.Event
import activity.EventAction
import activity.Move
import activity.PeripheralAction
import activity.RaiseEvent
import activity.Release
import activity.RequireEvent
import activity.ResourceAction
import activity.SchedulingType
import activity.SimpleAction
import activity.SyncBar
import activity.util.ActivityUtil
import com.google.common.collect.Sets
import java.util.Collections
import java.util.HashMap
import java.util.List
import java.util.Set
import machine.Import
import machine.MachinePackage
import machine.Peripheral
import machine.Resource
import machine.ResourceItem
import machine.ResourceType
import machine.SymbolicPosition
import machine.impl.MachineQueries
import machine.util.ResourcePeripheralKey
import org.eclipse.emf.common.util.URI
import org.eclipse.lsat.common.graph.directed.editable.EdgPackage
import org.eclipse.lsat.common.graph.directed.editable.EdgQueries
import org.eclipse.lsat.common.graph.directed.editable.Node
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.validation.Check

import static activity.impl.ActivityQueries.*

import static extension machine.util.ResourcePeripheralKey.createKey
import static extension org.eclipse.lsat.common.graph.directed.editable.EdgQueries.*
import static extension org.eclipse.lsat.common.xtend.Queries.*
import machine.Machine

/**
 * Custom validation rules. 
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
class ActivityValidator extends AbstractActivityValidator {

    public static val ALAP_SCENARIO_1 = 'invalidAlapScenario1'
    public static val ALAP_SCENARIO_2 = 'invalidAlapScenario2'
    public static val ALAP_SCENARIO_3 = 'invalidAlapScenario3'
    public static val ALAP_SCENARIO_4 = 'invalidAlapScenario4'
    public static val INVALID_CLAIM = 'invalidClaim'
    public static val INVALID_PASSIVE_CLAIM = 'invalidPassiveClaim'
    public static val SUGGEST_PASSIVE_CLAIM = 'suggestPassiveClaim'
    public static val INVALID_RELEASE = 'invalidRelease'
    public static val PASSIVE_CLAIM_INCOMING_EDGES = 'passiveClaimIncomingEdges'
    public static val PASSIVE_RELEASE_OUTGOING_EDGES = 'passiveReleaseOutgoingEdges'
    public static val INVALID_SYNC_BAR = 'invalidSyncBar'
    public static val REMOVE_SYNC_BAR = 'removeSyncBar'
    public static val NO_INCOMING_EDGES = 'noIncomingEdges'
    public static val NO_OUTGOING_EDGES = 'noOutgoingEdges'
    public static val DEAD_ACTION = 'deadAction'
    public static val DEAD_EVENT = 'deadEvent'
    public static val MORE_THAN_ONE_INCOMING_EDGE = 'moreThanOneIncomingEdge'
    public static val MORE_THAN_ONE_OUTGOING_EDGE = 'moreThanOneOutgoingEdge'
    public static val ONE_CLAIM_PER_RESOURCE_PER_ACT = 'oneClaimPerResourcePerActivity'
    public static val ONE_EVENT_WITH_SAME_NAME_PER_ACT = 'oneEventWithSameNamePerActivity'
    public static val ONE_RELEASE_PER_RESOURCE_PER_ACT = 'oneReleasePerResourcePerActivity'
    public static val NO_CLAIM_DEFINED_FOR_RESOURCE = 'noClaimDefinedForResource'
    public static val RESOURCE_NOT_PROPERLY_CLAIMED = 'resourceNotProperlyClaimed'
    public static val NO_RELEASE_DEFINED_FOR_RESOURCE = 'noReleaseDefinedForResource'
    public static val RESOURCE_NOT_PROPERLY_RELEASED = 'resourceNotProperlyReleased'
    public static val CYCLE_DETECTED = 'cycleDetected'
    public static val ACTION_IN_PARALLEL_FOR_SAME_PERIPHERAL = 'actionInParallelForSamePeripheral'
    public static val NO_LOC_PREREQUISTIE_FOR_PERIPHERAL_FOR_MOVE = 'noLocPrereqForPeripheralForMove'
    public static val DUPLICATE_ACTIVITY_NAME = 'duplicateActivityName'
    public static val MORE_THAN_ONE_LOCATION_PREREQ_FOR_PERIPHERAL = 'moreThanOneLocationPrereqForPeripheral'
    public static val SAME_SOURCE_TARGET_FOR_PERIPHERAL = 'sameSourceTargetForPeripheral'
    public static val NO_PATH_FOUND_FOR_PROFILE = 'noPathFoundForProfile'
    public static val SETTLING_ALREADY_DEFINED_FOR_PATH = 'settlingAlreadyDefinedForPath'
    public static val SETTLING_ALREADY_DEFINED_FOR_MOVE = 'settlingAlreadyDefinedForMove'
    public static val NO_CONCAT_FOUND_FOR_MOVE = 'noConcatFoundForMove'
    public static val INVALID_IMPORT = 'invalidImport'
    public static val INVALID_RESOURCE = 'invalidResource'
    public static val DUPLICATE_EVENT_RESOURCE_NAME = 'duplicateEventResourceName'

    @Check
    def checkUniqueEventAndResourceNames(Event event) {
        val imports = (event.eContainer as ActivitySet).imports
        val conflictingImports = imports.filter[load.filter(Machine).flatMap[resources].exists[name == event.name]]
        for (conflictingImport : conflictingImports) {
            error('''Import «conflictingImport.importURI» already defines a resource with name «event.name». Event and resource names should be unique.''',
                MachinePackage.Literals.IRESOURCE__NAME, DUPLICATE_EVENT_RESOURCE_NAME)
        }
    }

    @Check
    def checkImportIsValid(Import imp) {
        try {
            val isImportUriValid = EcoreUtil2.isValidUri(imp, URI.createURI(imp.importURI))
            if (!isImportUriValid) {
                error('''The import «imp.importURI» cannot be resolved. Make sure that the name is spelled correctly.''',
                    imp, MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
            }
            val isUnderstood = imp.importURI.matches(".*\\.(machine|activity)")
            if (!isUnderstood) {
                error('''Importing «imp.importURI» is not allowed. Only 'machine' files are allowed''', imp,
                    MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
            }
        } catch (IllegalArgumentException e) {
            error('''The import «imp.importURI» is not a valid URI.''', imp,
                MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
        }
    }

    /** If ASAP required, guarantees that it will be ASAP */
    def boolean concatenatesASAP(Action action) {
        val willDoASAP = switch (action) {
            PeripheralAction: action.schedulingType == SchedulingType::ASAP
            Release: true
            default: false
        }
        return willDoASAP && action.nearestPredecessors(Action).size <= 1
    }

    /** If ALAP required, guarantees that it will be ALAP */
    def boolean concatenatesALAP(Action action) {
        val willDoALAP = switch (action) {
            PeripheralAction: action.schedulingType == SchedulingType::ALAP
            Claim: true
            default: false
        }
        return willDoALAP && action.nearestSuccessors(Action).size <= 1
    }

    @Check
    def checkActionIsALAPWithSuccessorIsASAP(PeripheralAction action) {
        if (action.schedulingType == SchedulingType::ALAP && action.nearestSuccessors(Action).exists[concatenatesASAP]) {
            warning('''The ALAP keyword does not have any affect for action «action.name».''',
                ActivityPackage.Literals.PERIPHERAL_ACTION__SCHEDULING_TYPE, ALAP_SCENARIO_1)
        }
    }


    @Check
    def checkConcatenatedMove(Move move) {
        val successorMove = move.successorMove
        val isContinuing = !move.isStopAtTarget

        if (isContinuing) {
            val successorPeripheralActions = move.nearestSuccessors(PeripheralAction).toSet
            if (successorMove === null) {
                error('''«move.name» should be concatenated with another move. ''', move,
                    EdgPackage.Literals.NODE__NAME, NO_CONCAT_FOUND_FOR_MOVE)
                return
            } else if (!successorPeripheralActions.contains(successorMove)) {
                error('''Failed to concatenate «move.name» and «successorMove.name», only sync bars, claims, releases or events are allowed in between.''',
                    move, EdgPackage.Literals.NODE__NAME, NO_CONCAT_FOUND_FOR_MOVE)
                return
            }
        }

        val predecessorMove = move.predecessorMove
        val isContinuated = predecessorMove !== null && !predecessorMove.isStopAtTarget

        val actionsUntilPredecessorMove = Collections::singleton(predecessorMove as Action)
                .closure(true)[nearestSuccessors(Action)].until[it == move]
        val actionsUntilSuccessorMove = Collections::singleton(successorMove as Action)
                .closure(true)[nearestPredecessors(Action)].until[it == move]

        val requiresASAP = isContinuated && actionsUntilPredecessorMove.exists[!concatenatesALAP]
        val requiresALAP = isContinuing && actionsUntilSuccessorMove.exists[!concatenatesASAP]
                // We suppress the ALAP requirement if it would lead to ALAP_SCENARIO_1 warning.
                // In this case the actual error will be reported on the successor move.
                && !move.nearestSuccessors(Action).exists[concatenatesASAP]

        if (requiresASAP && requiresALAP) {
            // Warn about potential interrupt of concatenated move
            val predecessors = EdgQueries.allPredecessors(successorMove).toSet
            predecessors.remove(move)
            predecessors.removeAll(EdgQueries.allPredecessors(move))
            for (pred : predecessors.reject(Release).reject(SyncBar)) {
                val activity = EcoreUtil2.getContainerOfType(pred, Activity)
                val predIndex = activity.nodes.indexOf(pred);
                warning('''«pred.name» might interrupt concatenated move [«move.name»->«successorMove.name»]''',
                    activity, EdgPackage.Literals.EDITABLE_DIRECTED_GRAPH__NODES, predIndex, ALAP_SCENARIO_4)
            }
        } else if (requiresASAP && move.schedulingType == SchedulingType::ALAP) {
            error('''Move «move.name» must be set to ASAP to guarantee concatenation with predecessor move «predecessorMove.name».''',
                ActivityPackage.Literals.PERIPHERAL_ACTION__SCHEDULING_TYPE, ALAP_SCENARIO_3)
            return
        } else if (requiresALAP && move.schedulingType == SchedulingType::ASAP) {
            error('''Move «move.name» must be set to ALAP to guarantee concatenation with successor move «successorMove.name».''',
                ActivityPackage.Literals.PERIPHERAL_ACTION__SCHEDULING_TYPE, ALAP_SCENARIO_2)
            return
        }
    }

    /**
     * This method makes sure that a Claim has outgoing edges.
     */
    @Check
    def checkIfUsedInActivityFlow(Claim claim) {
        if (claim.outgoingEdges.isEmpty) {
            error('''Claim «claim.name» has no outgoing edges.''', claim,
                EdgPackage.Literals.NODE__OUTGOING_EDGES, INVALID_CLAIM)
        }
    }


   /**
     * This event action pointing to EventResource
     */
    @Check
    def checkEvent(EventAction event) {
        if (event.resource.resource.resourceType !== ResourceType.EVENT) {
            val name = event instanceof RequireEvent ? "Require" : "Raise"
            error('''«name» «event.resource.fqn» is not an Event''', event,
                ActivityPackage.Literals.RESOURCE_ACTION__RESOURCE, INVALID_RESOURCE)
        }
    }

    @Check
    def checkResource(ResourceAction action) {
        if ( action instanceof EventAction ) {
            return
        }
        
        if (action.resource.resource.resourceType !== ResourceType.REGULAR) {
            val name = action.class.simpleName
            error('''«name» «action.resource.fqn» is not a machine resource.''', action,
                ActivityPackage.Literals.RESOURCE_ACTION__RESOURCE, INVALID_RESOURCE)
        }
    }
 
   /**
     * This method checks passive claims in relation to actions on the resource.
     */
    @Check
    def checkPassiveClaim(Claim claim) {
        if(claim.resource.resource.resourceType !== ResourceType.REGULAR) return;

        val activity = claim.graph as Activity
        val hasActions = !activity.nodes.filter(PeripheralAction).filter[resource === claim.resource].empty
        // isCollisionArea = true implies hasActions = false as the 
        // collision area does not contain peripherals to perform actions.
        val isCollisionArea = claim.resource.resource.peripherals.empty
        if (claim.passive) {
            if (hasActions) {
                error('''Claim «claim.name» cannot be passive. There are actions for resource «claim.resource.fqn».''', claim,
                    ActivityPackage.Literals.CLAIM__PASSIVE, INVALID_PASSIVE_CLAIM)
                return
            }

            if (!claim.incomingEdges.isEmpty) {
                warning('''Passive claim «claim.name» should not have incoming edges as they may possibly block other activities.''', claim,
                    EdgPackage.Literals.NODE__INCOMING_EDGES, PASSIVE_CLAIM_INCOMING_EDGES)
            }
            for (release : ActivityUtil.getReleases(activity, claim.resource).reject[outgoingEdges.isEmpty]) {
                warning('''Passive release «release.name» should not have outgoing edges as they may possibly block other activities.''', release,
                    EdgPackage.Literals.NODE__OUTGOING_EDGES, PASSIVE_RELEASE_OUTGOING_EDGES)
            }
        } else if (!isCollisionArea && !hasActions) {
            info('''Claim «claim.name» can be made passive. There are no actions for resource «claim.resource.fqn».''', claim,
                ActivityPackage.Literals.CLAIM__PASSIVE, SUGGEST_PASSIVE_CLAIM)
        }
    }

    /**
     * This method makes sure that a Release has incoming edges.
     */
    @Check
    def checkIfUsedInActivityFlow(Release release) {
        if (release.incomingEdges.isEmpty) {
            error('''Release «release.name» has no incoming edges.''', release,
                EdgPackage.Literals.NODE__INCOMING_EDGES, INVALID_RELEASE)
        }
    }

    /**
     * This method makes sure that a SyncBar has incoming edges.
     */
    @Check
    def checkHasEdges(SyncBar syncBar) {
        // outgoing edges are mandatory on a syncBar: So not tested
        if (syncBar.outgoingEdges.size > 0 && syncBar.incomingEdges.isEmpty) {
            error('''SyncBar «syncBar.name» has no incoming edges.''', syncBar.outgoingEdges.get(0),
                EdgPackage.Literals.EDGE__SOURCE_NODE, INVALID_SYNC_BAR)
        }
        if (syncBar.incomingEdges.size === 1 && syncBar.outgoingEdges.size === 1) {
            warning('''SyncBar «syncBar.name» with only 1 incoming and 1 outgoing edge can be removed.''',
                syncBar.outgoingEdges.get(0), EdgPackage.Literals.EDGE__SOURCE_NODE, REMOVE_SYNC_BAR)

        }
    }

    /**
     * There are not many restriction on events
     * Only check if they are used.
     */
    @Check
    def checkIfHasPredecessorAction(RaiseEvent event) {
        if (event.nearestPredecessors(Action).empty) {
            error('''Event '«event.resource.name»' should be preceded with an action.''', event,
                EdgPackage.Literals.NODE__NAME, DEAD_EVENT)
        }
    }

    /**
     * There are not many restriction on events
     * Only check if they are used.
     */
    @Check
    def checkIfHasSuccessorAction(RequireEvent event) {
        if (event.nearestSuccessors(Action).empty) {
            error('''Event «event.resource.name» should be succeeded by an action.''', event,
                EdgPackage.Literals.NODE__NAME, DEAD_EVENT)
        }
    }

    /**
     * Check if event.resource.fqn is unique within activity.
     */
    @Check
    def checkEventNameUnique(Activity activity) {
        activity.nodes.filter(EventAction).groupBy[resource.fqn].values.filter[size > 2].flatten.forEach [
            error('''One require and/or raise event with name '«resource.fqn»' per activity are allowed.''', it,
                ActivityPackage.Literals.RESOURCE_ACTION__RESOURCE, ONE_EVENT_WITH_SAME_NAME_PER_ACT)
        ]
    }

    /**
     * Check if event.resource.fqn is unique within activity.
     */
    @Check
    def checkRequireRaise(Activity activity) {
        activity.allNodesInTopologicalOrder.filter(EventAction).groupBy[resource.fqn].values.filter[size == 2].filter [
            !(get(0) instanceof RequireEvent && get(1) instanceof RaiseEvent)
        ].forEach [ seq |
            seq.forEach [
                error('''Expecting 'require' followed by 'raise' for event '«resource.fqn»', not '«seq.get(0).type»' followed by '«seq.get(1).type»' ''',
                    it, ActivityPackage.Literals.RESOURCE_ACTION__RESOURCE, ONE_EVENT_WITH_SAME_NAME_PER_ACT)
            ]
        ]
    }

    def type(EventAction event) {
        return event instanceof RequireEvent ? 'require' : 'raise'
    }

    /**
     * This methods makes sure that a PeripheralAction is not a dead end by verifying
     * that it has incoming and outgoing edges.
     */
    @Check
    def checkIfUsedInActivityFlow(PeripheralAction action) {
        if (action.incomingEdges.isEmpty && action.outgoingEdges.isEmpty) {
            error('''«action.name» is a dead action. It has no incoming and no outgoing edges.''', action,
                EdgPackage.Literals.NODE__NAME, DEAD_ACTION)
        } else if (action.incomingEdges.isEmpty) {
            error('''«action.name» is a dead action. It has no incoming edges.''', action,
                EdgPackage.Literals.NODE__INCOMING_EDGES, NO_INCOMING_EDGES)
        } else if (action.outgoingEdges.isEmpty) {
            error('''«action.name» is a dead action. It has no outgoing edges.''', action,
                EdgPackage.Literals.NODE__OUTGOING_EDGES, NO_OUTGOING_EDGES)
        }
    }

    /**
     * This methods makes sure that an Action has only one incoming and one outgoing edge.
     */
    @Check
    def checkActionHasOneIncomingAndOneOutgoingEdge(Node node) {
        if (node instanceof SyncBar) {
            return
        }

        if (node.incomingEdges.size > 1) {
            warning('''«node.name» can only have one incoming dependency. Use sync bars instead.''', node,
                EdgPackage.Literals.NODE__INCOMING_EDGES, MORE_THAN_ONE_INCOMING_EDGE)
        } else if (node.outgoingEdges.size > 1) {
            warning('''«node.name» can only have one outgoing dependency. Use sync bars instead.''', node,
                EdgPackage.Literals.NODE__OUTGOING_EDGES, MORE_THAN_ONE_OUTGOING_EDGE)
        }
    }

    /**
     * This method makes sure that only one claim per resource in an activity is used.
     * The method collects all of the claims from an activity in a hashmap, with a resource as a key.
     * If a claim already exists for that resource, then we raise an error.
     */
    @Check
    def checkOneClaimPerResourcePerActivity(Activity activity) {
        activity.nodes.filter(Claim).groupBy[resource].values.filter[size > 1].flatten.forEach [ duplicateClaim |
            error('''Only one claim per resource «duplicateClaim.resource.name» per activity is allowed.''',
                duplicateClaim, EdgPackage.Literals.NODE__NAME, ONE_CLAIM_PER_RESOURCE_PER_ACT)
        ]
    }

    /**
     * This method makes sure that only one Release per resource in an activity is used.
     * The method collects all of the Release from an activity in a hashmap, with a resource as a key.
     * If a Release already exists for that resource, then we raise an error.
     */
    @Check
    def checkOneReleasePerResourcePerActivity(Activity activity) {
        activity.nodes.filter(Release).groupBy[resource].values.filter[size > 1].flatten.forEach [ duplicateRelease |
            error('''Only one release per resource «duplicateRelease.resource.name» per activity is allowed.''',
                duplicateRelease, EdgPackage.Literals.NODE__NAME, ONE_RELEASE_PER_RESOURCE_PER_ACT)
        ]
    }

    /**
     * This methods checks two things:
     * 1) If a claim is defined for a resource on the current PeripheralAction.
     * 2) If a claim exists for the PeripheralAction.
     */
    @Check
    def checkActionHasAValidClaimAndResource(PeripheralAction action) {
        var claim = action.graph.nodes.objectsOfKind(Claim).toMap[resource].get(action.getResource());
        if (null === claim) {
            error('''No claim defined for resource «action.resource.name». It is required before «action.name» can be used.''',
                claim, EdgPackage.Literals.NODE__NAME, NO_CLAIM_DEFINED_FOR_RESOURCE)
        } else if (!action.allPredecessors.contains(claim)) {
            warning('''Resource «action.resource.name» is not properly claimed (by «claim.name») before «action.name» is used.''',
                claim, EdgPackage.Literals.NODE__NAME, RESOURCE_NOT_PROPERLY_CLAIMED)
        }

    }

    /**
     * This methods checks two things:
     * 1) If a Release is defined for a resource on the current PeripheralAction.
     * 2) If a Release exists for the PeripheralAction.
     */
    @Check
    def checkActionHasAValidReleaseAndResource(PeripheralAction action) {
        var release = action.graph.nodes.objectsOfKind(Release).toMap[resource].get(action.getResource());
        if (null === release) {
            error('''No release defined for resource «action.resource.name». It is required before «action.name» can be used.''',
                action, EdgPackage.Literals.NODE__NAME, NO_RELEASE_DEFINED_FOR_RESOURCE)

        } else if (!action.allSuccessors.contains(release)) {
            warning('''Resource «action.resource.name» is not properly released (by «release.name») before «action.name» is used.''',
                action, EdgPackage.Literals.NODE__NAME, RESOURCE_NOT_PROPERLY_RELEASED)
        }

    }

    /**
     * This method makes sure that a PeripheralAction has not created a cycle.
     * It works by making sure there is no intersection between the predecessors and successors 
     * of the PeripheralAction. It does, however, show multiple errors as it is per action.
     */
    @Check
    def checkForCycles(PeripheralAction action) {
        var cycle = action.allSuccessors.toSet;
        cycle.retainAll(action.allPredecessors.toSet)
        if (!cycle.isEmpty) {
            error('''Actions «cycle.map[name].join(",")» have a cyclic dependency.''', action,
                EdgPackage.Literals.NODE__NAME, CYCLE_DETECTED)
        }
    }

    /**
     * This method makes sure that (expanded) activities do not have the same name
     */
    @Check
    def checkForPotentiallyDuplicateActivityNames(ActivitySet activitySet) {
        activitySet.activities.collect[expandedNames].groupBy[key].values.filter[size > 1].flatten.forEach [ da |

            error('''Activity '«da.value.name»' conflicts with other activities. Please remove all conflicting instances.''',
                da.value, EdgPackage.Literals.EDITABLE_DIRECTED_GRAPH__NAME, DUPLICATE_ACTIVITY_NAME)
        ]
    }

    private def expandedNames(Activity activity) {
        val result = newHashMap
        result.put(activity.name, activity)
        activity.nodes.filter(ResourceAction).map[resource].filter(Resource).reject[items.isEmpty].unique
            .map[items.toSet].toList.cartesianProduct.map[ActivityUtil.expandName(activity, it)]
            .forEach[result.put(it, activity)]
        return result.entrySet
    }

    private def cartesianProduct(List<Set<ResourceItem>> r) {
        Sets.cartesianProduct(r)
    }

    /**
     * This method checks if a PeripheralAction is done in parallel with another action on the same Peripheral.
     * It works by calculating the actions for the current peripheral action's peripheral and then checking if any of them are a predecessor
     * or successor of the current action. 
     */
    @Check
    def checkActionInParallelForSamePeripheral(PeripheralAction pAction) {
        val allNodesInTheGraph = pAction.graph.nodes
        val indexOfpAction = allNodesInTheGraph.indexOf(pAction)
        for (otherAction : getActionsFor(pAction.resource, pAction.peripheral, PeripheralAction,
            allNodesInTheGraph.subList(indexOfpAction + 1, allNodesInTheGraph.size()))) {
            // To be sure that peripheral actions cannot be done in parallel, 
            // all other actions for the same peripheral should either be a predecessor or successor.
            if (!(pAction.allSuccessors.contains(otherAction) || pAction.allPredecessors.contains(otherAction))) {
                // This warning will typically pop-up when using the actions, therefore it is written on all their references
                error('''«pAction.name» cannot be done in parallel with «otherAction.name», as they share the same peripheral.''',
                    pAction, EdgPackage.Literals.NODE__NAME, ACTION_IN_PARALLEL_FOR_SAME_PERIPHERAL)
            }
        }
    }

    /**
     * This method makes sure оnly one location prerequisite per peripheral is allowed.
     */
    // TODO: This validation is triggered more than once.
    // More can be found here: https://ci.tno.nl/gitlab/harm.munk-tno/l-sat/-/issues/73
    @Check
    def HashMap<ResourcePeripheralKey, SymbolicPosition> indexPrerequisites(Activity activity) {
        if (activity.prerequisites.isNullOrEmpty) {
            return new HashMap<ResourcePeripheralKey, SymbolicPosition>
        }
        var result = new HashMap(activity.prerequisites.size());
        for (prerequisite : activity.prerequisites) {
            val key = prerequisite.createKey;
            if (result.containsKey(key)) {
                error('''Only one location prerequisite for peripheral «key.fqn» is allowed.''', prerequisite,
                    ActivityPackage.Literals.LOCATION_PREREQUISITE__PERIPHERAL,
                    MORE_THAN_ONE_LOCATION_PREREQ_FOR_PERIPHERAL)

            }
            result.put(key, prerequisite.getPosition());
        }
        return result
    }

    /**
     * Multiple validations for Move cramped into this method:
     * 1) Check for location prerequisite for peripheral for move.
     * 2) Check that source and target position are different.
     * 3) Check that the path for a move is complete.
     * 4) Check if settling is already defined for a path.
     */
    @Check
    def checkMoveHasLocationPrerequisiteForPeripheral(Move move) {
        if(!move.positionMove) return
        val moveActivity = EcoreUtil2.getContainerOfType(move, Activity)
        var prerequisites = indexPrerequisites(moveActivity)
        val predecessorMove = move.predecessorMove
        val key = move.createKey

        if (null === predecessorMove && !prerequisites.containsKey(key)) {
            error('''This movement requires a location prerequisite for peripheral «fqn(move.peripheral)».''',
                ActivityPackage.Literals.PERIPHERAL_ACTION__PERIPHERAL, NO_LOC_PREREQUISTIE_FOR_PERIPHERAL_FOR_MOVE)
            return
        }

        var sourcePosition = if(null === predecessorMove) prerequisites.get(key) else predecessorMove.targetPosition
        var targetPosition = move.targetPosition
        if (sourcePosition == targetPosition) {
            warning('''Peripheral «move.fqn» already at «sourcePosition.name».''', move,
                ActivityPackage.Literals.PERIPHERAL_ACTION__PERIPHERAL, SAME_SOURCE_TARGET_FOR_PERIPHERAL)
        }

        var pathTargetRef = MachineQueries.findPath(sourcePosition, targetPosition, move.profile);
        if (null === pathTargetRef) {
            error('''Don't know how to move peripheral «fqn(move.peripheral)» from «sourcePosition.name» to «targetPosition.name» using speed profile «move.profile.name».''',
                move, ActivityPackage.Literals.MOVE__PROFILE, NO_PATH_FOUND_FOR_PROFILE)
        }

        if (move.isPassing) {
            if (null !== pathTargetRef && !pathTargetRef.getSettling().isEmpty()) {
                error('''«move.name» cannot pass symbolic position «targetPosition.name», as settling is defined for its path.''',
                    move, ActivityPackage.Literals.MOVE__STOP_AT_TARGET, SETTLING_ALREADY_DEFINED_FOR_PATH)
            }
        }
    }

    /**
     * Check if settling is already defined for a distance move.
     */
    @Check
    def checkSettling(Move move) {
        if (move.isContinuing) {
            val settling = move.distance?.settling
            val isSettling = (settling === null) ? false : !settling.empty
            if (isSettling) {
                error('''«move.name» cannot pass «move.distance.name», as settling has been defined for axes : «settling.join(',')»''',
                    move, ActivityPackage.Literals.MOVE__STOP_AT_TARGET, SETTLING_ALREADY_DEFINED_FOR_MOVE)
            }
        }
    }

    private def static String fqn(Peripheral peripheral) {
        return if (peripheral.getResource() === null)
            "<<unknown>>"
        else
            peripheral.getResource().getName() + '.' + peripheral.getName();
    }

    def static String names(Iterable<? extends Node> nodes) {
        val iNodes = nodes.iterator
        if(!iNodes.hasNext) return ''

        val result = new StringBuilder(iNodes.next?.name)
        while (iNodes.hasNext) {
            val nextName = iNodes.next?.name
            result.append(iNodes.hasNext ? ', ' : ' and ').append(nextName)
        }
        return result.toString
    }

    def static String id(Node node) {
        return id(node, false);
    }

    def static String id(Node node, boolean capitalize) {
        if (node === null) {
            return null
        }
        var id = switch it: node {
            Claim: '''claim «name»'''
            Release: '''release «name»'''
            SimpleAction: '''action «name»'''
            Move: '''«IF !isStopAtTarget»«IF positionMove»passing «ELSE»continuing «ENDIF»«ENDIF»move «name»'''
            SyncBar: '''sync-bar «name»'''
        }
        return capitalize ? id.toFirstUpper : id
    }
}
