/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.config;

import io.helidon.config.Config;
import io.helidon.config.ConfigException;
import io.helidon.config.ConfigThreadFactory;
import io.helidon.config.ConfigUtils;
import io.helidon.config.spi.ChangeEventType;
import io.helidon.config.spi.ChangeWatcher;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

public final class FileSystemWatcher
implements ChangeWatcher<Path> {
    private static final System.Logger LOGGER = System.getLogger(FileSystemWatcher.class.getName());
    private final List<WatchEvent.Modifier> watchServiceModifiers = new LinkedList<WatchEvent.Modifier>();
    private ScheduledExecutorService executor;
    private final boolean defaultExecutor;
    private final long initialDelay;
    private final long delay;
    private final TimeUnit timeUnit;
    private final List<TargetRuntime> runtimes = Collections.synchronizedList(new LinkedList());

    private FileSystemWatcher(Builder builder) {
        ScheduledExecutorService executor = builder.executor;
        if (executor == null) {
            this.executor = Executors.newSingleThreadScheduledExecutor(new ConfigThreadFactory("file-watch-polling"));
            this.defaultExecutor = true;
        } else {
            this.executor = executor;
            this.defaultExecutor = false;
        }
        this.watchServiceModifiers.addAll(builder.watchServiceModifiers);
        this.initialDelay = builder.initialDelay;
        this.delay = builder.delay;
        this.timeUnit = builder.timeUnit;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static FileSystemWatcher create() {
        return FileSystemWatcher.builder().build();
    }

    public synchronized void start(Path target, Consumer<ChangeWatcher.ChangeEvent<Path>> listener) {
        if (this.defaultExecutor && this.executor.isShutdown()) {
            this.executor = Executors.newSingleThreadScheduledExecutor(new ConfigThreadFactory("file-watch-polling"));
        }
        if (this.executor.isShutdown()) {
            throw new ConfigException("Cannot start a watcher for path " + String.valueOf(target) + ", as the executor service is shutdown");
        }
        Monitor monitor = new Monitor(listener, target, this.watchServiceModifiers);
        ScheduledFuture<?> future = this.executor.scheduleWithFixedDelay(monitor, this.initialDelay, this.delay, this.timeUnit);
        this.runtimes.add(new TargetRuntime(monitor, future));
    }

    public synchronized void stop() {
        this.runtimes.forEach(TargetRuntime::stop);
        if (this.defaultExecutor) {
            ConfigUtils.shutdownExecutor(this.executor);
        }
    }

    public Class<Path> type() {
        return Path.class;
    }

    public void initWatchServiceModifiers(WatchEvent.Modifier ... modifiers) {
        this.watchServiceModifiers.addAll(Arrays.asList(modifiers));
    }

    public static final class Builder
    implements io.helidon.common.Builder<Builder, FileSystemWatcher> {
        private final List<WatchEvent.Modifier> watchServiceModifiers = new LinkedList<WatchEvent.Modifier>();
        private ScheduledExecutorService executor;
        private long initialDelay = 1000L;
        private long delay = 100L;
        private TimeUnit timeUnit = TimeUnit.MILLISECONDS;

        private Builder() {
        }

        public FileSystemWatcher build() {
            return new FileSystemWatcher(this);
        }

        public Builder config(Config metaConfig) {
            metaConfig.get("initial-delay-millis").asLong().ifPresent(initDelay -> {
                this.initialDelay = this.timeUnit.convert((long)initDelay, TimeUnit.MILLISECONDS);
            });
            metaConfig.get("delay-millis").asLong().ifPresent(delayMillis -> {
                this.delay = this.timeUnit.convert((long)delayMillis, TimeUnit.MILLISECONDS);
            });
            return this;
        }

        public Builder executor(ScheduledExecutorService executor) {
            this.executor = executor;
            return this;
        }

        @Deprecated(since="4.0.0")
        public Builder schedule(long initialDelay, long delay, TimeUnit timeUnit) {
            this.initialDelay = initialDelay;
            this.delay = delay;
            this.timeUnit = timeUnit;
            return this;
        }

        public Builder initialDelay(Duration initialDelay) {
            this.initialDelay = initialDelay.toMillis();
            this.timeUnit = TimeUnit.MILLISECONDS;
            return this;
        }

        public Builder delay(Duration delay) {
            this.delay = delay.toMillis();
            this.timeUnit = TimeUnit.MILLISECONDS;
            return this;
        }

        public Builder addWatchServiceModifier(WatchEvent.Modifier modifier) {
            this.watchServiceModifiers.add(modifier);
            return this;
        }

        public Builder watchServiceModifiers(List<WatchEvent.Modifier> modifiers) {
            this.watchServiceModifiers.clear();
            this.watchServiceModifiers.addAll(modifiers);
            return this;
        }
    }

    private static final class Monitor
    implements Runnable {
        private final WatchService watchService;
        private final Consumer<ChangeWatcher.ChangeEvent<Path>> listener;
        private final Path target;
        private final List<WatchEvent.Modifier> watchServiceModifiers;
        private final boolean watchingFile;
        private final Path watchedDir;
        private volatile boolean failed = true;
        private volatile boolean shouldStop = false;
        private volatile boolean fileExists;
        private WatchKey watchKey;

        private Monitor(Consumer<ChangeWatcher.ChangeEvent<Path>> listener, Path target, List<WatchEvent.Modifier> watchServiceModifiers) {
            try {
                this.watchService = FileSystems.getDefault().newWatchService();
            }
            catch (IOException e) {
                throw new ConfigException("Cannot obtain WatchService.", e);
            }
            this.listener = listener;
            this.target = target;
            this.watchServiceModifiers = watchServiceModifiers;
            this.fileExists = Files.exists(target, new LinkOption[0]);
            this.watchingFile = !Files.isDirectory(target, new LinkOption[0]);
            this.watchedDir = this.watchingFile ? target.getParent() : target;
        }

        @Override
        public void run() {
            if (this.shouldStop) {
                return;
            }
            if (this.failed) {
                this.register();
            }
            if (this.failed) {
                return;
            }
            WatchKey key = this.watchService.poll();
            if (null == key) {
                return;
            }
            List<WatchEvent<?>> watchEvents = key.pollEvents();
            if (watchEvents.isEmpty()) {
                key.cancel();
                this.listener.accept((ChangeWatcher.ChangeEvent<Path>)ChangeWatcher.ChangeEvent.create((Object)this.target, (ChangeEventType)ChangeEventType.CHANGED));
                this.failed = true;
                return;
            }
            Iterator<WatchEvent<?>> iterator = watchEvents.iterator();
            while (iterator.hasNext()) {
                WatchEvent<?> watchEvent;
                WatchEvent<?> event = watchEvent = iterator.next();
                Path eventPath = (Path)event.context();
                if (this.watchingFile && !this.target.endsWith(eventPath)) continue;
                eventPath = this.watchedDir.resolve(eventPath);
                WatchEvent.Kind<?> kind = event.kind();
                if (kind.equals(StandardWatchEventKinds.OVERFLOW)) {
                    LOGGER.log(System.Logger.Level.DEBUG, "Overflow event on path: " + String.valueOf(eventPath));
                    continue;
                }
                if (kind.equals(StandardWatchEventKinds.ENTRY_CREATE)) {
                    LOGGER.log(System.Logger.Level.DEBUG, "Entry created. Path: " + String.valueOf(eventPath));
                    this.listener.accept((ChangeWatcher.ChangeEvent<Path>)ChangeWatcher.ChangeEvent.create((Object)eventPath, (ChangeEventType)ChangeEventType.CREATED));
                    continue;
                }
                if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                    LOGGER.log(System.Logger.Level.DEBUG, "Entry deleted. Path: " + String.valueOf(eventPath));
                    this.listener.accept((ChangeWatcher.ChangeEvent<Path>)ChangeWatcher.ChangeEvent.create((Object)eventPath, (ChangeEventType)ChangeEventType.DELETED));
                    continue;
                }
                if (kind != StandardWatchEventKinds.ENTRY_MODIFY) continue;
                LOGGER.log(System.Logger.Level.DEBUG, "Entry changed. Path: " + String.valueOf(eventPath));
                this.listener.accept((ChangeWatcher.ChangeEvent<Path>)ChangeWatcher.ChangeEvent.create((Object)eventPath, (ChangeEventType)ChangeEventType.CHANGED));
            }
            if (!key.reset()) {
                LOGGER.log(System.Logger.Level.TRACE, "Directory of '" + String.valueOf(this.target) + "' is no more valid to be watched.");
                this.failed = true;
            }
        }

        private void fire(Path target, ChangeEventType eventType) {
            this.listener.accept((ChangeWatcher.ChangeEvent<Path>)ChangeWatcher.ChangeEvent.create((Object)target, (ChangeEventType)eventType));
        }

        private synchronized void register() {
            if (this.shouldStop) {
                this.failed = true;
                return;
            }
            boolean oldFileExists = this.fileExists;
            try {
                Path cleanTarget = this.target(this.target);
                Path watchedDirectory = Files.isDirectory(cleanTarget, new LinkOption[0]) ? cleanTarget : this.parentDir(cleanTarget);
                WatchKey oldWatchKey = this.watchKey;
                this.watchKey = watchedDirectory.register(this.watchService, new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE}, this.watchServiceModifiers.toArray(new WatchEvent.Modifier[0]));
                this.failed = false;
                if (null != oldWatchKey) {
                    oldWatchKey.cancel();
                }
            }
            catch (IOException e) {
                LOGGER.log(System.Logger.Level.TRACE, "Failed to register watch service", (Throwable)e);
                this.failed = true;
            }
            this.fileExists = Files.exists(this.target, new LinkOption[0]);
            if (this.fileExists != oldFileExists) {
                if (this.fileExists) {
                    this.fire(this.target, ChangeEventType.CREATED);
                } else {
                    this.fire(this.target, ChangeEventType.DELETED);
                }
            }
        }

        private synchronized void stop() {
            this.shouldStop = true;
            if (null != this.watchKey) {
                this.watchKey.cancel();
            }
            try {
                this.watchService.close();
            }
            catch (IOException e) {
                LOGGER.log(System.Logger.Level.TRACE, "Failed to close watch service", (Throwable)e);
            }
        }

        private Path target(Path path) throws IOException {
            Path target = path;
            while (Files.isSymbolicLink(target)) {
                target = target.toRealPath(new LinkOption[0]);
            }
            return target;
        }

        private Path parentDir(Path path) {
            Path parent = path.toAbsolutePath().getParent();
            if (parent == null) {
                throw new ConfigException(String.format("Cannot find parent directory for '%s' to register watch service.", path));
            }
            return parent;
        }
    }

    private static final class TargetRuntime {
        private final Monitor monitor;
        private final ScheduledFuture<?> future;

        private TargetRuntime(Monitor monitor, ScheduledFuture<?> future) {
            this.monitor = monitor;
            this.future = future;
        }

        public void stop() {
            this.monitor.stop();
            this.future.cancel(true);
        }
    }
}

