/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

const {
  TRACER_FIELDS_INDEXES,
} = require("resource://devtools/server/actors/tracer.js");

function initialState() {
  return {
    // These fields are mutable as they are large arrays and UI will rerender based on their size

    // The three next array are always of the same size.
    // List of all trace resources, as defined by the server codebase (See the TracerActor)
    mutableTraces: [],
    // Array of arrays. This is of the same size as mutableTraces.
    // Store the indexes within mutableTraces of each children matching the same index in mutableTraces.
    mutableChildren: [],
    // Indexes of parents within mutableTraces.
    mutableParents: [],

    // Frames are also a trace resources, but they are stored in a dedicated array.
    mutableFrames: [],

    // List of indexes within mutableTraces of top level trace, without any parent.
    mutableTopTraces: [],

    // List of all trace resources indexes within mutableTraces which are about dom mutations
    mutableMutationTraces: [],

    // Index of the currently selected trace within `mutableTraces`.
    selectedTraceIndex: null,

    // Object like the one generated by `generateInlinePreview`, but for the currently selected trace
    previews: null,

    // Runtime versions to help show warning when there is a mismatch between frontend and backend versions
    localPlatformVersion: null,
    remotePlatformVersion: null,
  };
}

// eslint-disable-next-line complexity
function update(state = initialState(), action) {
  switch (action.type) {
    case "TRACING_TOGGLED": {
      if (action.enabled) {
        return initialState();
      }
      return state;
    }

    case "TRACING_CLEAR": {
      return initialState();
    }

    case "ADD_TRACES": {
      addTraces(state, action.traces);
      return { ...state };
    }

    case "SELECT_TRACE": {
      const { traceIndex } = action;
      if (
        traceIndex < 0 ||
        traceIndex >= state.mutableTraces.length ||
        traceIndex == state.selectedTraceIndex
      ) {
        return state;
      }

      const trace = state.mutableTraces[traceIndex];
      return {
        ...state,
        selectedTraceIndex: traceIndex,

        // Also compute the inline preview data when we select a trace
        // and we have the values recording enabled.
        previews: generatePreviewsForTrace(state, trace),
      };
    }

    case "SELECT_FRAME":
    case "PAUSED": {
      if (!state.previews && state.selectedTraceIndex == null) {
        return state;
      }

      // Reset the selected trace and previews when we pause/step/select a frame in the scope panel,
      // so that it is no longer highlighted, nor do we show inline variables.
      return {
        ...state,
        selectedTraceIndex: null,
        previews: null,
      };
    }

    case "SET_SELECTED_LOCATION": {
      // Traces are reference to the generated location only, so ignore any original source being selected
      // and wait for SET_GENERATED_SELECTED_LOCATION instead.
      if (action.location.source.isOriginal) return state;

      // Ignore if the currently selected trace matches the new location.
      if (
        state.selectedTrace &&
        locationMatchTrace(action.location, state.selectedTrace)
      ) {
        return state;
      }

      // Lookup for a trace matching the newly selected location
      for (const trace of state.mutableTraces) {
        if (locationMatchTrace(action.location, trace)) {
          return {
            ...state,
            selectedTrace: trace,
          };
        }
      }

      return {
        ...state,
        selectedTrace: null,
      };
    }

    case "SET_GENERATED_SELECTED_LOCATION": {
      // When selecting an original location, we have to wait for the newly selected original location
      // to be mapped to a generated location so that we can find a matching trace.

      // Ignore if the currently selected trace matches the new location.
      if (
        state.selectedTrace &&
        locationMatchTrace(action.generatedLocation, state.selectedTrace)
      ) {
        return state;
      }

      // Lookup for a trace matching the newly selected location
      for (const trace of state.mutableTraces) {
        if (locationMatchTrace(action.generatedLocation, trace)) {
          return {
            ...state,
            selectedTrace: trace,
          };
        }
      }

      return {
        ...state,
        selectedTrace: null,
      };
    }

    case "CLEAR_SELECTED_LOCATION": {
      return {
        ...state,
        selectedTrace: null,
      };
    }
    case "SET_RUNTIME_VERSIONS": {
      return {
        ...state,
        localPlatformVersion: action.localPlatformVersion,
        remotePlatformVersion: action.remotePlatformVersion,
      };
    }
  }
  return state;
}

function addTraces(state, traces) {
  const {
    mutableTraces,
    mutableMutationTraces,
    mutableFrames,
    mutableTopTraces,
    mutableChildren,
    mutableParents,
  } = state;

  function matchParent(traceIndex, depth) {
    // The very last element is the one matching traceIndex,
    // so pick the one added just before.
    // We consider that traces are reported by the server in the execution order.
    let idx = mutableTraces.length - 2;
    while (idx != null) {
      const trace = mutableTraces[idx];
      if (!trace) {
        break;
      }
      const currentDepth = trace[TRACER_FIELDS_INDEXES.DEPTH];
      if (currentDepth < depth) {
        mutableChildren[idx].push(traceIndex);
        mutableParents.push(idx);
        return;
      }
      idx = mutableParents[idx];
    }

    // If no parent was found, flag it as top level trace
    mutableTopTraces.push(traceIndex);
    mutableParents.push(null);
  }
  for (const traceResource of traces) {
    // For now, only consider traces from the top level target/thread
    if (!traceResource.targetFront.isTopLevel) {
      continue;
    }

    const type = traceResource[TRACER_FIELDS_INDEXES.TYPE];

    switch (type) {
      case "frame": {
        // Store the object used by SmartTraces
        mutableFrames.push({
          functionDisplayName: traceResource[TRACER_FIELDS_INDEXES.FRAME_NAME],
          source: traceResource[TRACER_FIELDS_INDEXES.FRAME_URL],
          sourceId: traceResource[TRACER_FIELDS_INDEXES.FRAME_SOURCEID],
          line: traceResource[TRACER_FIELDS_INDEXES.FRAME_LINE],
          column: traceResource[TRACER_FIELDS_INDEXES.FRAME_COLUMN],
        });
        break;
      }

      case "enter": {
        const traceIndex = mutableTraces.length;
        mutableTraces.push(traceResource);
        mutableChildren.push([]);
        const depth = traceResource[TRACER_FIELDS_INDEXES.DEPTH];
        matchParent(traceIndex, depth);
        break;
      }

      case "exit": {
        // The sidebar doesn't use this information yet
        break;
      }

      case "dom-mutation": {
        const traceIndex = mutableTraces.length;
        mutableTraces.push(traceResource);
        mutableChildren.push([]);
        mutableMutationTraces.push(traceIndex);

        const depth = traceResource[TRACER_FIELDS_INDEXES.DEPTH];
        matchParent(traceIndex, depth);
        break;
      }

      case "event": {
        const traceIndex = mutableTraces.length;
        mutableTraces.push(traceResource);
        mutableChildren.push([]);
        mutableParents.push(null);
        mutableTopTraces.push(traceIndex);
        break;
      }
    }
  }
}

function locationMatchTrace(location, trace) {
  return (
    trace.sourceId == location.sourceActor.id &&
    trace.lineNumber == location.line &&
    trace.columnNumber == location.column
  );
}

/**
 * Generate the previews object consumed by InlinePreviews React component.
 *
 * @param {Object} state
 * @param {Object} trace
 *        Trace reducer object.
 * @return {Object}
 *        Previews consumed by InlinePreviews.
 */
function generatePreviewsForTrace(state, trace) {
  let previews = state.previews;
  const argumentValues = trace[TRACER_FIELDS_INDEXES.ENTER_ARGS];
  const argumentNames = trace[TRACER_FIELDS_INDEXES.ENTER_ARG_NAMES];
  if (argumentNames && argumentValues) {
    const frameIndex = trace[TRACER_FIELDS_INDEXES.FRAME_INDEX];
    const frame = state.mutableFrames[frameIndex];
    // Subtracting 1 from line as codemirror lines are 0 indexed
    const line = frame.line - 1;
    const column = frame.column;

    const preview = [];
    for (let i = 0; i < argumentNames.length; i++) {
      const name = argumentNames[i];

      // Values are either primitives, or an Object Front
      const objectGrip = argumentValues[i]?.getGrip
        ? argumentValues[i]?.getGrip()
        : argumentValues[i];

      preview.push({
        // All the argument will be show at the exact same spot.
        // Ideally it would be nice to show them next to each argument,
        // but the tracer currently expose the location of the first instruction
        // in the function body.
        line,
        column,

        // This attribute helps distinguish pause from trace previews
        type: "trace",
        name,
        value: objectGrip,
      });
    }

    // This is the shape of data expected by InlinePreviews component
    previews = {
      [line]: preview,
    };
  }
  return previews;
}

export default update;
