// Copyright (C) 2010-2014 David Sugar, Tycho Softworks.
// Copyright (C) 2015-2020 Cherokees of Idaho.
//
// This file is part of GNU uCommon C++.
//
// GNU uCommon C++ is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// GNU uCommon C++ is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with GNU uCommon C++.  If not, see <http://www.gnu.org/licenses/>.

#include "local.h"

namespace ucommon {

bool HMAC::has(const char *id)
{
    return (EVP_get_digestbyname(id) != NULL);
}

void HMAC::set(const char *digest, const secure::keybytes& key)
{
    secure::init();

    release();

    size_t len = key.size() / 8;

    hmactype = EVP_get_digestbyname(digest);
    if(hmactype && len) {
        context = HMAC_CTX_new();
        HMAC_Init_ex((HMAC_CTX *)context, *key, (int)len, (const EVP_MD *)hmactype, NULL);
    }
}

void HMAC::release(void)
{
    if(context) {
        HMAC_CTX_free((HMAC_CTX *)context);
        context = NULL;
    }

    bufsize = 0;
    memset(textbuf, 0, sizeof(textbuf));
}

bool HMAC::put(const void *address, size_t size)
{
    if(!context)
        return false;

    HMAC_Update((HMAC_CTX *)context, (const uint8_t *)address, size);
    return true;
}

const uint8_t *HMAC::get(void)
{
    unsigned count = 0;
    unsigned size = 0;

    if(bufsize)
        return buffer;

    if(!context)
        return NULL;

	HMAC_Final((HMAC_CTX *)context, buffer, &size);
	release();

	if(!size)
		return NULL;

    bufsize = size;

    while(count < bufsize) {
        snprintf(textbuf + (count * 2), 3, "%2.2x", buffer[count]);
        ++count;
    }
    return buffer;
}

} // namespace ucommon
