/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::polyeval::f_estrin_polyeval7;

/// Computes log(1+x) - x
///
/// ulp 0.5
pub fn f_log1pmxf(x: f32) -> f32 {
    let ax = x.to_bits() & 0x7fff_ffffu32;
    let xd = x as f64;

    if !x.is_normal() {
        if x.is_nan() {
            return x + x;
        }
        if x.is_infinite() {
            return f32::NAN;
        }
        if x == 0. {
            return x;
        }
    }

    // Use log1p(x) = log(1 + x) for |x| > 2^-6;
    if ax > 0x3c80_0000u32 {
        if x == -1. {
            return f32::NEG_INFINITY;
        }
        let x1p = xd + 1.;
        if x1p <= 0. {
            if x1p == 0. {
                return f32::NEG_INFINITY;
            }
            return f32::NAN;
        }
        return (crate::logs::log1pf::core_logf(x1p) - xd) as f32;
    }

    // log(1+x) is expected to be used near zero
    // Polynomial generated by Sollya in form log(1+x) - x = x^2 * R(x):
    // d = [-2^-6; 2^-6];
    // f_log1pf = (log(1+x) - x)/x^2;
    // Q = fpminimax(f_log1pf, 6, [|0, D...|], d);
    // See ./notes/log1pmxf.sollya

    let xd_sqr = xd * xd;
    let p = f_estrin_polyeval7(
        xd,
        f64::from_bits(0xbfe0000000000000),
        f64::from_bits(0x3fd55555555561c8),
        f64::from_bits(0xbfd0000000000f5a),
        f64::from_bits(0x3fc999998d26deab),
        f64::from_bits(0xbfc555554878739c),
        f64::from_bits(0x3fc24ab2e3c10eca),
        f64::from_bits(0xbfc0017973fafec4),
    ) * xd_sqr;
    p as f32
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_log1pmxfm() {
        assert_eq!(f_log1pmxf(-0.0000014305108), -1.0231815e-12);
        assert_eq!(f_log1pmxf(0.0), 0.0);
        assert_eq!(f_log1pmxf(2.0), -0.9013877);
        assert_eq!(f_log1pmxf(-0.7), -0.50397277);
        assert_eq!(
            f_log1pmxf(-0.0000000000043243),
            -0.000000000000000000000009349785
        );
    }
}
