#!/usr/bin/env python3
# Copyright (C) 2024 The Android Open Source Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from python.generators.diff_tests.testing import Csv, DataPath, Path, DiffTestBlueprint, TestSuite


class ExportTests(TestSuite):

  def test_to_firefox_profile(self):
    return DiffTestBlueprint(
        trace=DataPath('zip/perf_track_sym.zip'),
        query="""
          INCLUDE PERFETTO MODULE export.to_firefox_profile;

          SELECT export_to_firefox_profile();
        """,
        out=Path('firefox_profile.out'))

  def test_svg_timeline_simple(self):
    return DiffTestBlueprint(
        trace=DataPath('example_android_trace_30s.pb'),
        query="""
          INCLUDE PERFETTO MODULE export.to_svg;
          INCLUDE PERFETTO MODULE slices.with_context;

          SELECT *
          FROM _svg_timeline!(
            (SELECT utid, ts, dur, depth, name, 'slice_link' AS href, 1 AS track_group_order FROM thread_slice
             WHERE ts >= 89092060412 and ts < 89095305364 AND tid = 5511),
            (SELECT ts.utid, ts.ts, ts.dur, ts.state, ts.io_wait, ts.blocked_function, 'thread_state_link' AS href, 0 AS track_group_order FROM thread_state ts
             JOIN thread USING(utid)
             WHERE ts >= 89092060412 and ts < 89095305364 AND tid = 5511),
            utid,          -- svg_group_key
            utid,          -- track_group_key
            track_group_order,          -- track_order
            2000,
            500);
        """,
        out=Csv("""
                 "svg_group_key","svg"
954,"<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 2510 124"><style>
rect { cursor: pointer; }
path { cursor: pointer; }
text { font-family: sans-serif; pointer-events: none; dominant-baseline: central; }
a { text-decoration: none !important; }
a:hover { text-decoration: none !important; }
</style><g transform="translate(500,5)"><g transform="translate(0,30)"><g transform="translate(0,0)"><a href="slice_link" target="_blank"><rect x="0.0" y="5.0" width="2000.0" height="16.0" fill="hsl(90,45%,78%)"><title>JIT compiling int kvd.a(java.lang.CharSequence) (2.9ms)</title></rect><text x="1000.0" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling int kvd.a(java.lang.CharSequence)</text></a><a href="slice_link" target="_blank"><rect x="12.6291599056378" y="21.0" width="1917.76614113481" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (2.8ms)</title></rect><text x="971.512230473041" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1871.12305009401" y="37.0" width="33.1521475186177" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (48.1μs)</title></rect><text x="1887.69912385332" y="45.0" text-anchor="middle" font-size="11" fill="#333">S...</text></a><a href="slice_link" target="_blank"><rect x="1874.89051180371" y="53.0" width="14.4588140107692" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect all (21.0μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1895.48514249912" y="53.0" width="6.99622254959131" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect code (10.2μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1934.19789541557" y="21.0" width="59.8460844816606" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (86.9μs)</title></rect><text x="1964.1209376564" y="29.0" text-anchor="middle" font-size="11" fill="#333">TrimMaps</text></a><a href="slice_link" target="_blank"><rect x="1936.56625039051" y="37.0" width="54.0333505437469" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (78.4μs)</title></rect><text x="1963.58292566238" y="45.0" text-anchor="middle" font-size="11" fill="#333">virt...</text></a></g></g><g transform="translate(0,0)"><text x="5" y="15" font-size="11" fill="#333">954</text><g transform="translate(0,20)"><a href="thread_state_link" target="_blank"><rect x="0.0" y="2.0" width="2.6397910340517" height="8.0" fill="#2f7d31"><title>Thread State: Running (309.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="4.36693117543252" y="2.0" width="21.1222762259537" height="8.0" fill="#2f7d31"><title>Thread State: Running (2.5ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="25.4892074013862" y="2.0" width="1974.51079259861" height="8.0" fill="#a0a0a0"><title>Thread State: S (231.6ms)</title></rect></a></g></g></g></svg>"
                 """))

  def test_svg_timeline_multi_group(self):
    return DiffTestBlueprint(
        trace=DataPath('example_android_trace_30s.pb'),
        query="""
          INCLUDE PERFETTO MODULE export.to_svg;
          INCLUDE PERFETTO MODULE slices.with_context;

          WITH
            slice_data AS (
              SELECT
                utid,
                1 AS tord,
                utid, ts, dur, depth, name, 'slice_link' AS href
              FROM thread_slice
              WHERE ts >= 86625813931 and ts < 86731829201 AND tid IN (5511, 5517)
            ),
            thread_state_data AS (
              SELECT
                utid,
                utid,
                0 AS tord,
                utid, ts, dur, state, io_wait, blocked_function, 'thread_state_link' AS href
              FROM thread_state
              JOIN thread USING(utid)
             WHERE ts >= 86625813931 and ts < 86731829201 AND tid IN (5511, 5517)
            )
          SELECT *
          FROM _svg_timeline!(
            slice_data,
            thread_state_data,
            utid,
            utid,
            tord,
            2000,
            50);
        """,
        out=Csv("""
"svg_group_key","svg"
954,"<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 2060 124"><style>
rect { cursor: pointer; }
path { cursor: pointer; }
text { font-family: sans-serif; pointer-events: none; dominant-baseline: central; }
a { text-decoration: none !important; }
a:hover { text-decoration: none !important; }
</style><g transform="translate(50,5)"><g transform="translate(0,30)"><g transform="translate(0,0)"><a href="slice_link" target="_blank"><rect x="0.0" y="5.0" width="337.55291791609" height="16.0" fill="hsl(210,45%,78%)"><title>JIT compiling void ilm.&lt;init&gt;(ilk, iqt, java.util.concurrent.Executor) (7.6ms)</title></rect><text x="168.776458958045" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling void ilm.&lt;init&gt;(ilk, iqt, java.uti...</text></a><a href="slice_link" target="_blank"><rect x="0.644053719213396" y="21.0" width="332.701630760564" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (7.5ms)</title></rect><text x="166.994869099496" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="152.179142570069" y="37.0" width="178.242789792147" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (4.0ms)</title></rect><text x="241.300537466143" y="45.0" text-anchor="middle" font-size="11" fill="#333">ScopedCodeCacheWrite</text></a><a href="slice_link" target="_blank"><rect x="153.532149129729" y="53.0" width="176.69286177279" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect code (4.0ms)</title></rect><text x="241.878580016124" y="61.0" text-anchor="middle" font-size="11" fill="#333">mprotect code</text></a><a href="slice_link" target="_blank"><rect x="333.595895279596" y="21.0" width="3.5585535974219" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (80.0μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="333.732588219657" y="37.0" width="3.22031307798695" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (72.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="408.883857884892" y="5.0" width="144.75159605577" height="16.0" fill="hsl(30,45%,78%)"><title>JIT compiling android.content.SharedPreferences got.f(java.lang.String) (3.3ms)</title></rect><text x="481.259655912777" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling andr...</text></a><a href="slice_link" target="_blank"><rect x="409.613628263883" y="21.0" width="140.966006738833" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (3.2ms)</title></rect><text x="480.0966316333" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="545.760775166847" y="37.0" width="3.13690947804737" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (70.5μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="550.80898378207" y="21.0" width="2.42568805968264" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (54.5μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="550.959599563081" y="37.0" width="2.07116715753948" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (46.6μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="555.507386578666" y="5.0" width="145.349344096297" height="16.0" fill="hsl(90,45%,78%)"><title>JIT compiling iqo ilk.a(iqt, java.util.concurrent.Executor) (3.3ms)</title></rect><text x="628.182058626815" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling iqo ...</text></a><a href="slice_link" target="_blank"><rect x="555.801634479253" y="21.0" width="141.540579699057" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (3.2ms)</title></rect><text x="626.571924328781" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="693.436123340179" y="37.0" width="2.03642677804465" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (45.8μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="697.578502137178" y="21.0" width="2.88669867822864" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (64.9μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="697.722134256754" y="37.0" width="2.5368483776821" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (57.0μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="702.654556434301" y="5.0" width="17.4081773070681" height="16.0" fill="hsl(330,45%,78%)"><title>JIT compiling void ilj.&lt;init&gt;(iqt, java.lang.Object) (391.4μs)</title></rect><text x="711.358645087835" y="13.0" text-anchor="middle" font-size="11" fill="#333">JI</text></a><a href="slice_link" target="_blank"><rect x="702.939507613614" y="21.0" width="15.6010993083773" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (350.7μs)</title></rect><text x="710.740057267803" y="29.0" text-anchor="middle" font-size="11" fill="#333"></text></a><a href="slice_link" target="_blank"><rect x="721.06820306032" y="5.0" width="365.259148996817" height="16.0" fill="hsl(330,45%,78%)"><title>JIT compiling void ilj.run() (8.2ms)</title></rect><text x="903.697777558729" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling void ilj.run()</text></a><a href="slice_link" target="_blank"><rect x="721.353154239634" y="21.0" width="360.567729861656" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (8.1ms)</title></rect><text x="901.637019170462" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1073.0175164463" y="37.0" width="6.56570931492328" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (147.6μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1073.27004030596" y="53.0" width="5.26834963714319" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect all (118.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1082.16179817984" y="21.0" width="3.69062041780622" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (83.0μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1082.31704008052" y="37.0" width="3.32684727630977" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (74.8μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1092.76099455168" y="5.0" width="24.2009668234269" height="16.0" fill="hsl(210,45%,78%)"><title>JIT compiling void ili.&lt;init&gt;(java.util.concurrent.Executor, iqt) (544.1μs)</title></rect><text x="1104.86147796339" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT</text></a><a href="slice_link" target="_blank"><rect x="1093.20350069152" y="21.0" width="22.1228160044525" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (497.3μs)</title></rect><text x="1104.26490869374" y="29.0" text-anchor="middle" font-size="11" fill="#333">Co</text></a><a href="slice_link" target="_blank"><rect x="1118.1782749947" y="5.0" width="36.428824212968" height="16.0" fill="hsl(30,45%,78%)"><title>JIT compiling void ikb.&lt;init&gt;(ikv) (819.0μs)</title></rect><text x="1136.39268710119" y="13.0" text-anchor="middle" font-size="11" fill="#333">JI...</text></a><a href="slice_link" target="_blank"><rect x="1118.46553923385" y="21.0" width="34.1584225358929" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (767.9μs)</title></rect><text x="1135.5447505018" y="29.0" text-anchor="middle" font-size="11" fill="#333">C...</text></a><a href="slice_link" target="_blank"><rect x="1148.77578639141" y="37.0" width="2.78479059958247" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (62.6μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1155.6982851864" y="5.0" width="77.8758317451805" height="16.0" fill="hsl(120,45%,78%)"><title>JIT compiling iqo ikb.a(iqo) (1.8ms)</title></rect><text x="1194.63620105899" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT comp...</text></a><a href="slice_link" target="_blank"><rect x="1155.98328084763" y="21.0" width="74.2523790265256" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (1.7ms)</title></rect><text x="1193.1094703609" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1224.73102227815" y="37.0" width="2.50673411786392" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (56.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1230.49049679365" y="21.0" width="2.57625935877355" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (57.9μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1230.68741825335" y="37.0" width="2.15457075747906" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (48.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1474.69161129706" y="5.0" width="448.220287278473" height="16.0" fill="hsl(240,45%,78%)"><title>JIT compiling keu keu.a(int, java.lang.Object[]) (10.1ms)</title></rect><text x="1698.80175493629" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling keu keu.a(int, java.lang.Object[])</text></a><a href="slice_link" target="_blank"><rect x="1475.44455675635" y="21.0" width="440.934059823912" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (9.9ms)</title></rect><text x="1695.91158666831" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1910.99674908336" y="37.0" width="3.12298663709746" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (70.2μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1916.60565229978" y="21.0" width="5.90079357524499" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (132.7μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1916.74928441935" y="37.0" width="5.5463171550218" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (124.7μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1924.45484293345" y="5.0" width="75.545157066549" height="16.0" fill="hsl(180,45%,78%)"><title>JIT compiling void ilc.run() (1.7ms)</title></rect><text x="1962.22742146673" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT comp...</text></a><a href="slice_link" target="_blank"><rect x="1924.78151815369" y="21.0" width="72.5750547883808" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (1.6ms)</title></rect><text x="1961.06904554788" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1991.38394106608" y="37.0" width="4.42270385663584" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (99.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1993.64971662348" y="53.0" width="2.01792229933806" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect code (45.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1997.59286090094" y="21.0" width="2.031800658368" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (45.7μs)</title></rect></a></g></g><g transform="translate(0,0)"><text x="5" y="15" font-size="11" fill="#333">954</text><g transform="translate(0,20)"><a href="thread_state_link" target="_blank"><rect x="0.0" y="2.0" width="10.9731071476223" height="8.0" fill="#2f7d31"><title>Thread State: Running (322.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="10.9731071476223" y="2.0" width="4.04525100148913" height="8.0" fill="#99ba34"><title>Thread State: R (118.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="15.0183581491114" y="2.0" width="37.078195814918" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.1ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="52.0965539640294" y="2.0" width="7.63076017068043" height="8.0" fill="#99ba34"><title>Thread State: R (223.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="59.7273141347098" y="2.0" width="59.9048378711397" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.8ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="119.63215200585" y="2.0" width="22.013659001406" height="8.0" fill="#a35b58"><title>Thread State: D (645.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="142.936257595837" y="2.0" width="111.307348661356" height="8.0" fill="#99ba34"><title>Thread State: R+ (3.3ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="254.243606257192" y="2.0" width="7.25802507654838" height="8.0" fill="#2f7d31"><title>Thread State: Running (213.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="261.50163133374" y="2.0" width="52.9130472789435" height="8.0" fill="#99ba34"><title>Thread State: R (1.6ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="317.052349394932" y="2.0" width="2.90567743674673" height="8.0" fill="#2f7d31"><title>Thread State: Running (85.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="321.346113179306" y="2.0" width="17.9382470272272" height="8.0" fill="#99ba34"><title>Thread State: R (526.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="339.284360206533" y="2.0" width="27.7646919295201" height="8.0" fill="#2f7d31"><title>Thread State: Running (814.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="367.049052136053" y="2.0" width="21.7510029375424" height="8.0" fill="#99ba34"><title>Thread State: R (638.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="388.800055073595" y="2.0" width="9.18566907346685" height="8.0" fill="#2f7d31"><title>Thread State: Running (269.5μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="399.324087489995" y="2.0" width="37.2734412528152" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.1ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="436.597528742811" y="2.0" width="10.5364376074272" height="8.0" fill="#a35b58"><title>Thread State: D (309.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="447.133966350238" y="2.0" width="15.2580100812625" height="8.0" fill="#2f7d31"><title>Thread State: Running (447.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="462.3919764315" y="2.0" width="2.39270233965652" height="8.0" fill="#99ba34"><title>Thread State: R (70.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="464.784678771157" y="2.0" width="3.80917748982664" height="8.0" fill="#2f7d31"><title>Thread State: Running (111.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="469.332271848856" y="2.0" width="13.3214370732201" height="8.0" fill="#2f7d31"><title>Thread State: Running (390.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="484.178456852173" y="2.0" width="18.030536195638" height="8.0" fill="#2f7d31"><title>Thread State: Running (529.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="502.208993047811" y="2.0" width="5.94450619586468" height="8.0" fill="#99ba34"><title>Thread State: R (174.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="508.153499243675" y="2.0" width="35.5161982315446" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="545.080822034755" y="2.0" width="21.8699087383495" height="8.0" fill="#2f7d31"><title>Thread State: Running (641.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="567.490322502591" y="2.0" width="2.89858875615978" height="8.0" fill="#2f7d31"><title>Thread State: Running (85.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="570.38891125875" y="2.0" width="81.8457356559441" height="8.0" fill="#99ba34"><title>Thread State: R (2.4ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="652.234646914694" y="2.0" width="12.9451576387946" height="8.0" fill="#2f7d31"><title>Thread State: Running (379.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="665.179804553489" y="2.0" width="3.30506324347011" height="8.0" fill="#99ba34"><title>Thread State: R (97.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="668.484867796959" y="2.0" width="2.4175808821011" height="8.0" fill="#2f7d31"><title>Thread State: Running (70.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="671.820126093314" y="2.0" width="22.1609876849511" height="8.0" fill="#2f7d31"><title>Thread State: Running (650.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="693.981113778265" y="2.0" width="8.1224692260098" height="8.0" fill="#99ba34"><title>Thread State: R (238.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="702.593485809262" y="2.0" width="84.1585540182169" height="8.0" fill="#99ba34"><title>Thread State: R (2.5ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="786.752039827479" y="2.0" width="14.6047949812141" height="8.0" fill="#2f7d31"><title>Thread State: Running (428.5μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="802.620630684683" y="2.0" width="16.9637579276924" height="8.0" fill="#2f7d31"><title>Thread State: Running (497.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="821.316821251592" y="2.0" width="3.23939070745543" height="8.0" fill="#2f7d31"><title>Thread State: Running (95.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="824.556211959047" y="2.0" width="2.61459849014512" height="8.0" fill="#a35b58"><title>Thread State: D (76.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="827.170810449192" y="2.0" width="8.72592724116901" height="8.0" fill="#2f7d31"><title>Thread State: Running (256.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="837.700159376033" y="2.0" width="12.2706424167902" height="8.0" fill="#2f7d31"><title>Thread State: Running (360.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="850.659494376001" y="2.0" width="14.8355519824364" height="8.0" fill="#2f7d31"><title>Thread State: Running (435.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="865.495046358437" y="2.0" width="2.04658387792065" height="8.0" fill="#a35b58"><title>Thread State: D (60.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="867.541630236358" y="2.0" width="40.0850573510044" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.2ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="908.365103175235" y="2.0" width="16.0372196627049" height="8.0" fill="#2f7d31"><title>Thread State: Running (470.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="924.40232283794" y="2.0" width="3.6476373649125" height="8.0" fill="#99ba34"><title>Thread State: R (107.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="928.049960202852" y="2.0" width="20.6877349296951" height="8.0" fill="#2f7d31"><title>Thread State: Running (607.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="948.737695132547" y="2.0" width="169.744250287679" height="8.0" fill="#a0a0a0"><title>Thread State: S (5.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1119.66408514869" y="2.0" width="10.3287869785027" height="8.0" fill="#99ba34"><title>Thread State: R (303.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1129.99287212719" y="2.0" width="32.4809138127201" height="8.0" fill="#2f7d31"><title>Thread State: Running (953.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1163.14121248133" y="2.0" width="8.33898070566793" height="8.0" fill="#2f7d31"><title>Thread State: Running (244.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1171.48019318699" y="2.0" width="21.3888667840957" height="8.0" fill="#99ba34"><title>Thread State: R (627.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1192.86905997109" y="2.0" width="18.2400271550995" height="8.0" fill="#2f7d31"><title>Thread State: Running (535.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1211.55993402756" y="2.0" width="41.0062791055516" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.2ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1252.56621313311" y="2.0" width="15.5970398624114" height="8.0" fill="#99ba34"><title>Thread State: R+ (457.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1268.79867823371" y="2.0" width="111.360616006343" height="8.0" fill="#99ba34"><title>Thread State: R (3.3ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1380.15929424006" y="2.0" width="62.989777134268" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.8ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1444.15017710625" y="2.0" width="35.7434108924734" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1479.89358799873" y="2.0" width="3.90678316867771" height="8.0" fill="#99ba34"><title>Thread State: R (114.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1483.80037116741" y="2.0" width="6.9651398796049" height="8.0" fill="#2f7d31"><title>Thread State: Running (204.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1492.29911982784" y="2.0" width="21.5326511273473" height="8.0" fill="#2f7d31"><title>Thread State: Running (631.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1514.69975944494" y="2.0" width="3.65827038579292" height="8.0" fill="#2f7d31"><title>Thread State: Running (107.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1518.69351527159" y="2.0" width="11.1381916128299" height="8.0" fill="#2f7d31"><title>Thread State: Running (326.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1530.21865341992" y="2.0" width="4.8493391253761" height="8.0" fill="#2f7d31"><title>Thread State: Running (142.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1535.0679925453" y="2.0" width="464.932007454702" height="8.0" fill="#a0a0a0"><title>Thread State: S (13.6ms)</title></rect></a></g></g></g></svg>"
957,"<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 2060 156"><style>
rect { cursor: pointer; }
path { cursor: pointer; }
text { font-family: sans-serif; pointer-events: none; dominant-baseline: central; }
a { text-decoration: none !important; }
a:hover { text-decoration: none !important; }
</style><g transform="translate(50,5)"><g transform="translate(0,30)"><g transform="translate(0,0)"><a href="slice_link" target="_blank"><rect x="0.0" y="5.0" width="2000.0" height="16.0" fill="hsl(90,45%,78%)"><title>Background concurrent copying GC (72.6ms)</title></rect><text x="1000.0" y="13.0" text-anchor="middle" font-size="11" fill="#333">Background concurrent copying GC</text></a><a href="slice_link" target="_blank"><rect x="0.192314260736225" y="21.0" width="13.7924238569933" height="16.0" fill="hsl(180,45%,78%)"><title>InitializePhase (500.5μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="30.6777206629438" y="21.0" width="7.08707949218619" height="16.0" fill="hsl(60,45%,78%)"><title>GrayAllDirtyImmuneObjects (257.2μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="37.8868187174656" y="21.0" width="74.8278282400251" height="16.0" fill="hsl(210,45%,78%)"><title>FlipThreadRoots (2.7ms)</title></rect><text x="75.3007328374782" y="29.0" text-anchor="middle" font-size="11" fill="#333">FlipThre...</text></a><a href="slice_link" target="_blank"><rect x="37.9930200587577" y="37.0" width="74.6039346494703" height="16.0" fill="hsl(300,45%,78%)"><title>ThreadListFlip (2.7ms)</title></rect><text x="75.2949873834928" y="45.0" text-anchor="middle" font-size="11" fill="#333">ThreadLi...</text></a><a href="slice_link" target="_blank"><rect x="40.1558734777182" y="53.0" width="3.76888003369454" height="16.0" fill="hsl(120,45%,78%)"><title>(Paused)FlipCallback (136.8μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="40.5505599881817" y="69.0" width="3.26512634397983" height="16.0" fill="hsl(0,45%,78%)"><title>(Paused)GrayAllNewlyDirtyImmuneObjects (118.5μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="44.8906511281667" y="53.0" width="58.617078043698" height="16.0" fill="hsl(270,45%,78%)"><title>FlipOtherThreads (2.1ms)</title></rect><text x="74.1991901500157" y="61.0" text-anchor="middle" font-size="11" fill="#333">FlipO...</text></a><a href="slice_link" target="_blank"><rect x="103.657000750946" y="53.0" width="8.7949811351394" height="16.0" fill="hsl(270,45%,78%)"><title>ResumeOtherThreads (319.2μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="112.820848298783" y="21.0" width="1779.62411531548" height="16.0" fill="hsl(150,45%,78%)"><title>MarkingPhase (64.6ms)</title></rect><text x="1002.63290595652" y="29.0" text-anchor="middle" font-size="11" fill="#333">MarkingPhase</text></a><a href="slice_link" target="_blank"><rect x="112.909827056186" y="37.0" width="226.217234016929" height="16.0" fill="hsl(300,45%,78%)"><title>ScanImmuneSpaces (8.2ms)</title></rect><text x="226.018444064651" y="45.0" text-anchor="middle" font-size="11" fill="#333">ScanImmuneSpaces</text></a><a href="slice_link" target="_blank"><rect x="381.87351428519" y="37.0" width="180.2601880505" height="16.0" fill="hsl(180,45%,78%)"><title>VisitConcurrentRoots (6.5ms)</title></rect><text x="472.00360831044" y="45.0" text-anchor="middle" font-size="11" fill="#333">VisitConcurrentRoots</text></a><a href="slice_link" target="_blank"><rect x="562.28727267171" y="37.0" width="7.86785499923594" height="16.0" fill="hsl(240,45%,78%)"><title>VisitNonThreadRoots (285.5μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="570.308698006965" y="37.0" width="1322.030064266" height="16.0" fill="hsl(270,45%,78%)"><title>ProcessMarkStack (48.0ms)</title></rect><text x="1231.32373013997" y="45.0" text-anchor="middle" font-size="11" fill="#333">ProcessMarkStack</text></a><a href="slice_link" target="_blank"><rect x="1766.94110142227" y="53.0" width="27.9221347968653" height="16.0" fill="hsl(120,45%,78%)"><title>ProcessReferences (1.0ms)</title></rect><text x="1780.9021688207" y="61.0" text-anchor="middle" font-size="11" fill="#333">Pro</text></a><a href="slice_link" target="_blank"><rect x="1767.17935175875" y="69.0" width="27.510225702513" height="16.0" fill="hsl(120,45%,78%)"><title>ProcessReferences (998.3μs)</title></rect><text x="1780.93446461001" y="77.0" text-anchor="middle" font-size="11" fill="#333">Pro</text></a><a href="slice_link" target="_blank"><rect x="1770.95685686246" y="85.0" width="11.4888963728288" height="16.0" fill="hsl(330,45%,78%)"><title>EnqueueFinalizerReferences (416.9μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1795.01823947413" y="53.0" width="91.8595307870979" height="16.0" fill="hsl(90,45%,78%)"><title>SweepSystemWeaks (3.3ms)</title></rect><text x="1840.94800486768" y="61.0" text-anchor="middle" font-size="11" fill="#333">SweepSyste...</text></a><a href="slice_link" target="_blank"><rect x="1892.56265586352" y="21.0" width="98.0094261543895" height="16.0" fill="hsl(330,45%,78%)"><title>ReclaimPhase (3.6ms)</title></rect><text x="1941.56736894072" y="29.0" text-anchor="middle" font-size="11" fill="#333">ReclaimPhase</text></a><a href="slice_link" target="_blank"><rect x="1892.92289720621" y="37.0" width="93.1196452489415" height="16.0" fill="hsl(60,45%,78%)"><title>RecordFree (3.4ms)</title></rect><text x="1939.48271983068" y="45.0" text-anchor="middle" font-size="11" fill="#333">RecordFree</text></a><a href="slice_link" target="_blank"><rect x="1897.06491485341" y="53.0" width="88.8398469306345" height="16.0" fill="hsl(0,45%,78%)"><title>ClearFromSpace (3.2ms)</title></rect><text x="1941.48483831872" y="61.0" text-anchor="middle" font-size="11" fill="#333">ClearFromS...</text></a><a href="slice_link" target="_blank"><rect x="1986.77018973542" y="37.0" width="3.00965342360649" height="16.0" fill="hsl(270,45%,78%)"><title>Sweep (109.2μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1987.59830668106" y="53.0" width="2.07820097463016" height="16.0" fill="hsl(60,45%,78%)"><title>SweepLargeObjects (75.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1991.71163839253" y="21.0" width="6.33791087108922" height="16.0" fill="hsl(60,45%,78%)"><title>EmptyRBMarkBitStack (230.0μs)</title></rect></a></g></g><g transform="translate(0,0)"><text x="5" y="15" font-size="11" fill="#333">957</text><g transform="translate(0,20)"><a href="thread_state_link" target="_blank"><rect x="6.04681068966177" y="2.0" width="2.80830855651452" height="8.0" fill="#99ba34"><title>Thread State: R (1.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="11.0825607602904" y="2.0" width="3.25751171770488" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.2ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="16.6683369785606" y="2.0" width="4.92900872539526" height="8.0" fill="#99ba34"><title>Thread State: R (1.8ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="26.4211649972932" y="2.0" width="2.56328742146456" height="8.0" fill="#99ba34"><title>Thread State: R (948.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="30.9218105384246" y="2.0" width="2.58299996320965" height="8.0" fill="#2f7d31"><title>Thread State: Running (955.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="33.5048105016342" y="2.0" width="3.98567803137326" height="8.0" fill="#a0a0a0"><title>Thread State: S (1.5ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="37.7826835026986" y="2.0" width="2.71818674140147" height="8.0" fill="#99ba34"><title>Thread State: R (1.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="40.5008702441001" y="2.0" width="3.46057171602684" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.3ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="48.997894987943" y="2.0" width="2.27827342981146" height="8.0" fill="#2f7d31"><title>Thread State: Running (842.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="56.3378278681037" y="2.0" width="13.6314835219975" height="8.0" fill="#a0a0a0"><title>Thread State: S (5.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="70.3375474017043" y="2.0" width="8.0738688501292" height="8.0" fill="#99ba34"><title>Thread State: R (3.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="81.9380307956216" y="2.0" width="4.17296475188136" height="8.0" fill="#99ba34"><title>Thread State: R (1.5ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="86.110995547503" y="2.0" width="8.02852540785694" height="8.0" fill="#2f7d31"><title>Thread State: Running (3.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="96.2582501404343" y="2.0" width="2.88068881368884" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.1ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="107.042984162489" y="2.0" width="8.77119426903879" height="8.0" fill="#2f7d31"><title>Thread State: Running (3.2ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="115.814178431528" y="2.0" width="5.38018557407347" height="8.0" fill="#99ba34"><title>Thread State: R (2.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="121.548941018316" y="2.0" width="2.30714063708239" height="8.0" fill="#2f7d31"><title>Thread State: Running (853.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="123.954090109419" y="2.0" width="2.61454651883738" height="8.0" fill="#2f7d31"><title>Thread State: Running (967.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="135.884790345824" y="2.0" width="2.0107225169036" height="8.0" fill="#2f7d31"><title>Thread State: Running (743.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="144.687106692496" y="2.0" width="2.51442378589206" height="8.0" fill="#2f7d31"><title>Thread State: Running (930.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="148.500422325188" y="2.0" width="2.39557535494556" height="8.0" fill="#99ba34"><title>Thread State: R (886.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="152.164194630974" y="2.0" width="3.03079721002511" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.1ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="165.143414993459" y="2.0" width="5.72279338883095" height="8.0" fill="#2f7d31"><title>Thread State: Running (2.1ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="172.892560881888" y="2.0" width="2.41360079944077" height="8.0" fill="#2f7d31"><title>Thread State: Running (892.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="176.70362705158" y="2.0" width="7.78547525661334" height="8.0" fill="#a0a0a0"><title>Thread State: S (2.9ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="184.489102308193" y="2.0" width="3.04121990216985" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.1ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="189.244907789963" y="2.0" width="3.74290906300464" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.4ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="193.554321939292" y="2.0" width="2.83675398947299" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="197.023626546532" y="2.0" width="1802.97637345347" height="8.0" fill="#a0a0a0"><title>Thread State: S (666.9ms)</title></rect></a></g></g></g></svg>"
                """))

  def test_svg_combine_tracks(self):
    return DiffTestBlueprint(
        trace=DataPath('example_android_trace_30s.pb'),
        query="""
          INCLUDE PERFETTO MODULE export.to_svg;
          INCLUDE PERFETTO MODULE slices.with_context;

          -- Test the full modular pipeline
          WITH
            slice_data AS (
              SELECT
                upid,
                utid,
                2 AS tord,
                utid, ts, dur, depth, name, 'slice_link' AS href
              FROM thread_slice
              WHERE ts >= 86625813931 and ts < 86731829201 AND tid IN (5511, 5517)
            ),
            thread_state_data AS (
              SELECT
                upid,
                utid,
                1 AS tord,
                utid, ts, dur, state, io_wait, blocked_function, 'thread_state_link' AS href
              FROM thread_state
              JOIN thread USING(utid)
              WHERE ts >= 86625813931 and ts < 86731829201 AND tid IN (5511, 5517)
            ),
            slice_positions AS (
              SELECT * FROM _slice_intervals_to_positions!(
                slice_data, upid, utid, tord, 2000, 20
              )
            ),
            thread_state_positions AS (
              SELECT * FROM _thread_state_intervals_to_positions!(
                thread_state_data, upid, utid, tord, 2000, 20
              )
            ),
            slice_tracks AS (
              SELECT * FROM _generate_tracks_by_group!(
                slice_positions, upid, utid, tord, 2, 2, 0
              )
            ),
            thread_state_tracks AS (
              SELECT * FROM _generate_tracks_by_group_with_label!(
                thread_state_positions, upid, utid, tord,
                1, 2, 0, utid, 10
              )
            ),
            all_tracks AS (
              SELECT * FROM slice_tracks
              UNION ALL
              SELECT * FROM thread_state_tracks
            )
          SELECT * FROM _combine_track_svgs!(
            all_tracks, upid, 2000, 50
          );
        """,
        out=Csv("""
"svg_group_key","svg"
31,"<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 2060 239"><style>
rect { cursor: pointer; }
path { cursor: pointer; }
text { font-family: sans-serif; pointer-events: none; dominant-baseline: central; }
a { text-decoration: none !important; }
a:hover { text-decoration: none !important; }
</style><g transform="translate(50,5)"><g transform="translate(0,129)"><g transform="translate(0,0)"><a href="slice_link" target="_blank"><rect x="0.0" y="5.0" width="1805.85769557053" height="16.0" fill="hsl(90,45%,78%)"><title>Background concurrent copying GC (72.6ms)</title></rect><text x="902.928847785265" y="13.0" text-anchor="middle" font-size="11" fill="#333">Background concurrent copying GC</text></a><a href="slice_link" target="_blank"><rect x="34.2091015708104" y="21.0" width="67.5642047350396" height="16.0" fill="hsl(210,45%,78%)"><title>FlipThreadRoots (2.7ms)</title></rect><text x="67.9912039383302" y="29.0" text-anchor="middle" font-size="11" fill="#333">FlipTh...</text></a><a href="slice_link" target="_blank"><rect x="34.3049938255365" y="37.0" width="67.3620447532935" height="16.0" fill="hsl(300,45%,78%)"><title>ThreadListFlip (2.7ms)</title></rect><text x="67.9860162021833" y="45.0" text-anchor="middle" font-size="11" fill="#333">Thread...</text></a><a href="slice_link" target="_blank"><rect x="40.5330638994859" y="53.0" width="52.9270507385352" height="16.0" fill="hsl(270,45%,78%)"><title>FlipOtherThreads (2.1ms)</title></rect><text x="66.9965892687535" y="61.0" text-anchor="middle" font-size="11" fill="#333">Flip...</text></a><a href="slice_link" target="_blank"><rect x="101.869198560576" y="21.0" width="1606.87395193268" height="16.0" fill="hsl(150,45%,78%)"><title>MarkingPhase (64.6ms)</title></rect><text x="905.306174526914" y="29.0" text-anchor="middle" font-size="11" fill="#333">MarkingPhase</text></a><a href="slice_link" target="_blank"><rect x="101.949540047476" y="37.0" width="204.258066460075" height="16.0" fill="hsl(300,45%,78%)"><title>ScanImmuneSpaces (8.2ms)</title></rect><text x="204.078573277513" y="45.0" text-anchor="middle" font-size="11" fill="#333">ScanImmuneSpaces</text></a><a href="slice_link" target="_blank"><rect x="344.804612253236" y="37.0" width="162.762123897993" height="16.0" fill="hsl(180,45%,78%)"><title>VisitConcurrentRoots (6.5ms)</title></rect><text x="426.185674202233" y="45.0" text-anchor="middle" font-size="11" fill="#333">VisitConcurrentRoots</text></a><a href="slice_link" target="_blank"><rect x="514.948175573344" y="37.0" width="1193.69908266518" height="16.0" fill="hsl(270,45%,78%)"><title>ProcessMarkStack (48.0ms)</title></rect><text x="1111.79771690593" y="45.0" text-anchor="middle" font-size="11" fill="#333">ProcessMarkStack</text></a><a href="slice_link" target="_blank"><rect x="1595.42209281164" y="53.0" width="25.2117009998385" height="16.0" fill="hsl(120,45%,78%)"><title>ProcessReferences (1.0ms)</title></rect><text x="1608.02794331155" y="61.0" text-anchor="middle" font-size="11" fill="#333">Pro</text></a><a href="slice_link" target="_blank"><rect x="1595.63721591344" y="69.0" width="24.8397763958826" height="16.0" fill="hsl(120,45%,78%)"><title>ProcessReferences (998.3μs)</title></rect><text x="1608.05710411138" y="77.0" text-anchor="middle" font-size="11" fill="#333">Pro</text></a><a href="slice_link" target="_blank"><rect x="1620.77375072191" y="53.0" width="82.9426202916894" height="16.0" fill="hsl(90,45%,78%)"><title>SweepSystemWeaks (3.3ms)</title></rect><text x="1662.24506086776" y="61.0" text-anchor="middle" font-size="11" fill="#333">SweepSyst...</text></a><a href="slice_link" target="_blank"><rect x="1708.84941822027" y="21.0" width="88.4955382296779" height="16.0" fill="hsl(330,45%,78%)"><title>ReclaimPhase (3.6ms)</title></rect><text x="1753.09718733511" y="29.0" text-anchor="middle" font-size="11" fill="#333">ReclaimPhase</text></a><a href="slice_link" target="_blank"><rect x="1709.17469052075" y="37.0" width="84.0804139907994" height="16.0" fill="hsl(60,45%,78%)"><title>RecordFree (3.4ms)</title></rect><text x="1751.21489751615" y="45.0" text-anchor="middle" font-size="11" fill="#333">RecordFree</text></a><a href="slice_link" target="_blank"><rect x="1712.91463774244" y="53.0" width="80.2160606264971" height="16.0" fill="hsl(0,45%,78%)"><title>ClearFromSpace (3.2ms)</title></rect><text x="1753.02266805569" y="61.0" text-anchor="middle" font-size="11" fill="#333">ClearFro...</text></a></g></g><g transform="translate(0,60)"><g transform="translate(0,0)"><a href="slice_link" target="_blank"><rect x="881.288013618691" y="5.0" width="188.812247655358" height="16.0" fill="hsl(210,45%,78%)"><title>JIT compiling void ilm.&lt;init&gt;(ilk, iqt, java.util.concurrent.Executor) (7.6ms)</title></rect><text x="975.694137446369" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling void ilm.&lt;i...</text></a><a href="slice_link" target="_blank"><rect x="881.648268926469" y="21.0" width="186.098651110226" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (7.5ms)</title></rect><text x="974.697594481582" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="966.410329053874" y="37.0" width="99.7011727132593" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (4.0ms)</title></rect><text x="1016.2609154105" y="45.0" text-anchor="middle" font-size="11" fill="#333">ScopedCodeC...</text></a><a href="slice_link" target="_blank"><rect x="967.167141381846" y="53.0" width="98.8342111866178" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect code (4.0ms)</title></rect><text x="1016.58424697515" y="61.0" text-anchor="middle" font-size="11" fill="#333">mprotect code</text></a><a href="slice_link" target="_blank"><rect x="1109.99965004552" y="5.0" width="80.9676727777079" height="16.0" fill="hsl(30,45%,78%)"><title>JIT compiling android.content.SharedPreferences got.f(java.lang.String) (3.3ms)</title></rect><text x="1150.48348643437" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT comp...</text></a><a href="slice_link" target="_blank"><rect x="1110.40785148066" y="21.0" width="78.8501807055204" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (3.2ms)</title></rect><text x="1149.83294183342" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1192.01439956315" y="5.0" width="81.3020267265946" height="16.0" fill="hsl(90,45%,78%)"><title>JIT compiling iqo ilk.a(iqt, java.util.concurrent.Executor) (3.3ms)</title></rect><text x="1232.66541292644" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compi...</text></a><a href="slice_link" target="_blank"><rect x="1192.17898888982" y="21.0" width="79.1715715343469" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (3.2ms)</title></rect><text x="1231.764774657" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1284.6218344997" y="5.0" width="204.309894059088" height="16.0" fill="hsl(330,45%,78%)"><title>JIT compiling void ilj.run() (8.2ms)</title></rect><text x="1386.77678152924" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling void ilj.run()</text></a><a href="slice_link" target="_blank"><rect x="1284.78122364961" y="21.0" width="201.685720649266" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (8.1ms)</title></rect><text x="1385.62408397425" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1506.74773319257" y="5.0" width="20.3766811484125" height="16.0" fill="hsl(30,45%,78%)"><title>JIT compiling void ikb.&lt;init&gt;(ikv) (819.0μs)</title></rect><text x="1516.93607376677" y="13.0" text-anchor="middle" font-size="11" fill="#333">JI</text></a><a href="slice_link" target="_blank"><rect x="1527.73477575787" y="5.0" width="43.5603132113738" height="16.0" fill="hsl(120,45%,78%)"><title>JIT compiling iqo ikb.a(iqo) (1.8ms)</title></rect><text x="1549.51493236355" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT...</text></a><a href="slice_link" target="_blank"><rect x="1527.89418978901" y="21.0" width="41.5335132171512" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (1.7ms)</title></rect><text x="1548.66094639759" y="29.0" text-anchor="middle" font-size="11" fill="#333">Co...</text></a><a href="slice_link" target="_blank"><rect x="1706.16560450568" y="5.0" width="250.714703958851" height="16.0" fill="hsl(240,45%,78%)"><title>JIT compiling keu keu.a(int, java.lang.Object[]) (10.1ms)</title></rect><text x="1831.52295648511" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling keu keu.a(int, java.l...</text></a><a href="slice_link" target="_blank"><rect x="1706.58676906088" y="21.0" width="246.639108964392" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (9.9ms)</title></rect><text x="1829.90632354308" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1957.7433636383" y="5.0" width="42.2566363617035" height="16.0" fill="hsl(180,45%,78%)"><title>JIT compiling void ilc.run() (1.7ms)</title></rect><text x="1978.87168181915" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT...</text></a><a href="slice_link" target="_blank"><rect x="1957.92609138057" y="21.0" width="40.5952918520209" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (1.6ms)</title></rect><text x="1978.22373730658" y="29.0" text-anchor="middle" font-size="11" fill="#333">Co...</text></a></g></g><g transform="translate(0,0)"><text x="5" y="15" font-size="11" fill="#333">954</text><g transform="translate(0,20)"><a href="thread_state_link" target="_blank"><rect x="217.484863658851" y="2.0" width="36.8844150680739" height="8.0" fill="#a0a0a0"><title>Thread State: S (13.6ms)</title></rect></a></g></g><g transform="translate(0,30)"><text x="5" y="15" font-size="11" fill="#333">957</text><g transform="translate(0,20)"><a href="thread_state_link" target="_blank"><rect x="197.023626546532" y="2.0" width="1802.97637345347" height="8.0" fill="#a0a0a0"><title>Thread State: S (666.9ms)</title></rect></a></g></g></g></svg>"
                """))

  def test_svg_timeline_with_counters(self):
    return DiffTestBlueprint(
        trace=DataPath('example_android_trace_30s.pb'),
        query="""
            INCLUDE PERFETTO MODULE export.to_svg;
            INCLUDE PERFETTO MODULE slices.with_context;

            -- Test the svg_timeline_with_counters convenience macro
            WITH
              slice_data AS (
                SELECT
                  upid,
                  1 AS tord,
                  utid, ts, dur, depth, name, 'slice_link' AS href
                FROM thread_slice
                WHERE ts >= 86625813931 and ts < 86731829201 AND tid IN (5511)
              ),
              thread_state_data AS (
                SELECT
                  upid,
                  2 AS tord,
                  utid, ts, dur, state, io_wait, blocked_function, 'thread_state_link' AS href
                FROM thread_state
                JOIN thread USING(utid)
                WHERE ts >= 86625813931 and ts < 86731829201 AND tid IN (5511)
              ),
              counter_data AS (
                SELECT
                  upid,
                  3 AS tord,
                  ts,
                  lead(ts, 1, ts + 1000000) OVER (PARTITION BY track_id ORDER BY ts) - ts AS dur,
                  process_counter_track.name,
                  value,
                  'counter_link' AS href
                FROM counter
                JOIN process_counter_track ON counter.track_id = process_counter_track.id
                JOIN process USING(upid)
                WHERE ts >= 86625813931 and ts < 86731829201 AND pid = 5506
                AND process_counter_track.name LIKE '%heap size%'
              )
            SELECT *
            FROM _svg_timeline_with_counters!(
              slice_data,
              thread_state_data,
              counter_data,
              upid,
              upid,
              tord,
              2000,
              50,
              1);
          """,
        out=Csv("""
"svg_group_key","svg"
31,"<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 2060 173"><style>
rect { cursor: pointer; }
path { cursor: pointer; }
text { font-family: sans-serif; pointer-events: none; dominant-baseline: central; }
a { text-decoration: none !important; }
a:hover { text-decoration: none !important; }
</style><g transform="translate(50,5)"><g transform="translate(0,0)"><g transform="translate(0,0)"><a href="slice_link" target="_blank"><rect x="0.0" y="5.0" width="337.55291791609" height="16.0" fill="hsl(210,45%,78%)"><title>JIT compiling void ilm.&lt;init&gt;(ilk, iqt, java.util.concurrent.Executor) (7.6ms)</title></rect><text x="168.776458958045" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling void ilm.&lt;init&gt;(ilk, iqt, java.uti...</text></a><a href="slice_link" target="_blank"><rect x="0.644053719213396" y="21.0" width="332.701630760564" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (7.5ms)</title></rect><text x="166.994869099496" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="152.179142570069" y="37.0" width="178.242789792147" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (4.0ms)</title></rect><text x="241.300537466143" y="45.0" text-anchor="middle" font-size="11" fill="#333">ScopedCodeCacheWrite</text></a><a href="slice_link" target="_blank"><rect x="153.532149129729" y="53.0" width="176.69286177279" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect code (4.0ms)</title></rect><text x="241.878580016124" y="61.0" text-anchor="middle" font-size="11" fill="#333">mprotect code</text></a><a href="slice_link" target="_blank"><rect x="333.595895279596" y="21.0" width="3.5585535974219" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (80.0μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="333.732588219657" y="37.0" width="3.22031307798695" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (72.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="408.883857884892" y="5.0" width="144.75159605577" height="16.0" fill="hsl(30,45%,78%)"><title>JIT compiling android.content.SharedPreferences got.f(java.lang.String) (3.3ms)</title></rect><text x="481.259655912777" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling andr...</text></a><a href="slice_link" target="_blank"><rect x="409.613628263883" y="21.0" width="140.966006738833" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (3.2ms)</title></rect><text x="480.0966316333" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="545.760775166847" y="37.0" width="3.13690947804737" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (70.5μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="550.80898378207" y="21.0" width="2.42568805968264" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (54.5μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="550.959599563081" y="37.0" width="2.07116715753948" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (46.6μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="555.507386578666" y="5.0" width="145.349344096297" height="16.0" fill="hsl(90,45%,78%)"><title>JIT compiling iqo ilk.a(iqt, java.util.concurrent.Executor) (3.3ms)</title></rect><text x="628.182058626815" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling iqo ...</text></a><a href="slice_link" target="_blank"><rect x="555.801634479253" y="21.0" width="141.540579699057" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (3.2ms)</title></rect><text x="626.571924328781" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="693.436123340179" y="37.0" width="2.03642677804465" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (45.8μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="697.578502137178" y="21.0" width="2.88669867822864" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (64.9μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="697.722134256754" y="37.0" width="2.5368483776821" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (57.0μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="702.654556434301" y="5.0" width="17.4081773070681" height="16.0" fill="hsl(330,45%,78%)"><title>JIT compiling void ilj.&lt;init&gt;(iqt, java.lang.Object) (391.4μs)</title></rect><text x="711.358645087835" y="13.0" text-anchor="middle" font-size="11" fill="#333">JI</text></a><a href="slice_link" target="_blank"><rect x="702.939507613614" y="21.0" width="15.6010993083773" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (350.7μs)</title></rect><text x="710.740057267803" y="29.0" text-anchor="middle" font-size="11" fill="#333"></text></a><a href="slice_link" target="_blank"><rect x="721.06820306032" y="5.0" width="365.259148996817" height="16.0" fill="hsl(330,45%,78%)"><title>JIT compiling void ilj.run() (8.2ms)</title></rect><text x="903.697777558729" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling void ilj.run()</text></a><a href="slice_link" target="_blank"><rect x="721.353154239634" y="21.0" width="360.567729861656" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (8.1ms)</title></rect><text x="901.637019170462" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1073.0175164463" y="37.0" width="6.56570931492328" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (147.6μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1073.27004030596" y="53.0" width="5.26834963714319" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect all (118.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1082.16179817984" y="21.0" width="3.69062041780622" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (83.0μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1082.31704008052" y="37.0" width="3.32684727630977" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (74.8μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1092.76099455168" y="5.0" width="24.2009668234269" height="16.0" fill="hsl(210,45%,78%)"><title>JIT compiling void ili.&lt;init&gt;(java.util.concurrent.Executor, iqt) (544.1μs)</title></rect><text x="1104.86147796339" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT</text></a><a href="slice_link" target="_blank"><rect x="1093.20350069152" y="21.0" width="22.1228160044525" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (497.3μs)</title></rect><text x="1104.26490869374" y="29.0" text-anchor="middle" font-size="11" fill="#333">Co</text></a><a href="slice_link" target="_blank"><rect x="1118.1782749947" y="5.0" width="36.428824212968" height="16.0" fill="hsl(30,45%,78%)"><title>JIT compiling void ikb.&lt;init&gt;(ikv) (819.0μs)</title></rect><text x="1136.39268710119" y="13.0" text-anchor="middle" font-size="11" fill="#333">JI...</text></a><a href="slice_link" target="_blank"><rect x="1118.46553923385" y="21.0" width="34.1584225358929" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (767.9μs)</title></rect><text x="1135.5447505018" y="29.0" text-anchor="middle" font-size="11" fill="#333">C...</text></a><a href="slice_link" target="_blank"><rect x="1148.77578639141" y="37.0" width="2.78479059958247" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (62.6μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1155.6982851864" y="5.0" width="77.8758317451805" height="16.0" fill="hsl(120,45%,78%)"><title>JIT compiling iqo ikb.a(iqo) (1.8ms)</title></rect><text x="1194.63620105899" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT comp...</text></a><a href="slice_link" target="_blank"><rect x="1155.98328084763" y="21.0" width="74.2523790265256" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (1.7ms)</title></rect><text x="1193.1094703609" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1224.73102227815" y="37.0" width="2.50673411786392" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (56.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1230.49049679365" y="21.0" width="2.57625935877355" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (57.9μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1230.68741825335" y="37.0" width="2.15457075747906" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (48.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1474.69161129706" y="5.0" width="448.220287278473" height="16.0" fill="hsl(240,45%,78%)"><title>JIT compiling keu keu.a(int, java.lang.Object[]) (10.1ms)</title></rect><text x="1698.80175493629" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT compiling keu keu.a(int, java.lang.Object[])</text></a><a href="slice_link" target="_blank"><rect x="1475.44455675635" y="21.0" width="440.934059823912" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (9.9ms)</title></rect><text x="1695.91158666831" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1910.99674908336" y="37.0" width="3.12298663709746" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (70.2μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1916.60565229978" y="21.0" width="5.90079357524499" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (132.7μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1916.74928441935" y="37.0" width="5.5463171550218" height="16.0" fill="hsl(30,45%,78%)"><title>virtual void art::MemMapArenaPool::TrimMaps() (124.7μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1924.45484293345" y="5.0" width="75.545157066549" height="16.0" fill="hsl(180,45%,78%)"><title>JIT compiling void ilc.run() (1.7ms)</title></rect><text x="1962.22742146673" y="13.0" text-anchor="middle" font-size="11" fill="#333">JIT comp...</text></a><a href="slice_link" target="_blank"><rect x="1924.78151815369" y="21.0" width="72.5750547883808" height="16.0" fill="hsl(210,45%,78%)"><title>Compiling (1.6ms)</title></rect><text x="1961.06904554788" y="29.0" text-anchor="middle" font-size="11" fill="#333">Compiling</text></a><a href="slice_link" target="_blank"><rect x="1991.38394106608" y="37.0" width="4.42270385663584" height="16.0" fill="hsl(330,45%,78%)"><title>ScopedCodeCacheWrite (99.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1993.64971662348" y="53.0" width="2.01792229933806" height="16.0" fill="hsl(0,45%,78%)"><title>mprotect code (45.4μs)</title></rect></a><a href="slice_link" target="_blank"><rect x="1997.59286090094" y="21.0" width="2.031800658368" height="16.0" fill="hsl(60,45%,78%)"><title>TrimMaps (45.7μs)</title></rect></a></g></g><g transform="translate(0,69)"><text x="5" y="15" font-size="11" fill="#333">31</text><g transform="translate(0,20)"><a href="thread_state_link" target="_blank"><rect x="0.0" y="2.0" width="10.9731071476223" height="8.0" fill="#2f7d31"><title>Thread State: Running (322.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="10.9731071476223" y="2.0" width="4.04525100148913" height="8.0" fill="#99ba34"><title>Thread State: R (118.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="15.0183581491114" y="2.0" width="37.078195814918" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.1ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="52.0965539640294" y="2.0" width="7.63076017068043" height="8.0" fill="#99ba34"><title>Thread State: R (223.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="59.7273141347098" y="2.0" width="59.9048378711397" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.8ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="119.63215200585" y="2.0" width="22.013659001406" height="8.0" fill="#a35b58"><title>Thread State: D (645.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="141.645811007256" y="2.0" width="1.29044658858099" height="8.0" fill="#2f7d31"><title>Thread State: Running (37.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="142.936257595837" y="2.0" width="111.307348661356" height="8.0" fill="#99ba34"><title>Thread State: R+ (3.3ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="254.243606257192" y="2.0" width="7.25802507654838" height="8.0" fill="#2f7d31"><title>Thread State: Running (213.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="261.50163133374" y="2.0" width="52.9130472789435" height="8.0" fill="#99ba34"><title>Thread State: R (1.6ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="314.414678612684" y="2.0" width="1.62770419958317" height="8.0" fill="#2f7d31"><title>Thread State: Running (47.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="316.042382812267" y="2.0" width="1.00996658266467" height="8.0" fill="#99ba34"><title>Thread State: R (29.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="317.052349394932" y="2.0" width="2.90567743674673" height="8.0" fill="#2f7d31"><title>Thread State: Running (85.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="319.958026831678" y="2.0" width="0.724204146503261" height="8.0" fill="#99ba34"><title>Thread State: R (21.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="320.682230978182" y="2.0" width="0.663882201123931" height="8.0" fill="#2f7d31"><title>Thread State: Running (19.5μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="321.346113179306" y="2.0" width="17.9382470272272" height="8.0" fill="#99ba34"><title>Thread State: R (526.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="339.284360206533" y="2.0" width="27.7646919295201" height="8.0" fill="#2f7d31"><title>Thread State: Running (814.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="367.049052136053" y="2.0" width="21.7510029375424" height="8.0" fill="#99ba34"><title>Thread State: R (638.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="388.800055073595" y="2.0" width="9.18566907346685" height="8.0" fill="#2f7d31"><title>Thread State: Running (269.5μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="397.985724147062" y="2.0" width="1.33836334293316" height="8.0" fill="#99ba34"><title>Thread State: R (39.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="399.324087489995" y="2.0" width="37.2734412528152" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.1ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="436.597528742811" y="2.0" width="10.5364376074272" height="8.0" fill="#a35b58"><title>Thread State: D (309.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="447.133966350238" y="2.0" width="15.2580100812625" height="8.0" fill="#2f7d31"><title>Thread State: Running (447.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="462.3919764315" y="2.0" width="2.39270233965652" height="8.0" fill="#99ba34"><title>Thread State: R (70.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="464.784678771157" y="2.0" width="3.80917748982664" height="8.0" fill="#2f7d31"><title>Thread State: Running (111.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="468.593856260983" y="2.0" width="0.73841558787229" height="8.0" fill="#99ba34"><title>Thread State: R (21.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="469.332271848856" y="2.0" width="13.3214370732201" height="8.0" fill="#2f7d31"><title>Thread State: Running (390.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="482.653708922076" y="2.0" width="1.52474793009675" height="8.0" fill="#a35b58"><title>Thread State: D (44.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="484.178456852173" y="2.0" width="18.030536195638" height="8.0" fill="#2f7d31"><title>Thread State: Running (529.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="502.208993047811" y="2.0" width="5.94450619586468" height="8.0" fill="#99ba34"><title>Thread State: R (174.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="508.153499243675" y="2.0" width="35.5161982315446" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="543.66969747522" y="2.0" width="1.41112455953478" height="8.0" fill="#a35b58"><title>Thread State: D (41.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="545.080822034755" y="2.0" width="21.8699087383495" height="8.0" fill="#2f7d31"><title>Thread State: Running (641.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="566.950730773104" y="2.0" width="0.539591729486406" height="8.0" fill="#a35b58"><title>Thread State: D (15.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="567.490322502591" y="2.0" width="2.89858875615978" height="8.0" fill="#2f7d31"><title>Thread State: Running (85.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="570.38891125875" y="2.0" width="81.8457356559441" height="8.0" fill="#99ba34"><title>Thread State: R (2.4ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="652.234646914694" y="2.0" width="12.9451576387946" height="8.0" fill="#2f7d31"><title>Thread State: Running (379.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="665.179804553489" y="2.0" width="3.30506324347011" height="8.0" fill="#99ba34"><title>Thread State: R (97.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="668.484867796959" y="2.0" width="2.4175808821011" height="8.0" fill="#2f7d31"><title>Thread State: Running (70.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="670.90244867906" y="2.0" width="0.917677414253803" height="8.0" fill="#99ba34"><title>Thread State: R (26.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="671.820126093314" y="2.0" width="22.1609876849511" height="8.0" fill="#2f7d31"><title>Thread State: Running (650.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="693.981113778265" y="2.0" width="8.1224692260098" height="8.0" fill="#99ba34"><title>Thread State: R (238.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="702.103583004275" y="2.0" width="0.4899028049875" height="8.0" fill="#2f7d31"><title>Thread State: Running (14.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="702.593485809262" y="2.0" width="84.1585540182169" height="8.0" fill="#99ba34"><title>Thread State: R (2.5ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="786.752039827479" y="2.0" width="14.6047949812141" height="8.0" fill="#2f7d31"><title>Thread State: Running (428.5μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="801.356834808693" y="2.0" width="1.26379587598967" height="8.0" fill="#99ba34"><title>Thread State: R+ (37.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="802.620630684683" y="2.0" width="16.9637579276924" height="8.0" fill="#2f7d31"><title>Thread State: Running (497.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="819.584388612376" y="2.0" width="1.73243263921632" height="8.0" fill="#99ba34"><title>Thread State: R (50.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="821.316821251592" y="2.0" width="3.23939070745543" height="8.0" fill="#2f7d31"><title>Thread State: Running (95.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="824.556211959047" y="2.0" width="2.61459849014512" height="8.0" fill="#a35b58"><title>Thread State: D (76.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="827.170810449192" y="2.0" width="8.72592724116901" height="8.0" fill="#2f7d31"><title>Thread State: Running (256.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="835.896737690361" y="2.0" width="1.8034216856712" height="8.0" fill="#99ba34"><title>Thread State: R (52.9μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="837.700159376033" y="2.0" width="12.2706424167902" height="8.0" fill="#2f7d31"><title>Thread State: Running (360.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="849.970801792823" y="2.0" width="0.688692583178254" height="8.0" fill="#a35b58"><title>Thread State: D (20.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="850.659494376001" y="2.0" width="14.8355519824364" height="8.0" fill="#2f7d31"><title>Thread State: Running (435.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="865.495046358437" y="2.0" width="2.04658387792065" height="8.0" fill="#a35b58"><title>Thread State: D (60.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="867.541630236358" y="2.0" width="40.0850573510044" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.2ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="907.626687587363" y="2.0" width="0.73841558787229" height="8.0" fill="#99ba34"><title>Thread State: R (21.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="908.365103175235" y="2.0" width="16.0372196627049" height="8.0" fill="#2f7d31"><title>Thread State: Running (470.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="924.40232283794" y="2.0" width="3.6476373649125" height="8.0" fill="#99ba34"><title>Thread State: R (107.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="928.049960202852" y="2.0" width="20.6877349296951" height="8.0" fill="#2f7d31"><title>Thread State: Running (607.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="948.737695132547" y="2.0" width="169.744250287679" height="8.0" fill="#a0a0a0"><title>Thread State: S (5.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1118.48194542023" y="2.0" width="1.18213972845923" height="8.0" fill="#2f7d31"><title>Thread State: Running (34.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1119.66408514869" y="2.0" width="10.3287869785027" height="8.0" fill="#99ba34"><title>Thread State: R (303.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1129.99287212719" y="2.0" width="32.4809138127201" height="8.0" fill="#2f7d31"><title>Thread State: Running (953.1μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1162.47378593991" y="2.0" width="0.667426541417406" height="8.0" fill="#99ba34"><title>Thread State: R+ (19.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1163.14121248133" y="2.0" width="8.33898070566793" height="8.0" fill="#2f7d31"><title>Thread State: Running (244.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1171.48019318699" y="2.0" width="21.3888667840957" height="8.0" fill="#99ba34"><title>Thread State: R (627.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1192.86905997109" y="2.0" width="18.2400271550995" height="8.0" fill="#2f7d31"><title>Thread State: Running (535.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1211.10908712619" y="2.0" width="0.450846901369019" height="8.0" fill="#a35b58"><title>Thread State: D (13.2μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1211.55993402756" y="2.0" width="41.0062791055516" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.2ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1252.56621313311" y="2.0" width="15.5970398624114" height="8.0" fill="#99ba34"><title>Thread State: R+ (457.7μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1268.16325299552" y="2.0" width="0.635425238190744" height="8.0" fill="#2f7d31"><title>Thread State: Running (18.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1268.79867823371" y="2.0" width="111.360616006343" height="8.0" fill="#99ba34"><title>Thread State: R (3.3ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1380.15929424006" y="2.0" width="62.989777134268" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.8ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1443.14907137432" y="2.0" width="1.00110573193098" height="8.0" fill="#99ba34"><title>Thread State: R (29.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1444.15017710625" y="2.0" width="35.7434108924734" height="8.0" fill="#2f7d31"><title>Thread State: Running (1.0ms)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1479.89358799873" y="2.0" width="3.90678316867771" height="8.0" fill="#99ba34"><title>Thread State: R (114.6μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1483.80037116741" y="2.0" width="6.9651398796049" height="8.0" fill="#2f7d31"><title>Thread State: Running (204.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1490.76551104701" y="2.0" width="1.53360878083044" height="8.0" fill="#99ba34"><title>Thread State: R+ (45.0μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1492.29911982784" y="2.0" width="21.5326511273473" height="8.0" fill="#2f7d31"><title>Thread State: Running (631.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1513.83177095519" y="2.0" width="0.867988489754897" height="8.0" fill="#99ba34"><title>Thread State: R (25.5μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1514.69975944494" y="2.0" width="3.65827038579292" height="8.0" fill="#2f7d31"><title>Thread State: Running (107.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1518.35802983074" y="2.0" width="0.33548544085544" height="8.0" fill="#99ba34"><title>Thread State: R (9.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1518.69351527159" y="2.0" width="11.1381916128299" height="8.0" fill="#2f7d31"><title>Thread State: Running (326.8μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1529.83170688442" y="2.0" width="0.386946535501084" height="8.0" fill="#a35b58"><title>Thread State: D (11.4μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1530.21865341992" y="2.0" width="4.8493391253761" height="8.0" fill="#2f7d31"><title>Thread State: Running (142.3μs)</title></rect></a><a href="thread_state_link" target="_blank"><rect x="1535.0679925453" y="2.0" width="464.932007454702" height="8.0" fill="#a0a0a0"><title>Thread State: S (13.6ms)</title></rect></a></g></g><g transform="translate(0,99)"><text x="5" y="15" font-size="11" fill="#333">31</text><text x="5" y="30" font-size="9" fill="#000">5.90K</text><g transform="translate(0,20)"><path d="M0,34 L0.0,21.0722891566265 L80.9138181648717,21.0722891566265 L80.9138181648717,21.0722891566265 L104.750886322443,21.0722891566265 L104.750886322443,20.6729776247849 L135.398009696324,20.6729776247849 L135.73627605623,19.8743545611015 L164.487813765669,19.8743545611015 L164.487813765669,19.4750430292599 L179.692031296721,19.4750430292599 L179.692031296721,19.0757314974182 L184.316736816334,19.0757314974182 L184.316736816334,18.6764199655766 L188.222779499289,18.6764199655766 L188.79845972881,18.6764199655766 L189.965187019302,18.6764199655766 L190.418324733279,18.6764199655766 L191.473979487682,18.6764199655766 L192.689184597322,18.6514629948365 L196.3029143637,18.6514629948365 L196.3029143637,18.6514629948365 L199.646045747253,18.6514629948365 L199.646045747253,18.6514629948365 L213.94011550686,18.6514629948365 L213.94011550686,18.2521514629948 L216.693498780108,18.2521514629948 L217.512989598554,17.4535283993115 L234.328126530441,17.4535283993115 L234.328126530441,17.0542168674699 L240.234430694616,17.0542168674699 L240.234430694616,17.0542168674699 L249.088127442648,17.0542168674699 L249.46979836156,17.0542168674699 L251.9385153712,17.0542168674699 L251.9385153712,17.0542168674699 L254.928038071901,17.0542168674699 L254.928038071901,17.0542168674699 L277.948164295348,17.0542168674699 L277.948164295348,16.6549053356282 L292.169468309225,16.6549053356282 L292.169468309225,16.6299483648881 L295.021130679939,16.6299483648881 L295.021130679939,16.6299483648881 L297.985140028904,16.6299483648881 L298.161282640746,16.6299483648881 L319.025444862724,16.6299483648881 L319.179897449315,16.6299483648881 L325.624874003317,16.6299483648881 L325.624874003317,16.2306368330465 L330.749945122398,16.2306368330465 L330.749945122398,15.8313253012048 L334.892151741309,15.8313253012048 L335.056799865193,15.8063683304647 L351.743022070728,15.8063683304647 L351.743022070728,15.8063683304647 L357.824194404584,15.8063683304647 L357.824194404584,15.8063683304647 L400.758753846043,15.8063683304647 L400.758753846043,15.4070567986231 L443.487809488935,15.4070567986231 L443.487809488935,15.0077452667814 L529.167992321388,15.0077452667814 L529.167992321388,15.0077452667814 L530.401088638298,15.0077452667814 L530.872117559545,14.6084337349398 L539.210106385162,14.6084337349398 L539.210106385162,14.6084337349398 L550.992054576123,14.6084337349398 L551.602218269518,14.6084337349398 L567.042575228023,14.6084337349398 L567.042575228023,14.2091222030981 L611.941638244666,14.2091222030981 L611.941638244666,13.8098106712565 L637.881094815911,13.8098106712565 L637.881094815911,13.4104991394148 L665.434019672314,13.4104991394148 L665.963746458506,13.4104991394148 L669.636198598332,13.4104991394148 L669.636198598332,13.0111876075731 L677.32701589086,13.0111876075731 L677.32701589086,12.6118760757315 L715.746667906634,12.6118760757315 L715.746667906634,12.2125645438898 L724.158696920143,12.2125645438898 L724.158696920143,11.8132530120482 L737.626879141991,11.8132530120482 L737.626879141991,11.4139414802065 L752.538783756465,11.4139414802065 L752.538783756465,11.4139414802065 L783.414375396325,11.4139414802065 L783.414375396325,11.0146299483649 L792.314050096189,11.0146299483649 L792.314050096189,11.0146299483649 L808.501132127411,11.0146299483649 L808.501132127411,10.6153184165232 L835.376249821838,10.6153184165232 L835.376249821838,10.2160068846816 L886.153530258094,10.2160068846816 L886.153530258094,9.81669535283993 L961.070411814294,9.81669535283993 L961.070411814295,9.41738382099828 L973.477841513093,9.41738382099828 L973.477841513093,9.01807228915663 L998.832672251182,9.01807228915663 L998.832672251182,8.61876075731497 L1005.68865640232,8.61876075731497 L1005.68865640232,8.21944922547332 L1019.03559505929,8.21944922547332 L1019.03559505929,7.82013769363167 L1069.93286912676,7.82013769363167 L1069.93286912676,7.42082616179002 L1074.6405359294,7.42082616179002 L1074.6405359294,7.42082616179002 L1127.60189162019,7.42082616179002 L1127.60189162019,7.02151462994836 L1168.08560075077,7.02151462994836 L1168.08560075077,6.62220309810671 L1181.32783908246,6.62220309810671 L1181.32783908246,6.22289156626506 L1312.3079460084,6.22289156626506 L1312.3079460084,6.22289156626506 L1313.78737729006,6.22289156626506 L1313.78737729006,6.22289156626506 L1315.17874952005,6.22289156626506 L1315.17874952005,6.22289156626506 L1417.43155764613,6.22289156626506 L1417.43155764613,6.22289156626506 L1418.43613668005,6.22289156626506 L1419.99727931107,5.7986230636833 L1530.28939113538,5.7986230636833 L1530.28939113538,5.0 L1532.71470358601,5.0 L1532.71470358601,5.0 L1850.13432681655,5.0 L1850.13432681655,5.0 L1901.9072651916,5.0 L1901.9072651916,34.0 L1927.35527725851,34.0 L1927.35527725851,33.6006884681583 L1975.49149689126,33.6006884681583 L1975.49149689126,33.2013769363167 L2000.0,33.2013769363167 L2000,34 L0,34 Z" fill="steelblue" fill-opacity="0.7" stroke="steelblue" stroke-width="1"/></g></g></g></svg>"
                  """))
