#include <stdio.h>
#include <sys/types.h>
#include <grp.h>
#include <pwd.h>

#include <wchar.h>
#include <sys/cygwin.h>
#include <windows.h>
#include <sddl.h>

extern int windows_only;

static void
print_windows_mapping (const char *cygname, unsigned long id,
		       const char *sid_string)
{
  PSID sid;
  WCHAR name[NAME_MAX + 1];
  DWORD nlen = NAME_MAX + 1;
  /* DNLEN is moot.  Domain names can now be twice as long. */
  WCHAR dom[32];
  DWORD dlen = 32;
  SID_NAME_USE use;

  if (!cygname || !sid_string)
    return;
  if (!ConvertStringSidToSidA (sid_string, &sid))
    return;
  if (LookupAccountSidW (NULL, sid, name, &nlen, dom, &dlen, &use))
    printf ("%s:%lu:%ls\\%ls:%s\n", cygname, id, dom, name, sid_string);
  LocalFree (sid);
}

static PSID
get_sid_from_special_string (const char *key)
{
  PSID sid = NULL;

  if (!strncmp (key, "U-", 2))
    {
      WCHAR wnam[4096];

      /* DNLEN is moot.  Domain names can now be twice as long. */
      WCHAR dom[32];
      DWORD dlen = 32;
      DWORD slen = SECURITY_MAX_SID_SIZE;
      SID_NAME_USE use;

      memset (wnam, 0, sizeof wnam);
      mbstowcs (wnam, key + 2, sizeof wnam / sizeof (wchar_t) - 1);
      sid = (PSID) LocalAlloc (LMEM_FIXED, SECURITY_MAX_SID_SIZE);
      if (sid && !LookupAccountNameW (NULL, wnam, sid, &slen,
				      dom, &dlen, &use))
	{
	  LocalFree (sid);
	  sid = NULL;
	}
      if (sid
	  && (use == SidTypeDomain || use == SidTypeComputer)
	  && wcslen (wnam) < sizeof wnam / sizeof (wchar_t) / 2 - 2)
	{
	  wcpcpy (wnam + wcslen (wnam) + 1, wnam);
	  wnam[wcslen (wnam)] = L'\\';
	  dlen = 32;
	  slen = SECURITY_MAX_SID_SIZE;
	  if (!LookupAccountNameW (NULL, wnam, sid, &slen,
				   dom, &dlen, &use))
	    {
	      LocalFree (sid);
	      sid = NULL;
	    }
	}
    }
  else if (!strncmp (key, "S-1-", 4))
    ConvertStringSidToSidA (key, &sid);
  return sid;
}

int
putgrent (const struct group *grp, FILE *stream)
{
  unsigned int i = 0;
  int ret;

  if (windows_only)
    {
      print_windows_mapping (grp->gr_name, (unsigned long) grp->gr_gid,
			     grp->gr_passwd);
      return 0;
    }
  ret = fprintf (stream,
	   "%s:%s:%lu:", grp->gr_name ? grp->gr_name : "",
	   grp->gr_passwd ? grp->gr_passwd : "",
	   (unsigned long int) grp->gr_gid);
  if (ret < 0)
    return -1;

  while (grp->gr_mem[i] != NULL)
    {
      fputs_unlocked (grp->gr_mem[i], stream);
      ++i;
      if (grp->gr_mem[i] != NULL)
	putchar_unlocked (',');
    }
  putchar_unlocked ('\n');
  return 0;
}

int
putpwent (const struct passwd *pwd, FILE *stream)
{
  int ret;

  if (windows_only)
    {
      char *c = strstr (pwd->pw_gecos, "S-1-");
      print_windows_mapping (pwd->pw_name, (unsigned long) pwd->pw_uid, c);
      return 0;
    }

  ret = fprintf (stream, "%s:", pwd->pw_name ? pwd->pw_name : "");
  ret = fprintf (stream, "%s:", pwd->pw_passwd ? pwd->pw_passwd : "");
  ret = fprintf (stream, "%lu:", (unsigned long int) pwd->pw_uid);
  ret = fprintf (stream, "%lu:", (unsigned long int) pwd->pw_gid);
  ret = fprintf (stream, "%s:", pwd->pw_gecos ? pwd->pw_gecos : "");
  ret = fprintf (stream, "%s:", pwd->pw_dir ? pwd->pw_dir : "");
  ret = fprintf (stream, "%s\n", pwd->pw_shell ? pwd->pw_shell : "");
  return ret > 0 ? 0 : -1;
}

struct group *
getgrsid (const char *key)
{
  struct group *grp;

  PSID sid = get_sid_from_special_string (key);

  if (!sid)
    return NULL;
  grp = (struct group *) cygwin_internal (CW_GETGRSID, 0, sid);
  LocalFree (sid);
  return grp;
}

struct passwd *
getpwsid (const char *key)
{
  struct passwd *pwd;

  PSID sid = get_sid_from_special_string (key);

  if (!sid)
    return NULL;
  pwd = (struct passwd *) cygwin_internal (CW_GETPWSID, 0, sid);
  LocalFree (sid);
  return pwd;
}
