/* configure.c -- MiKTeX configuration utility
   Time-stamp: "97/08/27 14:39:53 cschenk"

   Copyright (C) 1995, 1996, 1997
		Christian Schenk <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#include <stdio.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <getopt.h>

#include "miktex.h"
#include "configure.rc"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#define E_UNKNOWN_ROOT "the TeXMF root directories are still undefined"

static int	verbose_p;
static int	debug_p;
static int	quiet_p;
static int	personal_p;
static int	print_only_p;
static char *	update_root;
static char *	texmf_root_directories;

#define OPT_LIST_MODES 1

static struct option long_options[] =

{
  "config",		required_argument,	0,	'c',
  "debug",		no_argument,		0,	'd',
  "help",		no_argument,		0,	'h',
  "list-modes",		no_argument,		0,	OPT_LIST_MODES,
  "update-fndb",	optional_argument,      0,	'u',
  "personal",		required_argument,	0,	'p',
  "print-only",		no_argument,		0,	'n',
  "root-directory",	required_argument,	0,	'r',
  "root-directories",	required_argument,	0,	'r',
  "quiet",		no_argument,		0,	'q',
  "verbose",		no_argument,		0,	'v',
  "version",		no_argument,		0,	'V',
  0,			no_argument,		0,	0,
};

static void
show_version (void)

{
  printf ("This is configure version %s (%s %s)\n",
	  VER_FILEVERSION_STR, VER_PRODUCTNAME_STR,
	  VER_PRODUCTVERSION_STR);
}

static void
usage (void)

{
  /*     01234567890123456789012345678901234567890123456789012345678901234567890123456789 */
  puts   ("Usage: configure [OPTION]...");
  puts   ("");
  puts   ("This program configures MiKTeX for your system.");
  puts   ("");
  puts   ("Options:");
  puts   ("--config, c                     Specify name of master configuration file");
  puts   ("--debug, -d                     Print debugging information.");
  puts   ("--update-fndb, -u               Refresh all file name data bases.");
  puts   ("--update-fndb ROOT, -u ROOT     Refresh file name data base of");
  puts   ("                                TeXMF ROOT.");
  puts   ("--help, -h                      Print this help screen and exit.");
  puts   ("--list-modes                    List known METAFONT modes.");
  puts   ("--personal CFGFILE, -p CFGFILE  Process a persolan configuration file.");
  puts   ("--root-directories ROOTS, -r ROOTS");
  puts   ("                                Specify the TEXMF root directories.");
  puts   ("--verbose, -v                   Print information on what is being done.");
  puts   ("--version, -V                   Print the version number and exit.");
  puts   ("--print-only, -n                Print what would be done.");
  puts   ("");
  puts   ("Email problem reports to cschenk@berlin.snafu.de.");
}

static void
warning (const char *	fmt,
	                ...)

{
  va_list marker;
  va_start (marker, fmt);
  vfprintf (stdout, fmt, marker);
  va_end (marker);
  fputc ('\n', stdout);
}

static void
my_fatal_error (const char *msg)

{
  fprintf (stderr, "configure: %s\n", msg);
  exit (1);
}

static void
fatal_file_error (const char *filename)

{
  fprintf (stderr, "configure: ");
  perror (filename);
  exit (1);
}

/* Get a configuration parameter. */
char *
my_get_cfg_value (const char *	section,
		  const char *	key,
		  char *	buf,
		  size_t	bufsize,
		  char *	defval)

{
  HKEY hKey;
  LONG res;
  char path[_MAX_PATH];

  sprintf (path, "Software\\MiK\\MiKTeX\\CurrentVersion\\%s", section);

  /* First try current user tree. */
  res = RegOpenKeyEx (HKEY_CURRENT_USER,
		      path,
		      0,
		      KEY_READ,
		      &hKey);
  if (res == ERROR_SUCCESS)
    {
      DWORD type;
      DWORD len = bufsize;
      res = RegQueryValueEx (hKey,
			     key,
			     0,
			     &type,
			     buf,
			     &len);
      RegCloseKey (hKey);
      if (res == ERROR_SUCCESS)
	return (buf);
    }

  /* Then try local machine tree. */
  res = RegOpenKeyEx (HKEY_LOCAL_MACHINE,
		      path,
		      0,
		      KEY_READ,
		      &hKey);
  if (res == ERROR_SUCCESS)
    {
      DWORD type;
      DWORD len = bufsize;
      res = RegQueryValueEx (hKey,
			     key,
			     0,
			     &type,
			     buf,
			     &len);
      RegCloseKey (hKey);
      if (res == ERROR_SUCCESS)
	return (buf);
      else
	return (defval);
    }
  else
    return (defval);
}

/* Change a configuration parameter. */
int
my_set_cfg_value (const char *	section,
		  const char *	key,
		  const char *	value)

{
  HKEY hKey;
  LONG res;
  char path[_MAX_PATH];
  DWORD disp;
  DWORD len;

  sprintf (path, "Software\\MiK\\MiKTeX\\CurrentVersion\\%s", section);
  res = RegCreateKeyEx ((personal_p
			 ? HKEY_CURRENT_USER
			 : HKEY_LOCAL_MACHINE),
			path,
			0,
			"",
			REG_OPTION_NON_VOLATILE,
			KEY_ALL_ACCESS,
			0,
			&hKey,
			&disp);
  if (res != ERROR_SUCCESS)
    fatal_error ("cannot create Registry key '%s'!", path);

  len = strlen (value) + 1;
  res = RegSetValueEx (hKey,
		       key,
		       0,
		       REG_SZ,
		       value,
		       len);
  RegCloseKey (hKey);
  if (res != ERROR_SUCCESS)
    {
      fprintf (stderr, "Cannot set value '%s'!\n", value);
      exit (3);
    }
  return (0);
}

static void
parse_line (char *	a_line,
	    char **	key,
	    char **	value,
	    char **	comment)

{
  if (key)
    *key = 0;
  if (value)
    *value = 0;
  if (isalnum (*a_line))
    {
      if (key)
	*key = a_line;
      if ((a_line = strchr (a_line, '=')) != 0)
	{
	  char *cmnt;

	  if (value)
	    *value = a_line + 1;
	  if ((cmnt = strchr (a_line, ';')) != 0
	      && isspace (cmnt[-1]))
	    {
	      *cmnt = 0;
	      if (comment)
		*comment = cmnt+1;
	      cmnt++;
	      cmnt += strlen (cmnt) - 1;
	      if (*cmnt == '\n')
		*cmnt = 0;
	    }
	  do
	    {
	      a_line--;
	    }
	  while ((*a_line == ' ') || (*a_line == '\t'));
	  *++a_line = 0;
	  if (value)
	    {
	      while ((*(*value) == ' ') || (*(*value) == '\t'))
		(*value)++;
	      a_line = &((*value)[ strlen (*value) - 1 ]);
	      while ((*a_line == ' ') || (*a_line == '\t') ||
		     (*a_line == '\n'))
		a_line--;
	      *++a_line = 0;
	    }
	}
    }
}

static int
fnamecmp (const char **s1,
	  const char **s2)

{
  return (_stricmp (*s1, *s2));
}

/* Recurse into DIRNAME and write found files to FNDB_STREAM. */
static void
dump_listing (FILE *		fndb_stream,
	      const char *	parent_root,
	      const char *	dirname)

{
  WIN32_FIND_DATA data;
  HANDLE hnd;
  char *search_pattern;
  char *dir_path;
  if (verbose_p)
    printf ("  %s...\n", dirname);
  search_pattern = (char *) _alloca (strlen (parent_root)
				     + strlen (dirname) + 4);
  sprintf (search_pattern, "%s\\%s\\*", parent_root, dirname);
  hnd = FindFirstFile (search_pattern, &data);
  dir_path = search_pattern;
  dir_path[strlen (dir_path) - 2] = 0;
  if (hnd != INVALID_HANDLE_VALUE)
    {
      static char **subdirs = 0;
      static char **files = 0;
      static size_t max_subdirs = 0;
      static size_t max_files = 0;
      static size_t subdir_base = 0;
      size_t nsubdirs = 0;
      size_t nfiles = 0;
      size_t i;
      BOOL go_on;
      size_t this_subdir_base = subdir_base;
      do
	{
	  if (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
	    {
	      if (strcmp (data.cFileName, ".") != 0
		  && strcmp (data.cFileName, "..") != 0)
		{
		  size_t l;
		  if (subdirs == 0)
 		    {
#ifdef DEBUG
		      max_subdirs = 10;
#else
		      max_subdirs = 50;
#endif
		      subdirs = (char **) malloc (max_subdirs
						  * sizeof (char *));
		    }
		  if (this_subdir_base + nsubdirs == max_subdirs)
		    {
#ifdef DEBUG
		      max_subdirs += 10;
#else
		      max_subdirs += 50;
#endif
		      subdirs = (char **) realloc (subdirs,
						   max_subdirs
						   * sizeof (char *));
		    }
		  subdirs[this_subdir_base + nsubdirs++]
		    = strdup (data.cFileName);
		}
	    }
	  else
	    {
	      if (files == 0)
		{
#if defined (DEBUG)
		  max_files = 10;
#else
		  max_files = 200;
#endif
		  files = (char **) malloc (max_files
					    * sizeof (char *));
		}
	      if (nfiles == max_files)
		{
		  max_files *= 2;
		  files = (char **) realloc (files,
					     max_files
					     * sizeof (char *));
		}
	      files[nfiles++] = strdup (data.cFileName);
	    }
	  go_on = FindNextFile (hnd, &data);
	}
      while (go_on);
      FindClose (hnd);
      fprintf (fndb_stream,
	       "%d %d %s\n", (int) nsubdirs, (int) nfiles,
	       (strcmp (dirname, ".") == 0 ? parent_root : dirname));
      qsort (files, nfiles, sizeof (files[0]), fnamecmp);
      for (i = 0; i < nfiles; i++)
	{
	  fprintf (fndb_stream, "%s\n", files[i]);
	  free (files[i]);
	}
      qsort (&subdirs[subdir_base], nsubdirs, sizeof (subdirs[0]), fnamecmp);
      subdir_base += nsubdirs;
      for (i = this_subdir_base; i < this_subdir_base + nsubdirs; i++)
	{
	  dump_listing (fndb_stream, dir_path, subdirs[i]);
	  free (subdirs[i]);
	}
      subdir_base = this_subdir_base;
    }
}

/* Create a filename database. */
static void
make_fndb (size_t root_idx)

{
  FILE *fndb_stream;
  char full_fndb_filename[_MAX_PATH];
  const char *root;
  const char *local_root;
  
  local_root = get_root_directory (0);
  xassert (local_root != 0);

  root = get_root_directory (root_idx);
  xassert (root != 0);

  if (root_idx == 0)
    sprintf (full_fndb_filename,
	     "%s\\miktex\\config\\texmf.fndb", local_root);
  else
    sprintf (full_fndb_filename,
	     "%s\\miktex\\config\\texmf%u.fndb", local_root, root_idx);
  
  fndb_stream = fopen (full_fndb_filename, "wb");
  if (fndb_stream == 0)
    fatal_file_error (full_fndb_filename);
  else
    {
      if (verbose_p)
	printf ("creating file name data base for %s...\n", root);
       dump_listing (fndb_stream, root, ".");
      fclose (fndb_stream);
    }
}

/* List all known MF modes. */
static void
list_modes ()

{
  mf_mode modes[1000];
  int n = get_metafont_modes (modes, 1000);
  int i;
  for (i = 0; i < n; i++)
    printf ("%-8s  %5dx%-5d  %s\n",
	    modes[i].mnemonic,
	    modes[i].hor_res,
	    modes[i].vert_res,
	    modes[i].description);
}

/* Remove obsolete keys left by a previous 1.05 installation. */
static void
remove_105_keys ()

{
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\BibTeX");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\dvips");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\fndb");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\LaTeX");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\MakeIndex");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\MakeTeXPK");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\METAFONT");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\MetaPost");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\MiKTeX");
  RegDeleteKey (HKEY_LOCAL_MACHINE, "SOFTWARE\\MiK\\MiKTeX\\TeX");
}

static void
find_config_file (char *config_filename)

{
  char *my_root_directories;
  char *root;
  int found;

  my_root_directories = strdup (texmf_root_directories);

  found = 0;
  root = strtok (my_root_directories, ";");
  while (root && ! found)
    {
      sprintf (config_filename,
	       "%s\\miktex\\config\\miktex.environment", root);
      if (_access (config_filename, 0) == 0)
	found = 1;
      else
	root = strtok (0, ";");
    }

  free (my_root_directories);

  if (! found)
    my_fatal_error ("configuration file not found");
}

static int
check_texmf_roots (const char *roots)

{
  char *my_roots = strdup (roots);
  int ok = 1;
  char *root = strtok (my_roots, ";");
  while (root && *root)
    {
      if (access_directory (root, 0) != 0)
	{
	  warning ("%s: directory root does not exist", root);
	  ok = 0;
	}
      else
	{
	  char subdir[_MAX_PATH];
	  _makepath (subdir, 0, root, "tex", 0);
	  if (access_directory (subdir, 0) != 0)
	    {
	      warning ("%s: this doesn't seem to be a TEXMF root directory",
		       root);
	      ok = 0;
	    }
	}
      root = strtok (0, ";");
    }

  free (my_roots);

  if (! ok)
    printf ("Run configure once again and specify \
valid TeXMF root directories!\n");

  return (ok);
}

int
main (int argc,
      char **argv)

{
  char config_line[2000];
  char section[_MAX_PATH];
  char config_filename[_MAX_PATH];
  char *key;
  char *value;
  FILE *config_file;
  int c;
  int option_index;
  int make_fndb_flag;
  int alt_config;
  int ret;

  alt_config = 0;
  *config_filename = 0;
  print_only_p = 0;
  verbose_p = 0;
  quiet_p = 0;
  debug_p = 0;
  texmf_root_directories = 0;
  make_fndb_flag = 0;
  update_root = 0;
  ret = 0;

  remove_105_keys ();

  option_index = 0;
  while ((c=getopt_long (argc, argv, "c:dghnpr:u::vV",
			 long_options, &option_index))
	 != EOF)
    {
      switch (c)
	{
	case OPT_LIST_MODES:
	  list_modes ();
	  exit (0);
	  break;
	case 'h':
	  usage ();
	  exit (0);
	  break;
	case 'n':
	  print_only_p = 1;
	  break;
	case 'c':
	  alt_config = 1;
	  strcpy (config_filename, optarg);
	  break;
	case 'd':
	  debug_p = 1;
	  break;
	case 'p':
	  personal_p = 1;
	  strcpy (config_filename, optarg);
	  break;
	case 'r':
	  texmf_root_directories = strdup (optarg);
	  break;
	case 'v':
	  verbose_p = 1;
	  break;
	case 'q':
	  quiet_p = 1;
	  break;
	case 'V':
	  show_version ();
	  exit (0);
	  break;
	case 'u':
	  make_fndb_flag++;
	  if (optarg)
	    update_root = strdup (optarg);
	  else
	    update_root = 0;
	  break;
	default:
	  usage ();
	  exit (1);
	  break;
	}
    }

  if (argc - optind > 0)
    {
      usage ();
      exit (1);
    }

  if (texmf_root_directories && personal_p)
    my_fatal_error ("you cannot use --root-directories together \
with --personal");

  if (alt_config && personal_p)
    my_fatal_error ("you cannot use --config together with --personal");

  if (texmf_root_directories == 0)
    {
      size_t buf_size = (_MAX_PATH + 1) * MAX_TEXMF_ROOT_DIRECTORIES;
      texmf_root_directories = (char *) malloc (buf_size);
      if (get_cfg_value ("MiKTeX", "TeXMF Root Directories",
			 texmf_root_directories, buf_size, 0) == 0
	  && get_cfg_value ("MiKTeX", "Root Directory",
			    texmf_root_directories, buf_size, 0) == 0)
	my_fatal_error (E_UNKNOWN_ROOT);
    }

  check_texmf_roots (texmf_root_directories);

  if (! personal_p)
    my_set_cfg_value ("MiKTeX", "TeXMF Root Directories",
		      texmf_root_directories);

  if (verbose_p)
    printf ("opening MiKTeX configuration file...\n");

  if (*config_filename == 0)
    find_config_file (config_filename);

  config_file = fopen (config_filename, "r");
  if (config_file == 0)
    fatal_file_error (config_filename);

  while (fgets (config_line, sizeof (config_line) - 1, config_file))
    {
      if (*config_line == '[')
	{
	  const char *sec = strtok (config_line + 1, "]");
	  if (sec)
	    {
	      strcpy (section, sec);
	      if (verbose_p)
		printf ("reading section [%s]...", sec);
	    }
	}
      else
	{
	  parse_line (config_line, &key, &value, 0);
	  if (key && value)
	    {
	      if (print_only_p)
		printf ("%s: [%s]%s=%s\n",
			(personal_p
			 ? "HKEY_CURRENT_USER"
			 : "HKEY_LOCAL_MACHINE"),
			section, key, value);
	      else
		my_set_cfg_value (section, key, value);
	    }
	}
    }

  fclose (config_file);

  if (make_fndb_flag)
    {
      if (update_root == 0)
	{
	  size_t i;
	  for (i = 0; i < get_number_of_texmf_roots (); i++)
	    make_fndb (i);
	}
      else
	{
	  size_t i;
	  size_t num_of_roots;

	  num_of_roots = get_number_of_texmf_roots ();
	  for (i = 0; i < num_of_roots; i++)
	    {
	      const char *texmf_root = get_root_directory (i);
	      if (_stricmp (update_root, texmf_root) == 0)
		{
		  make_fndb (i);
		  break;
		}
	    }
	  if (i == num_of_roots)
	    {
	      fprintf (stderr,
		       "`%s' is not one of the TeXMF root directories\n",
		       update_root);
	      exit (4);
	    }
	}
    }

  if (verbose_p)
    printf ("MiKTeX has been successfully configured for your system!");

  return (ret);
}

/* configure.c ends here */
