;;; t2latexinfo.el

;;; Latexinfo mode utilities to convert texinfo files to latexinfo.

;;; Written for Latexinfo by Mike Clarkson
;;; Please send bug reports to:  mike@apl.ists.ca

;;; Copyleft  (C) 1991 Michael E. Clarkson
;; It's not perfect, but it does 98%

(require 'latexinfo)
(require 'latexnfo-upd)

(defun tex-to-latexinfo (arg)
  (interactive "FNew File Name: ")
  (let ((case-fold-search nil)
	(alist latex-alist)
	(input-buffer (current-buffer)))
    (find-file arg)
    (erase-buffer)
    (insert-buffer-substring input-buffer)
    (goto-char (point-min))
    ;; Input the @input files
    (t2l-input-files)
    ;; Fix the header for LaTeX
    (goto-char (point-min))
    (if (re-search-forward "^\\\\input texinfo\\(.*\\)" nil t)
	(replace-match "\\\\documentstyle[twoside,latexinfo,elisp]{report}
\\\\pagestyle{headings}

\\\\begin{document}

\\\\comment Delete any of these if you don't want that particular index.
\\\\newindex{cp}
\\\\newindex{vr}
\\\\newindex{fn}
\\\\newindex{tp}
\\\\newindex{pg}
\\\\newindex{ky}

" t nil))
    ;; Assume that any \ from here on in the file are to print out as \
    (quietly-replace-string "\\" "\\back " nil)
    (goto-char (point-min))
    (if (yes-or-no-p
	 "Would you like all occurences of `@@' replaced by `@'? ")
	(quietly-replace-string "@@" "@" nil)
      (if (yes-or-no-p 
	   "Would you like all occurences of `@@' replaced by `\\back'? ")
	  ;; Assume that your working on the
	  ;; TeXinfo manual.
	  (let ((case-fold-search nil)
		(alist texinfo-alist))
	    (while alist
	      (goto-char (point-min))
	      (quietly-replace-regexp (car (car alist)) (cdr (car alist)) nil)
	      (message "%s" (car (car alist)))
	      (setq alist (cdr alist)))
	    (message "Quietly Replacing @@ by \\back ")
	    (goto-char (point-min))
	    (quietly-replace-string "@@" "\\back " nil)
	    )
	(error 
	 "You'd better make up your mind between one of these two options.")))
    (setq alist latex-delete-list)
    (while alist
      (goto-char (point-min))
      (delete-matching-lines (car alist))
      (setq alist (cdr alist)))
    (if (re-search-forward "^@appendix"  nil t nil)
	(progn
	  (beginning-of-line 1)
	  (insert "\\appendix\n")))
    (setq alist latex-alist)
    (while alist
      (goto-char (point-min))
      (quietly-replace-regexp (car (car alist)) (cdr (car alist)) nil)
      (message "%s" (car (car alist)))
      (setq alist (cdr alist)))
    )
  ;; protect commands in section headings.
  (goto-char (point-min))
  (while (and nil (re-search-forward
	  "^\\\\chapter\\|\\\\section\\|\\\\subsection\\|\\\\subsubsection"
	  nil t))
    (narrow-to-region (point) (save-excursion (forward-line 1) (point)))
    (quietly-replace-string "\\" "\\protect\\")
    (widen)
    (forward-line 1)
    )
  )

(defun t2l-input-files ()
  (if (and (save-excursion (re-search-forward "^@input" nil t))
	   (yes-or-no-p
	    "Would you like to do the @input files now, to do it all at once?"))
      (while (re-search-forward "^@input" nil t)
	(save-excursion
	  (skip-chars-forward " 	{")
	  (let ((file-name
		 (buffer-substring 
		  (point)
		  (progn
		    (skip-chars-forward "^ 	\n")
		    (point)))))
	    (setq file-name
		  (expand-file-name
		   (if (file-readable-p (expand-file-name file-name))
		       file-name
		     (concat file-name ".texinfo"))))
	    (beginning-of-line 1)
	    (if (file-readable-p file-name)
		(progn
		  (delete-region (point) (progn (forward-line 1) (point)))
		  (message "Inserting file %s..." file-name) (sit-for 1)
		  (insert-file file-name)
		  (message "Inserting file %s...done" file-name))
	      (error "I can't find the file %s" file-name))
	    )))))

(setq texinfo-alist
      '(
	("@@@@" . "\\\\back back")
	;; begin end environments
	;; change the name of the group command
	("@@end[ \t]+group" . "\\\\back end\\\\{same\\\\}")
	("@@group" . "\\\\back begin\\\\{same\\\\}")
	("@@end[ \t]+quotation" . "\\\\back end\\\\{quote\\\\}")
	("@@quotation" . "\\\\back begin\\\\{quote\\\\}")
	;; We turn the @table into a description environment
	;; There may be a lost of information here.
	("@@table[ \t]+.*" . "\\\\back begin\\\\{description\\\\}")
	("@@end[ \t]table" . "\\\\back end\\\\{description\\\\}")

	("@@end[ \t]+\\([a-z]+\\)" . "\\\\back end\\\\{\\1\\\\}")

	("@@itemize[ \t]+\\([@a-z]+\\)" . "\\\\back begin\\\\{itemize\\\\}")

	("{@@bye}" . "{\\\\back end\\\\{document\\\\}}")
	("{@@sc}" . "{\\\\back s}")
	("{@@page}" . "{\\\\back clearpage}")
	("{@@itemize}" . "{\\\\back begin\\\\{itemize\\\\}}")

	;; This is broken:
	("@@def\\([a-z]+\\)[ \t]+\\([a-z---@{}]+\\)[ \t]+\\([a-z---@{}]+\\)[ \t]+\\([a-z---@{}]+\\)" . "\\\\back def\\1\\\\{\\2\\\\}\\\\{\\3\\\\}\\\\{\\4\\\\}")

	("@@display" . "\\\\back begin\\\\{display\\\\}")
	("@@enumerate" . "\\\\back begin\\\\{enumerate\\\\}")
	("@@example" . "\\\\back begin\\\\{example\\\\}")
	("@@smallexample" . "\\\\back begin\\\\{smallexample\\\\}")
	("@@flushleft" . "\\\\back begin\\\\{flushleft\\\\}")
	("@@flushright" . "\\\\back begin\\\\{flushright\\\\}")
	("@@format" . "\\\\back begin\\\\{format\\\\}")
	("@@ignore" . "\\\\back begin\\\\{ignore\\\\}")
	("@@ifinfo" . "\\\\back begin\\\\{ifinfo\\\\}")
	("@@iftex" . "\\\\back begin\\\\{iftex\\\\}")
	("@@lisp" . "\\\\back begin\\\\{lisp\\\\}")
	("@@menu" . "\\\\back begin\\\\{menu\\\\}")
	("@@tex" . "\\\\back begin\\\\{tex\\\\}")

	;; Do the following manually because of differences with LaTeXinfo
	;; @@table
	;; @@titlepage
	;; @@subtitle
	;; @@itemx
	;; @@settitle

	;; commands with arguments already in braces will be picked up
	;; automatically by the global substitution of @@ -> \back
	
	;; The arguments themselves, assumed to be inside of something like @code{}
	("{@@printindex}" . "{\\\\back printindex}")

	("{@@author}" . "{\\\\back author}")
	("{@@date}" . "{\\\\back date}")
	("{@@title}" . "{\\\\back title}")

	;; Sections and chapters are done LaTeX style
	("{@@chapter}" . "{\\\\back chapter}")
	("{@@top}" . "{\\\\back top}")
	("{@@section}" . "{\\\\back section}")
	("{@@subsection}" . "{\\\\back subsection}")
	("{@@subsubsection}" . "{\\\\back subsubsection}")

	("^@@top[ \t]+\\([^ \t]\\)$" . "\\\\back top\\\\{\\1\\\\}")
	("^@@chapter[ \t]+\\([^ \t]\\)$" . "\\\\back chapter\\\\{\\1\\\\}")
	("^@@section[ \t]+\\([^ \t]+\\)$" . "\\\\back section\\\\{\\1\\\\}")
	("^@@subsection[ \t]+\\([^ \t]+\\)$" . "\\\\back subsection\\\\{\\1\\\\}")
	("^@@subsubsection[ \t]+\\([^ \t]+\\)$" . "\\\\back subsubsection\\\\{\\1\\\\}")

	("{@@appendix}" . "{\\\\back chapter}")
	("{@@appendixsec}" . "{\\\\back section}")
	("{@@appendixsection}" . "{\\\\back section}")
	("{@@appendixsubsec}" . "{\\\\back subsection}")
	("{@@appendixsubsubsec}" . "{\\\\back subsubsection}")

	("^@@appendix[ \t]+\\(.+\\)$" . "\\\\back chapter\\\\{\\1\\\\}")
	("^@@appendixsec[ \t]+\\(.+\\)$" . "\\\\back section\\\\{\\1\\\\}")
	("^@@appendixsection[ \t]+\\(.+\\)$" . "\\\\back section\\\\{\\1\\\\}")
	("^@@appendixsubsec[ \t]+\\(.+\\)$" . "\\\\back subsection\\\\{\\1\\\\}")
	("^@@appendixsubsubsec[ \t]+\\(.+\\)$" . "\\\\back subsubsection\\\\{\\1\\\\}")

	("{@@unnumbered}" . "{\\\\back unnumbered}")
	("{@@unnumberedsec}" . "{\\\\back unnumberedsec}")
	("{@@unnumberedsubsec}" . "{\\\\back unnumberedsubsec}")
	("{@@unnumberedsubsubsec}" . "{\\\\back unnumberedsubsubsec}")

	("^@@unnumbered[ \t]+\\(.+\\)$" . "\\\\back unnumbered\\\\{\\1\\\\}")
	("^@@unnumberedsec[ \t]+\\(.+\\)$" . "\\\\back unnumberedsec\\\\{\\1\\\\}")
	("^@@unnumberedsubsec[ \t]+\\(.+\\)$" . "\\\\back unnumberedsubsec\\\\{\\1\\\\}")
	("^@@unnumberedsubsubsec[ \t]+\\(.+\\)$" . "\\\\back unnumberedsubsubsec\\\\{\\1\\\\}")

	("{@@chapheading}" . "{\\\\back chapter*}")
	("{@@majorheading}" . "{\\\\back chapter*}")
	("{@@heading}" . "{\\\\back section*}")
	("{@@subheading}" . "{\\\\back subsection*}")
	("{@@subsubheading}" . "{\\\\back subsubsection*}")

	("^@@chapheading[ \t]+\\(.+\\)$" . "\\\\back chapter*\\\\{\\1\\\\}")
	("^@@majorheading[ \t]+\\(.+\\)$" . "\\\\back chapter*\\\\{\\1\\\\}")
	("^@@heading[ \t]+\\(.+\\)$" . "\\\\back section*\\\\{\\1\\\\}")
	("^@@subheading[ \t]+\\(.+\\)$" . "\\\\back subsection*\\\\{\\1\\\\}")
	("^@@subsubheading[ \t]+\\(.+\\)$" . "\\\\back subsubsection*\\\\{\\1\\\\}")

	("{@@cindex}" . "{\\\\back cindex}")
	("{@@vindex}" . "{\\\\back vindex}")
	("{@@findex}" . "{\\\\back findex}")
	("{@@tindex}" . "{\\\\back tindex}")
	("{@@pindex}" . "{\\\\back pindex}")
	("{@@kindex}" . "{\\\\back kindex}")
	("{@@center}" . "{\\\\back begin\\\\{center\\\\}}")
	("{@@setfilename}" . "{\\\\back setfilename}")

	;; Examples of arguments on their own lines.
	("^@@printindex[ \t]+\\(.+\\)$" . "\\\\back printindex\\\\{\\1\\\\}")

	("^@@author[ \t]+\\(.+\\)$" . "\\\\back author\\\\{\\1\\\\}")
	("^@@date[ \t]+\\(.+\\)$" . "\\\\back date\\\\{\\1\\\\}")
	("^@@title[ \t]+\\(.+\\)$" . "\\\\back title\\\\{\\1\\\\}")

	("^@@itemize[ \t]+\\([^ 	]+\\)$" . "\\\\back begin\\\\{itemize\\\\}")
	("^@@bye$" . "\\\\back end\\\\{document\\\\}")
	("^@@page$" . "\\\\back clearpage")
	("{@@page}" . "{\\\\back clearpage}")

	("{@@sp}" . "{\\\\back sp}")
	("{@@need}" . "{\\\\back need}")
	("{@@vskip}" . "{\\\\back vspace}")

	("^@@sp[ \t]+\\([^ \t]+\\)$" . "\\\\back sp\\\\{\\1\\\\}")
	("^@@need[ \t]+\\([^ \t]+\\)$" . "\\\\back need\\\\{\\1\\\\}")
	("^@@vskip[ \t]+\\([^ \t]\\)$" . "\\\\back vspace\\\\{\\1\\\\}")

	("{@@synindex}" . "{\\\\back synindex}")
	("{@@syncodeindex}" . "{\\\\back syncodeindex}")

	("^@@synindex[ \t]+\\([^ \t]\\)[ \t]+\\([^ \t]\\)$"
	 . "\\\\back synindex\\\\{\\1\\\\}\\\\{\\2\\\\}")
	("^@@syncodeindex[ \t]+\\([^ \t]\\)[ \t]+\\([^ \t]\\)$"
	 . "\\\\back syncodeindex\\\\{\\1\\\\}\\\\{\\2\\\\}")

	("@@footnotestyle[ \t]+\\([a-z]+\\)" . "\\\\back footnotestyle\\\\{\\1\\\\}")

	("^@@cindex[ \t]+\\(.+\\)$" . "\\\\back cindex\\\\{\\1\\\\}")
	("^@@vindex[ \t]+\\(.+\\)$" . "\\\\back vindex\\\\{\\1\\\\}")
	("^@@findex[ \t]+\\(.+\\)$" . "\\\\back findex\\\\{\\1\\\\}")
	("^@@tindex[ \t]+\\(.+\\)$" . "\\\\back tindex\\\\{\\1\\\\}")
	("^@@pindex[ \t]+\\(.+\\)$" . "\\\\back pindex\\\\{\\1\\\\}")
	("^@@kindex[ \t]+\\(.+\\)$" . "\\\\back kindex\\\\{\\1\\\\}")

	("^@@center[ \t]+\\(.+\\)$" . "\\\\back begin\\\\{center\\\\}\n\\1\n\\\\back end\\\\{center\\\\}")
	("^@@setfilename[ \t]+\\(.+\\)$" . "\\\\back setfilename\\\\{\\1\\\\}")
	))

(setq latex-alist
      '(
	;; begin end environments

	("^@def\\([a-z]+\\)[ \t]+\\([a-z---@{}]+\\)[ \t]+\\([a-z---@{}]+\\)[ \t]+\\([a-z---@{}]+\\)" . "\\\\def\\1{\\2}{\\3}{\\4}")

	("^@def\\([a-z]+\\)" . "\\\\def\\1")

	("^@enumerate" . "\\\\begin{enumerate}")
	("^@example" . "\\\\begin{example}")
	("^@smallexample" . "\\\\begin{smallexample}")
	("^@flushleft" . "\\\\begin{flushleft}")
	("^@flushright" . "\\\\begin{flushright}")
	("^@format" . "\\\\begin{format}")
	("^@group" . "\\\\begin{same}")
	("^@ignore" . "\\\\begin{ignore}")
	("^@ifinfo" . "\\\\begin{ifinfo}")
	("^@iftex" . "\\\\begin{iftex}")
	("^@lisp" . "\\\\begin{lisp}")
	("^@menu" . "\\\\begin{menu}")
	("^@quotation" . "\\\\begin{quote}")
	("@TeX{}" . "\\\\TeX{}")
	("^@tex" . "\\\\begin{tex}")

	;; change the name of the group command
	("^@end[ \t]+group" . "\\\\end{same}")
	("^@end[ \t]+quotation" . "\\\\end{quote}")
    
	;; We turn the @table into a description environment
	;; There may be a lost of information here.
	("^@table[ \t]+.*" . "\\\\begin{description}")
	("^@end[ \t]table" . "\\\\end{description}")

	;; Take care of the titlepage before we remove the @end commands.
	("^@titlepage" . "\\\\pagestyle{empty}
\\\\date{\\\\today}

")
	;; They will probably have to fix the location of the \maketitle
	("^@end[ \t]titlepage" . "\\\\maketitle

\\\\clearpage
\\\\pagestyle{headings}
\\\\pagenumbering{roman}
\\\\tableofcontents

\\\\clearpage
\\\\pagenumbering{arabic}
")

	;; commands with arguments on the rest of the line
	("^@end[ \t]+\\(.+\\)" . "\\\\end{\\1}")
	;; The elisp manual has some @ifinfo 2**24 - 1@end ifinfo
	;; and @tex $2^{24}-1$@end tex with the @end on the smae line
	("@end[ \t]+ifinfo" . "\\\\end{ifinfo}")
	("@end[ \t]+tex" . "\\\\end{tex}")
	("@end[ \t]+iftex" . "\\\\end{iftex}")
	("@end[ \t]+defvar" . "\\\\end{defvar}")
	("^@printindex[ \t]+\\(.+\\)" . "\\\\printindex{\\1}")

	("^@author[ \t]+\\(.+\\)" . "\\\\author{\\1}")
	("^@date[ \t]+\\(.+\\)" . "\\\\date{\\1}")
	("^@title[ \t]+\\(.+\\)" . "\\\\title{\\1}")
	;; This is a little different from texinfo
	("}[ \t]*\n+@subtitle[ \t]+\\(.+\\)" . "\\\\\\\\\n\t\\\\n{\\1} }")
	("}[ \t]*\n+@subtitle[ \t]+\\(.+\\)" . "\\\\\\\\\n\t\\\\n{\\1} }")
	("}[ \t]*\n+@subtitle[ \t]+\\(.+\\)" . "\\\\\\\\\n\t\\\\n{\\1} }")
	("}[ \t]*\n+@subtitle[ \t]+\\(.+\\)" . "\\\\\\\\\n\t\\\\n{\\1} }")

	;; Sections and chapters are done LaTeX style
	("^@top[ \t]+\\(.+\\)" . "\\\\top{\\1}")
	("^@chapter[ \t]+\\(.+\\)" . "\\\\chapter{\\1}")
	("^@section[ \t]+\\(.+\\)" . "\\\\section{\\1}")
	("^@subsection[ \t]+\\(.+\\)" . "\\\\subsection{\\1}")
	("^@subsubsection[ \t]+\\(.+\\)" . "\\\\subsubsection{\\1}")

	("^@unnumbered[ \t]+\\(.+\\)" . "\\\\unnumbered{\\1}")
	("^@unnumberedsec[ \t]+\\(.+\\)" . "\\\\unnumberedsec{\\1}")
	("^@unnumberedsubsec[ \t]+\\(.+\\)" . "\\\\unnumberedsubsec{\\1}")
	("^@unnumberedsubsubsec[ \t]+\\(.+\\)" . "\\\\unnumberedsubsubsec{\\1}")

	("^@chapheading[ \t]+\\(.+\\)" . "\\\\chapter*{\\1}")
	("^@majorheading[ \t]+\\(.+\\)" . "\\\\chapter*{\\1}")
	("^@heading[ \t]+\\(.+\\)" . "\\\\section*{\\1}")
	("^@subheading[ \t]+\\(.+\\)" . "\\\\subsection*{\\1}")
	("^@subsubheading[ \t]+\\(.+\\)" . "\\\\subsubsection*{\\1}")

	("^@appendix[ \t]+\\(.+\\)" . "\\\\chapter{\\1}")
	("^@appendixsec[ \t]+\\(.+\\)" . "\\\\section{\\1}")
	("^@appendixsection[ \t]+\\(.+\\)" . "\\\\section{\\1}")
	("^@appendixsubsec[ \t]+\\(.+\\)" . "\\\\subsection{\\1}")
	("^@appendixsubsubsec[ \t]+\\(.+\\)" . "\\\\subsubsection{\\1}")

	;; The order is important here
	;; We turn the @itemize into an itemize environment
	;; There may be a lost of information here.
	("^@itemize[ \t]+.*" . "\\\\begin{itemize}")
	("^@item[ \t]+\\(.+\\)[ \t]*" . "\\\\item[\\1]")
	;; This is a little different from texinfo
	("\\]\n+@itemx[ \t]+\\(.+\\)" . ", \\1]")
	("\\]\n+@itemx[ \t]+\\(.+\\)" . ",\n \\1]")
	("\\]\n+@itemx[ \t]+\\(.+\\)" . ", \\1]")
	("\\]\n+@itemx[ \t]+\\(.+\\)" . ",\n \\1]")
	("\\]\n+@itemx[ \t]+\\(.+\\)" . ", \\1]")
	("\\]\n+@itemx[ \t]+\\(.+\\)" . ",\n \\1]")
	("^@item" . "\\\\item")
	("^@sp[ \t]+\\(.+\\)" . "\\\\sp{\\1}")
	("^@need[ \t]+\\(.+\\)" . "\\\\need{\\1}")
	("@vskip[ \t]+\\(.+\\)" . "\\\\vspace{\\1}")

	("^@cindex[ \t]+\\(.+\\)" . "\\\\cindex{\\1}")
	("^@vindex[ \t]+\\(.+\\)" . "\\\\vindex{\\1}")
	("^@findex[ \t]+\\(.+\\)" . "\\\\findex{\\1}")
	("^@tindex[ \t]+\\(.+\\)" . "\\\\tindex{\\1}")
	("^@pindex[ \t]+\\(.+\\)" . "\\\\pindex{\\1}")
	("^@kindex[ \t]+\\(.+\\)" . "\\\\kindex{\\1}")
	("^@center[ \t]+\\(.+\\)" . "\\\\begin{center}\n\\1\n\\\\end{center}")
	("^@setfilename[ \t]+\\(.+\\)" . "\\\\setfilename{\\1}")
	("^@settitle[ \t]+\\(.+\\)" . "\\\\markboth{\\1}{\\1}")

	;; commands with arguments already in braces
	("@b{" . "\\\\b{")
	("@t{" . "\\\\t{")
	("@i{" . "\\\\i{")
	("@r{" . "\\\\r{")
	("@key{" . "\\\\key{")
	("@w{" . "\\\\w{")
	("@sc{" . "\\\\s{")
	("@dmn{" . "\\\\dmn{")

	("@titlefont{" . "{\\\\Large ")
	("@code{" . "\\\\code{")
	("@samp{" . "\\\\samp{")
	("@file{" . "\\\\file{")
	("@kbd{" . "\\\\kbd{")

	("@today{" . "\\\\today{")
	("@footnote{" . "\\\\footnote{")
	("@footnotestyle{" . "\\\\footnotestyle{")

	("@strong{" . "\\\\strong{")
	("@emph{" . "\\\\emph{")

	("@defn{" . "\\\\defn{")
	("@dfn[ \t]*{" . "\\\\dfn{")
	("@ctrl{" . "\\\\ctrl{")
	("@var{" . "\\\\var{")
	("@cite{" . "\\\\cite{")
	("@pxref{" . "\\\\pxref{")
	("@ref{" . "\\\\sxref{")
	("@xref{" . "\\\\xref{")
	("@inforef{" . "\\\\inforef{")
	("@synindex{" . "\\\\synindex{")
	("@syncodeindex{" . "\\\\syncodeindex{")
	("@copyright{}" . "\\\\copyright\\\\ ")
	("@copyright" . "\\\\copyright")

	;; commands without arguments
	("@\\." . "\\\\.")
	("^@node" . "\\\\node")
	("@noindent" . "\\\\noindent")
	("@comment" . "\\\\comment")
	("@bullet" . "\\\\bullet")
	("@smallbook" . "\\\\smallbook")

	("@arrow" . "\\\\arrow")
	("@Arrow" . "\\\\arrow")
	("@equiv" . "\\\\equiv")
	("@result" . "\\\\result")
	("@equiv" . "\\\\equiv")
	("@error" . "\\\\error")
	("@expansion" . "\\\\expansion")
	("@point" . "\\\\point")
	("@print" . "\\\\print")
    
	("@page" . "\\\\clearpage")
	("^@bye" . "\\\\end{document}")
	("@TeX" . "\\\\TeX")

	("@exdent" . "\\\\exdent")
	("@minus" . "\\\\minus")
	("@finalout" . "\\\\finalout")
	("@cartouche" . "\\\\cartouche")

	("@dots" . "\\\\dots")
	("@refill" . "\\\\refill")
    
	("@:" . "\\\\:")
	("@{" . "\\\\{")
	("@}" . "\\\\}")
	("@\\*" . "\\\\*")

	("@c[ \t]-*-texinfo-*-" . "\\\\c -*-latexinfo-*-")
	("@c[ \t]+" . "\\\\c ")
	))

  (setq latex-delete-list
	'(
	;;; Still to do
	  "@syncodeindex"
	;;; Texinfo 2 commands to be ignored because they have been replaced.
	  ;; This is in the chapter command.
	  "@setchapternewpage[ \t]"
	  ;; Haven't made a \summarycontents yet.
	  "@summarycontents"
	  ))


(defun tex-to-latexinfo-optionals ()
  (goto-char (point-min))
  (replace-regexp "{\\([a-zA-Z 	]+\\) Texinfo \\([a-zA-Z 	]+\\)}" "{\\1 LaTeXinfo \\2}" nil)
  (goto-char (point-min))
  (replace-regexp "{texinfo\\([a-z---]+\\)}" "{latexinfo\\1}" nil)
  (goto-char (point-min))
  (query-replace " Texinfo " " \\LaTeX info " nil)
  (goto-char (point-min))
  (query-replace " Texinfo " " LaTeXinfo " nil)
  (goto-char (point-min))
  (query-replace " Texinfo" " \\LaTeX info" nil)
  (goto-char (point-min))
  (replace-regexp "^Texinfo " "\\\\LaTeX info " nil)
  (goto-char (point-min))
  (query-replace-regexp "Texinfo$" "\\\\LaTeX info" nil)
  (goto-char (point-min))
  (query-replace-regexp "Texinfo$" "LaTeXinfo" nil)
  )
