

use core::cmp::min;

use crate::unix::solarish::*;
use crate::{c_char, c_int, size_t};

const PTEM: &[u8] = b"ptem\0";
const LDTERM: &[u8] = b"ldterm\0";

pub unsafe fn cfmakeraw(termios: *mut crate::termios) {
    (*termios).c_iflag &=
        !(IMAXBEL | IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON);
    (*termios).c_oflag &= !OPOST;
    (*termios).c_lflag &= !(ECHO | ECHONL | ICANON | ISIG | IEXTEN);
    (*termios).c_cflag &= !(CSIZE | PARENB);
    (*termios).c_cflag |= CS8;

    
    
    
    
    
    
    
    
    
    
    
    
    
    (*termios).c_cc[VMIN] = 1;
    (*termios).c_cc[VTIME] = 0;
}

pub unsafe fn cfsetspeed(termios: *mut crate::termios, speed: crate::speed_t) -> c_int {
    
    
    crate::cfsetispeed(termios, speed);
    crate::cfsetospeed(termios, speed);
    0
}

unsafe fn bail(fdm: c_int, fds: c_int) -> c_int {
    let e = *___errno();
    if fds >= 0 {
        crate::close(fds);
    }
    if fdm >= 0 {
        crate::close(fdm);
    }
    *___errno() = e;
    return -1;
}

pub unsafe fn openpty(
    amain: *mut c_int,
    asubord: *mut c_int,
    name: *mut c_char,
    termp: *const termios,
    winp: *const crate::winsize,
) -> c_int {
    
    
    let fdm = crate::posix_openpt(O_RDWR | O_NOCTTY);
    if fdm < 0 {
        return -1;
    }

    
    if crate::grantpt(fdm) < 0 || crate::unlockpt(fdm) < 0 {
        return bail(fdm, -1);
    }

    
    let subordpath = crate::ptsname(fdm);
    if subordpath.is_null() {
        return bail(fdm, -1);
    }

    
    
    let fds = crate::open(subordpath, O_RDWR | O_NOCTTY);
    if fds < 0 {
        return bail(fdm, -1);
    }

    
    let setup = crate::ioctl(fds, I_FIND, LDTERM.as_ptr());
    if setup < 0 {
        return bail(fdm, fds);
    } else if setup == 0 {
        
        
        if crate::ioctl(fds, I_PUSH, PTEM.as_ptr()) < 0
            || crate::ioctl(fds, I_PUSH, LDTERM.as_ptr()) < 0
        {
            return bail(fdm, fds);
        }
    }

    
    if !termp.is_null() && crate::tcsetattr(fds, TCSAFLUSH, termp) != 0 {
        return bail(fdm, fds);
    }

    
    if !winp.is_null() && crate::ioctl(fds, TIOCSWINSZ, winp) < 0 {
        return bail(fdm, fds);
    }

    
    
    
    
    
    if !name.is_null() {
        crate::strcpy(name, subordpath);
    }

    *amain = fdm;
    *asubord = fds;
    0
}

pub unsafe fn forkpty(
    amain: *mut c_int,
    name: *mut c_char,
    termp: *const termios,
    winp: *const crate::winsize,
) -> crate::pid_t {
    let mut fds = -1;

    if openpty(amain, &mut fds, name, termp, winp) != 0 {
        return -1;
    }

    let pid = crate::fork();
    if pid < 0 {
        return bail(*amain, fds);
    } else if pid > 0 {
        
        
        crate::close(fds);
        return pid;
    }

    

    
    crate::close(*amain);

    
    
    
    
    if crate::setsid() < 0
        || crate::ioctl(fds, TIOCSCTTY, 0) < 0
        || crate::dup2(fds, 0) < 0
        || crate::dup2(fds, 1) < 0
        || crate::dup2(fds, 2) < 0
    {
        
        
        
        crate::_exit(EXIT_FAILURE);
    }
    
    
    if fds > 2 {
        crate::close(fds);
    }

    0
}

pub unsafe fn getpwent_r(
    pwd: *mut passwd,
    buf: *mut c_char,
    buflen: size_t,
    result: *mut *mut passwd,
) -> c_int {
    let old_errno = *crate::___errno();
    *crate::___errno() = 0;
    *result = native_getpwent_r(pwd, buf, min(buflen, c_int::max_value() as size_t) as c_int);

    let ret = if (*result).is_null() {
        *crate::___errno()
    } else {
        0
    };
    *crate::___errno() = old_errno;

    ret
}

pub unsafe fn getgrent_r(
    grp: *mut crate::group,
    buf: *mut c_char,
    buflen: size_t,
    result: *mut *mut crate::group,
) -> c_int {
    let old_errno = *crate::___errno();
    *crate::___errno() = 0;
    *result = native_getgrent_r(grp, buf, min(buflen, c_int::max_value() as size_t) as c_int);

    let ret = if (*result).is_null() {
        *crate::___errno()
    } else {
        0
    };
    *crate::___errno() = old_errno;

    ret
}
