% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BackgroundSamplingFunctions.R
\name{mSampling3D}
\alias{mSampling3D}
\title{3D background sampling}
\usage{
mSampling3D(occs, envBrick, mShp, depthLimit = "all", verbose = TRUE)
}
\arguments{
\item{occs}{A \code{data.frame} with at least three columns
named "longitude", "latitude", and "depth", or that
can be coerced into this format.}

\item{envBrick}{A \code{SpatRaster} vector object to serve
as a template for generating background sampling
coordinates.}

\item{mShp}{A shapefile defining the area from
which background points should be sampled.}

\item{depthLimit}{An argument controlling the depth
extent of sampling. Refer to \code{Details} for more information.}

\item{verbose}{\code{logical}. Switching to \code{FALSE} mutes message describing
which columns in \code{occs} are interpreted as x, y, and z coordinates.}
}
\value{
A \code{data.frame} with 3D coordinates of points for background
sampling.
}
\description{
Samples XYZ coordinates from a shapefile
from maximum to minimum occurrence depth at XYZ
resolution of envBrick.
}
\details{
This function is designed to sample background points for
distributional modeling in three dimensions. If a voxel (3D pixel)
in the \code{SpatRaster} vector intersects with an occurrence from
\code{occs}, it is removed. Note that this function returns points
representing every voxel in the background area within the
specified depth range. It is up to the user to downsample from
these data as necessary, depending on the model type being used.

\code{depthLimit} argument options:
\itemize{
\item \code{occs} Samples background from the full depth extent of \code{occs}.
\item \code{all} Samples background from the full depth extent of \code{envBrick}.
\item A \code{vector} of length 2 with maximum and minimum depth values from
which to sample.
}
}
\examples{
library(terra)

# Create test raster
r1 <- rast(ncol=10, nrow=10)
values(r1) <- 1:100
r2 <- rast(ncol=10, nrow=10)
values(r2) <- c(rep(20, times = 50), rep(60, times = 50))
r3 <- rast(ncol=10, nrow=10)
values(r3) <- 8
envBrick <- c(r1, r2, r3)
names(envBrick) <- c(0, 10, 30)

# Create test occurrences
set.seed(0)
longitude <- sample(ext(envBrick)[1]:ext(envBrick)[2],
                    size = 10, replace = FALSE)
set.seed(0)
latitude <- sample(ext(envBrick)[3]:ext(envBrick)[4],
                   size = 10, replace = FALSE)
set.seed(0)
depth <- sample(0:35, size = 10, replace = TRUE)
occurrences <- data.frame(longitude,latitude,depth)

# Generate background sampling buffer
buffPts <- vect(occurrences,
                c("longitude", "latitude"))
crs(buffPts) <- crs(envBrick)
mShp <- aggregate(buffer(buffPts, width = 1000000))

# Here's the function
occSample3d <- mSampling3D(occs = occurrences,
                           envBrick = envBrick,
                           mShp = mShp,
                           depthLimit = "occs")

}
\keyword{backgroundSampling}
