% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/newton_raphson_anim.R
\name{newton_raphson_anim}
\alias{newton_raphson_anim}
\title{Newton-Raphson root finding with tangent animation (Plotly)}
\usage{
newton_raphson_anim(
  f,
  x0,
  df = NULL,
  h = 1e-04,
  max_iter = 10L,
  tol = 1e-08,
  xlim = NULL,
  n_curve = 600L,
  frame_ms = 600L,
  transition_ms = 400L,
  title = NULL,
  safe_mode = TRUE
)
}
\arguments{
\item{f}{Function. A real-valued function f(x). Must accept a numeric vector
and return a numeric vector of the same length.}

\item{x0}{Numeric scalar. Initial guess.}

\item{df}{Optional function. Derivative f'(x). If NULL, a numerical derivative
is used.}

\item{h}{Numeric scalar. Step size for numerical derivative (when df is NULL).}

\item{max_iter}{Integer. Maximum number of iterations.}

\item{tol}{Numeric scalar. Stopping tolerance based on |f(x_n)|.}

\item{xlim}{Numeric vector of length 2. Plot range for x. If NULL, it is chosen
around the iterates.}

\item{n_curve}{Integer. Number of points used to draw the curve.}

\item{frame_ms}{Integer. Frame duration in milliseconds.}

\item{transition_ms}{Integer. Transition duration in milliseconds.}

\item{title}{Character. Plot title. If NULL, a default title is used.}

\item{safe_mode}{Logical. If TRUE, use calmer animation defaults intended to
reduce flicker and visual stress.}
}
\value{
A list with components:
\describe{
\item{plot}{A plotly object (htmlwidget) with animation frames.}
\item{iterates}{Data frame with iterations (n, x, fx, dfx).}
\item{root}{Last iterate (approximate root).}
\item{converged}{Logical. TRUE if convergence was detected within max_iter.}
}
}
\description{
Builds a Plotly animation of the Newton-Raphson method for finding a root of
a real function. Each frame shows the tangent line at the current iterate
and how its x-intercept defines the next iterate.
}
\details{
The Newton-Raphson update is
\deqn{
  x_{n+1} = x_n - \frac{f(x_n)}{f'(x_n)}.
}
If a derivative function is not provided, the derivative is approximated
numerically by the central difference
\deqn{
  f'(x) \approx \frac{f(x+h)-f(x-h)}{2h}.
}
}
\examples{
\donttest{
library(plotly)

f <- function(x) x^3 - 2*x - 5
out <- newton_raphson_anim(f, x0 = 2)
out$plot
out$root

g <- function(x) cos(x) - x
newton_raphson_anim(g, x0 = 1)
}

}
