% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/deming_regression.R
\name{deming_regression}
\alias{deming_regression}
\title{Deming Regression for Method Comparison}
\usage{
deming_regression(
  x,
  y = NULL,
  data = NULL,
  error_ratio = 1,
  conf_level = 0.95,
  ci_method = c("jackknife", "bootstrap"),
  boot_n = 1999,
  weighted = FALSE,
  na_action = c("omit", "fail")
)
}
\arguments{
\item{x}{Numeric vector of measurements from method 1 (reference method),
or a formula of the form \code{method1 ~ method2}.}

\item{y}{Numeric vector of measurements from method 2 (test method).
Ignored if \code{x} is a formula.}

\item{data}{Optional data frame containing the variables specified in
\code{x} and \code{y} (or in the formula).}

\item{error_ratio}{Ratio of error variances (Var(error_y) / Var(error_x)).
Default is 1 (orthogonal regression, assuming equal error variances).
Can be estimated from replicate measurements or set based on prior
knowledge of method precision.}

\item{conf_level}{Confidence level for intervals (default: 0.95).}

\item{ci_method}{Method for calculating confidence intervals:
\code{"jackknife"} (default) uses delete-one jackknife resampling,
\code{"bootstrap"} uses BCa bootstrap resampling.}

\item{boot_n}{Number of bootstrap resamples when \code{ci_method = "bootstrap"}
(default: 1999).}

\item{weighted}{Logical; if \code{TRUE}, performs weighted Deming regression
where weights are inversely proportional to the variance at each point.
Requires replicate measurements to estimate weights. Default is \code{FALSE}.}

\item{na_action}{How to handle missing values: \code{"omit"} (default) removes
pairs with any NA, \code{"fail"} stops with an error.}
}
\value{
An object of class \code{c("deming_regression", "valytics_comparison", "valytics_result")},
which is a list containing:

\describe{
\item{input}{List with original data and metadata:
\itemize{
\item \code{x}, \code{y}: Numeric vectors (after NA handling)
\item \code{n}: Number of paired observations
\item \code{n_excluded}: Number of pairs excluded due to NAs
\item \code{var_names}: Named character vector with variable names
}
}
\item{results}{List with statistical results:
\itemize{
\item \code{intercept}: Intercept point estimate
\item \code{slope}: Slope point estimate
\item \code{intercept_ci}: Named numeric vector with lower and upper CI
\item \code{slope_ci}: Named numeric vector with lower and upper CI
\item \code{intercept_se}: Standard error of intercept
\item \code{slope_se}: Standard error of slope
\item \code{residuals}: Perpendicular residuals
\item \code{fitted_x}: Fitted x values
\item \code{fitted_y}: Fitted y values
}
}
\item{settings}{List with analysis settings:
\itemize{
\item \code{error_ratio}: Error variance ratio used
\item \code{conf_level}: Confidence level used
\item \code{ci_method}: CI method used
\item \code{boot_n}: Number of bootstrap samples (if applicable)
\item \code{weighted}: Whether weighted regression was used
}
}
\item{call}{The matched function call.}
}
}
\description{
Performs Deming regression to assess agreement between two measurement
methods. Unlike ordinary least squares, Deming regression accounts for
measurement error in both variables, making it appropriate for method
comparison studies where neither method is a perfect reference.
}
\details{
Deming regression (also known as errors-in-variables regression or Model II
regression) is designed for situations where both X and Y are measured with
error. This is the typical case in method comparison studies where both the
reference and test methods have measurement uncertainty.

The error ratio (lambda, \eqn{\lambda}) represents the ratio of error variances:
\deqn{\lambda = \frac{Var(\epsilon_y)}{Var(\epsilon_x)}}

When \eqn{\lambda} = 1 (default), this is equivalent to orthogonal regression, which
minimizes perpendicular distances to the regression line. When \eqn{\lambda} != 1, the
regression minimizes a weighted combination of horizontal and vertical
distances.

\strong{Choosing the error ratio:}
\itemize{
\item If both methods have similar precision: use \eqn{\lambda} = 1
\item If precision differs: estimate from replicate measurements as
\eqn{\lambda} = CV_y² / CV_x² (squared coefficient of variation ratio)
\item If one method is much more precise: consider ordinary least squares
}
}
\section{Interpretation}{

\itemize{
\item \strong{Slope = 1}: No proportional difference between methods
\item \strong{Slope != 1}: Proportional (multiplicative) difference exists
\item \strong{Intercept = 0}: No constant difference between methods
\item \strong{Intercept != 0}: Constant (additive) difference exists
}

Use the confidence intervals to test these hypotheses: if 1 is within the
slope CI and 0 is within the intercept CI, the methods are considered
equivalent.
}

\section{Comparison with Other Methods}{

\itemize{
\item \strong{Ordinary Least Squares (OLS)}: Assumes X is measured without
error. Biases slope toward zero when both variables have error.
\item \strong{Passing-Bablok}: Non-parametric, robust to outliers, but assumes
linear relationship and no ties.
\item \strong{Deming}: Parametric, accounts for error in both variables,
allows specification of error ratio.
}
}

\section{Assumptions}{

\itemize{
\item Linear relationship between X and Y
\item Measurement errors are normally distributed
\item Error variances are constant (homoscedastic) or known ratio
\item Errors in X and Y are independent
}
}

\examples{
# Simulated method comparison data
set.seed(42)
true_values <- rnorm(50, mean = 100, sd = 20)
method_a <- true_values + rnorm(50, sd = 5)
method_b <- 1.05 * true_values + 3 + rnorm(50, sd = 5)

# Basic analysis (orthogonal regression, lambda = 1)
dm <- deming_regression(method_a, method_b)
dm

# Using formula interface with data frame
df <- data.frame(reference = method_a, test = method_b)
dm <- deming_regression(reference ~ test, data = df)

# With known error ratio (e.g., test method has 2x variance)
dm <- deming_regression(method_a, method_b, error_ratio = 2)

# With bootstrap confidence intervals
dm_boot <- deming_regression(method_a, method_b, ci_method = "bootstrap")

# Using package example data
data(glucose_methods)
dm <- deming_regression(reference ~ poc_meter, data = glucose_methods)
summary(dm)
plot(dm)

}
\references{
Deming WE (1943). Statistical Adjustment of Data. Wiley.

Linnet K (1990). Estimation of the linear relationship between the
measurements of two methods with proportional errors.
\emph{Statistics in Medicine}, 9(12):1463-1473.
\doi{10.1002/sim.4780091210}

Linnet K (1993). Evaluation of regression procedures for methods comparison
studies. \emph{Clinical Chemistry}, 39(3):424-432.
\doi{10.1093/clinchem/39.3.424}

Cornbleet PJ, Gochman N (1979). Incorrect least-squares regression
coefficients in method-comparison analysis.
\emph{Clinical Chemistry}, 25(3):432-438.
}
\seealso{
\code{\link[=plot.deming_regression]{plot.deming_regression()}} for visualization,
\code{\link[=summary.deming_regression]{summary.deming_regression()}} for detailed summary,
\code{\link[=pb_regression]{pb_regression()}} for non-parametric alternative,
\code{\link[=ba_analysis]{ba_analysis()}} for Bland-Altman analysis
}
