% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tidylaslog.R
\name{tidylaslog}
\alias{tidylaslog}
\title{Universal entry point for reading, indexing, and exporting LAS well logs}
\usage{
tidylaslog(
  x,
  county = NULL,
  curves_any = NULL,
  curves_all = NULL,
  curves = NULL,
  output = c("wide", "long"),
  out_dir = NULL,
  prefix = NULL,
  formats = c("csv", "parquet"),
  write_index = TRUE,
  write_meta = TRUE,
  meta_sections = c("VERSION", "WELL", "CURVE", "PARAMETER", "OTHER"),
  manifest = TRUE
)
}
\arguments{
\item{x}{Path to a \code{.las} file OR a directory containing \code{.las} files.}

\item{county}{Optional county filter (directory mode only).}

\item{curves_any}{Keep wells that contain \emph{at least one} of these curves
(directory mode).}

\item{curves_all}{Keep wells that contain \emph{all} of these curves
(directory mode).}

\item{curves}{Curves to actually keep/export. Defaults to \code{curves_all},
then \code{curves_any}, otherwise all curves.}

\item{output}{Output format:
\describe{
\item{"wide"}{One row per depth per well, curves as columns
(ML- and spreadsheet-ready).}
\item{"long"}{One row per curve measurement with columns
\code{depth}, \code{mnemonic}, and \code{value} (tidy format).}
}}

\item{out_dir}{If \code{NULL}, data are returned to R only.
If provided, outputs are written to this directory.
If relative (e.g. \code{"exports"}), it is created inside \code{x} when \code{x} is a directory.}

\item{prefix}{Optional filename prefix for exported files.}

\item{formats}{Output formats to write. One or both of \code{"csv"} and \code{"parquet"}.}

\item{write_index}{Write index tables (wells, curves, files) when exporting
directories?}

\item{write_meta}{Write metadata tables (\code{WELL}, \code{CURVE}, etc.) for
single-file exports?}

\item{meta_sections}{Which metadata sections to export
(\code{"VERSION"}, \code{"WELL"}, \code{"CURVE"}, \code{"PARAMETER"}, \code{"OTHER"}).}

\item{manifest}{Write a JSON manifest describing the export?}
}
\value{
If \code{out_dir} is \code{NULL}:
\describe{
\item{Single file}{An S3 object of class \code{"laslog"} containing
\code{VERSION}, \code{WELL}, \code{CURVE}, \code{PARAMETER}, \code{OTHER}, and \code{LOG}.}
\item{Directory}{A list with \code{index}, \code{apis}, and combined \code{data}.}
}

If \code{out_dir} is provided:
\describe{
\item{Single file}{A list containing exported data paths, metadata paths,
and an optional manifest.}
\item{Directory}{The full batch export result (see \code{batch_export_laslogs()}).}
}
}
\description{
\code{tidylaslog()} works with either a \strong{single LAS file} or a \strong{directory of LAS files}.
It can return data directly to R or export analysis-ready tables to disk.
}
\details{
The function supports two equivalent representations of LAS log data:
\itemize{
\item \strong{Wide format}: one row per depth step per well, with each curve stored as a
separate column.
\item \strong{Long format}: one row per measurement, with curve names stored in a
\code{mnemonic} column and values in a \code{value} column.
}

Both formats contain the same information but are optimized for different workflows
(machine learning vs tidy analysis).
}
\examples{
# ---- Single file mode (return to R) ----
las_text <- c(
  " ~Version Information",
  " VERS. 2.0:",
  " WRAP. NO:",
  " ~Well Information",
  " STRT.M 1000:",
  " STOP.M 1002:",
  " STEP.M 1:",
  " NULL. -999.25:",
  " API . 1111111111:",
  " CNTY. TEST:",
  " ~Curve Information",
  " DEPT.M:",
  " GR.API:",
  " ~ASCII Log Data",
  " 1000 80",
  " 1001 82",
  " 1002 79"
)
f <- tempfile(fileext = ".las")
writeLines(las_text, f)
obj <- tidylaslog(f, output = "long")
head(obj$LOG)

# ---- Directory mode (return to R) ----
td <- tempdir()
f1 <- file.path(td, "a.las")
f2 <- file.path(td, "b.las")
writeLines(las_text, f1)
writeLines(sub("1111111111", "2222222222", las_text), f2)
res <- tidylaslog(td, county = "TEST", curves_any = "GR", output = "wide")
names(res)

# ---- Export mode (CSV only, no arrow needed) ----
out_dir <- file.path(td, "exports_demo")
ex <- tidylaslog(td,
  county = "TEST",
  curves_any = "GR",
  output = "wide",
  out_dir = out_dir,
  formats = "csv",
  write_index = TRUE,
  manifest = FALSE
)
names(ex)
}
