
#' @importFrom stringr str_replace_all

ngram_frequencies <- function(x, min_freq = c(6, 4, 2)) {

  # --- Eingabe in Keilschrift konvertieren (falls noetig) ---
  # Pruefen ob bereits Keilschrift enthalten ist (U+12000..U+1254F)
  has_cuneiform <- any(grepl("[\U00012000-\U0001254F]", x, perl = TRUE))
  if (!has_cuneiform && length(x) > 0) {
    x <- as.cuneiform(x)
  }

  x <- str_replace_all(x, " ", "")

  # --- Vorverarbeitung ---
  # Kommentarzeilen (beginnen mit #) und Leerzeilen entfernen
  x <- x[!grepl("^\\s*#", x)]
  x <- x[nchar(trimws(x)) > 0]

  # Zeilennummern entfernen: "42)\t" oder "42) " am Zeilenanfang
  x <- sub("^\\s*\\d+\\)\\s*", "", x)

  # --- Tokenisierung ---
  # Jedes Keilschriftzeichen (U+12000..U+1254F) ist ein Token.
  # Jede Zeichenfolge in \u27E8...\u27E9 ist ein Token.
  # Alles andere wird ignoriert.
  tokenize_line <- function(line) {
    chars  <- strsplit(line, "")[[1]]
    n_ch   <- length(chars)
    tokens <- character(0)
    i      <- 1L

    while (i <= n_ch) {
      cp <- utf8ToInt(chars[i])

      if (cp == 0x27E8L) {
        # \u27E8 gefunden - alles bis \u27E9 sammeln
        bracket <- chars[i]
        i <- i + 1L
        while (i <= n_ch && utf8ToInt(chars[i]) != 0x27E9L) {
          bracket <- paste0(bracket, chars[i])
          i <- i + 1L
        }
        if (i <= n_ch) {
          bracket <- paste0(bracket, chars[i])  # \u27E9 anhaengen
          i <- i + 1L
        }
        tokens <- c(tokens, bracket)

      } else if (cp >= 0x12000L && cp <= 0x1254FL) {
        tokens <- c(tokens, chars[i])
        i <- i + 1L

      } else {
        i <- i + 1L
      }
    }
    tokens
  }

  lines_tok <- lapply(x, tokenize_line)
  lines_tok <- lines_tok[lengths(lines_tok) > 0L]

  if (length(lines_tok) == 0L) {
    return(data.frame(frequency  = integer(0),
                      length     = integer(0),
                      sign_names = character(0),
                      combination = character(0),
                      stringsAsFactors = FALSE))
  }

  # --- max_n automatisch bestimmen ---
  max_n <- max(lengths(lines_tok))

  # --- min_freq aufbereiten ---
  # min_freq ist ein unbenannter Vektor: c(6, 4, 2)
  # Der i-te Wert gilt fuer Kombinationen der Laenge i.
  # Fuer Laengen > length(min_freq) wird der letzte Wert verwendet.
  if (is.null(min_freq)) {
    min_freq <- c(6L, 4L, 2L)
  }
  min_freq <- as.integer(min_freq)

  get_min_freq <- function(n) {
    if (n <= length(min_freq)) return(min_freq[n])
    return(min_freq[length(min_freq)])
  }

  # --- Hilfsfunktion: N-Gramme extrahieren ---
  extract_ngrams <- function(tokens, n) {
    len <- length(tokens)
    if (len < n) return(character(0))
    vapply(seq_len(len - n + 1L), function(i) {
      paste(tokens[i:(i + n - 1L)], collapse = " ")
    }, character(1))
  }

  # --- Hauptanalyse: lang -> kurz mit Maskierung ---
  results  <- vector("list", max_n)
  working  <- lapply(lines_tok, identity)   # veraenderbare Kopie

  for (n in seq.int(max_n, 1L)) {
    mf <- get_min_freq(n)

    # Alle N-Gramme der Laenge n sammeln
    all_ng <- unlist(lapply(working, extract_ngrams, n = n),
                     use.names = FALSE)
    if (length(all_ng) == 0L) next

    freq <- table(all_ng)
    freq <- sort(freq[freq >= mf], decreasing = TRUE)
    if (length(freq) == 0L) next

    results[[n]] <- data.frame(
      combination = names(freq),
      length      = n,
      frequency   = as.integer(freq),
      stringsAsFactors = FALSE
    )

    # Maskierung: fuer jedes haeufige N-Gramm alle Vorkommen bis auf
    # das erste durch Platzhalter ersetzen
    for (ng_str in names(freq)) {
      ng_tok      <- strsplit(ng_str, " ", fixed = TRUE)[[1]]
      first_found <- FALSE

      for (li in seq_along(working)) {
        toks <- working[[li]]
        tlen <- length(toks)
        if (tlen < n) next

        pos <- 1L
        while (pos <= tlen - n + 1L) {
          match <- TRUE
          for (k in seq_len(n)) {
            if (toks[pos + k - 1L] != ng_tok[k]) { match <- FALSE; break }
          }

          if (match) {
            if (!first_found) {
              first_found <- TRUE
              pos <- pos + 1L            # erstes Vorkommen behalten
            } else {
              # Alle Zeichen dieses Vorkommens maskieren
              for (k in seq_len(n)) {
                toks[pos + k - 1L] <- paste0("\U0001F6AB", li, "_", pos, "_", k)
              }
              pos <- pos + n
            }
          } else {
            pos <- pos + 1L
          }
        }
        working[[li]] <- toks
      }
    }
  }

  # --- Ergebnis zusammenbauen ---
  results <- results[!vapply(results, is.null, logical(1))]

  if (length(results) == 0L) {
    return(data.frame(frequency  = integer(0),
                      length     = integer(0),
                      sign_names = character(0),
                      combination = character(0),
                      stringsAsFactors = FALSE))
  }

  out <- do.call(rbind, results)
  rownames(out) <- NULL
  out <- out[order(-out$length, -out$frequency), ]

  # Keilschrift zusammenfuegen (Leerzeichen entfernen)
  out$combination <- str_replace_all(out$combination, " ", "")

  # Zeichennamen-Spalte ergaenzen
  out$sign_names <- as.character(as.sign_name(out$combination))

  # Spaltenreihenfolge: frequency, length, sign_names, combination
  out <- out[, c("frequency", "length", "sign_names", "combination")]
  out
}
