% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_herit.R
\name{est_herit}
\alias{est_herit}
\title{Estimate heritability with a linear mixed model}
\usage{
est_herit(
  pheno,
  kinship,
  addcovar = NULL,
  weights = NULL,
  reml = TRUE,
  cores = 1,
  ...
)
}
\arguments{
\item{pheno}{A numeric matrix of phenotypes, individuals x phenotypes.}

\item{kinship}{A kinship matrix.}

\item{addcovar}{An optional numeric matrix of additive covariates.}

\item{weights}{An optional numeric vector of positive weights for the
individuals. As with the other inputs, it must have \code{names}
for individual identifiers.}

\item{reml}{If true, use REML; otherwise, use maximimum likelihood.}

\item{cores}{Number of CPU cores to use, for parallel calculations.
(If \code{0}, use \code{\link[parallel:detectCores]{parallel::detectCores()}}.)
Alternatively, this can be links to a set of cluster sockets, as
produced by \code{\link[parallel:makeCluster]{parallel::makeCluster()}}.}

\item{...}{Additional control parameters (see details).}
}
\value{
A vector of estimated heritabilities, corresponding to the
columns in \code{pheno}. The result has attributes \code{"sample_size"},
\code{"log10lik"} and \code{"resid_sd"}.
}
\description{
Estimate the heritability of a set of traits via a linear mixed
model, with possible allowance for covariates.
}
\details{
We fit the model \eqn{y = X \beta + \epsilon}{y = Xb + e} where
\eqn{\epsilon}{e} is multivariate normal with mean 0 and covariance
matrix \eqn{\sigma^2 [h^2 (2 K) + I]}{sigmasq*[hsq*2*K+I]} where
\eqn{K} is the kinship matrix and \eqn{I} is the identity matrix.

If \code{weights} are provided, the covariance matrix becomes
\eqn{\sigma^2 [h^2 (2 K) + D]}{sigmasq*[hsq*2*K+D]} where
\eqn{D} is a diagonal matrix with the reciprocal of the weights.

For each of the inputs, the row names are used as
individual identifiers, to align individuals.

If \code{reml=TRUE}, restricted maximum likelihood (reml) is used
to estimate the heritability, separately for each phenotype.

Additional control parameters include \code{tol} for the tolerance
for convergence, \code{quiet} for controlling whether messages will
be display, \code{max_batch} for the maximum number of phenotypes
in a batch, and \code{check_boundary} for whether the 0 and 1
boundary values for the estimated heritability will be checked
explicitly.
}
\examples{
# read data
iron <- read_cross2(system.file("extdata", "iron.zip", package="qtl2"))
\dontshow{iron <- iron[,c("19","X")] # subset to two chromosomes}

# insert pseudomarkers into map
map <- insert_pseudomarkers(iron$gmap, step=1)

# calculate genotype probabilities
probs <- calc_genoprob(iron, map, error_prob=0.002)

# kinship matrix
kinship <- calc_kinship(probs)

# grab phenotypes and covariates; ensure that covariates have names attribute
pheno <- iron$pheno
covar <- match(iron$covar$sex, c("f", "m")) # make numeric
names(covar) <- rownames(iron$covar)

# perform genome scan
hsq <- est_herit(pheno, kinship, covar)

}
