% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simul.rasch.R
\name{simul.rasch}
\alias{simul.rasch}
\title{Simulate data according to the Rasch model}
\usage{
simul.rasch(persons, items, design = NULL, sum0 = TRUE)
}
\arguments{
\item{persons}{Either a vector of specified person parameters, an integer indicating the number of persons, 
or a list with vectors of specified person parameters or integers corresponding to the
number of bookelts specified in \code{design}.
The number of persons for a test booklet design with unequal number of persons per booklet is specified in a list.}

\item{items}{Either a vector of specified item parameters or an integer indicating the number of items.}

\item{design}{Test booklet design matrix with 0 (item not administered) and 1 (item administered) entries.}

\item{sum0}{If \code{TRUE}, specified item parameters need to be normalized to sum-0.}
}
\value{
Returns a 0-1 matrix according to the Rasch model.
}
\description{
This function simulates data according to the Rasch model 
based on user-specified item and person parameters and an optionally specified test booklet design.
}
\details{
If \code{persons} is an integer value, the corresponding parameter vector 
is drawn from N(0, 1.5). If \code{items} is an integer value, the corresponding parameter vector
is equally spaced between [-3, 3]. Note that item parameters need to be normalized to sum-0. 
This precondition can be overruled using argument \code{sum0 = FALSE}. If \code{!is.null(design)}, 
the integer value represents the number of persons in each booklet. Note that unequal number of persons 
per booklet is specified in a list corresponding to the number of booklets.
}
\examples{
\dontrun{

# simulate Rasch model based data
# 100 persons, 20 items,
# person parameter drawn from a normal distribution: N(0, 1.5)
# item parameters equally spaced between [-3, 3]
simul.rasch(100, items = 20)

# simulate Rasch model based data
# 100 persons, 17 items
# person parameter drawn from a uniform distribution: U[-4, 4]
# item parameters: [-4.0, -3.5, -3.0, ... , 3.0, 3.5, 4.0]
simul.rasch(runif(100, -4, 4), items = seq(-4, 4, by = 0.5))

# specify test booklet design with three items
design <- matrix(c(1, 1, 0, 1, 0, 1, 0, 1, 1), ncol = 3)
# 100 persons in each booklet
# person parameter drawn from a normal distribution: N(0, 1.5)
# item parameters equally spaced between [-3, 3]
simul.rasch(100, items = 3, design = design)

# 100 persons in each booklet
# person parameter drawn from a uniform distribution: U[-4, 4]
# item parameters equally spaced between [-3, 3]
simul.rasch(runif(100, -4, 4), items = 3, design = design)

# 10 persons in booklet 1 / 20 persons in booklet 2 / 30 persons in booklet 3
# person parameter drawn from a normal distribution: N(0, 1.5)
# item parameters equally spaced between [-3, 3]
simul.rasch(list(10, 20, 30), items = 3, design = design)

# 10 persons in booklet 1 / 20 persons in booklet 2 / 30 persons in booklet 3
# person parameter drawn from a uniform distribution: U[-4, 4]
# item parameters equally spaced between [-4, 4]
simul.rasch(list(runif(10, -4, 4), runif(20, -4, 4), runif(30, -4, 4)), 
            items = seq(-4, 4, length.out = 3), design = design)
}
}
\references{
Kubinger, K. D., Rasch, D., & Yanagida, T. (2009). On designing data-sampling for Rasch model 
calibrating an achievement test. \emph{Psychology Science Quarterly, 51}, 370-384.

Kubinger, K. D., Rasch, D., & Yanagida, T. (2011). A new approach for testing the Rasch model.
\emph{Educational Research and Evaluation, 17}, 321-333.
}
\seealso{
\code{\link{aov.rasch}}, \code{\link{pwr.rasch}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at},
Jan Steinfeld \email{jan.steinfeld@univie.ac.at}
}
