#' Construct Piecewise Linear Envelope
#'
#' Builds a piecewise linear envelope around the data using
#' convex hulls and intersection logic similar to the original C++ code.
#'
#' @param data A numeric matrix with two columns: x and f(x) values.
#'   Must have at least two rows.
#' @param smallconst Small constant for numerical tolerance (default 1e-6).
#'
#' @return A list with element 'PWL': a numeric matrix of segments (slope, intercept, lower x-bound, upper x-bound).
#'   Returns NULL if envelope construction fails due to invalid input.
#' @keywords internal
build_pwl_envelope <- function(data, smallconst = 1e-6) {
  if (!is.matrix(data) || ncol(data) != 2 || nrow(data) < 2) {
    warning("Invalid input data: must be a numeric matrix with at least 2 rows and 2 columns (x, y)")
    return(NULL)
  }

  # Helper functions -------------------------------------------------------

  # Orientation check (cross product) - analogous to Angley()
  angley <- function(a, b, c) {
    (c[1] - a[1]) * (b[2] - a[2]) - (c[2] - a[2]) * (b[1] - a[1])
  }

  # Intersection of two lines defined by points a,b and c,d
  intersection <- function(a, b, c, d) {
    grad1 <- if (b[1] == a[1]) Inf else (b[2] - a[2]) / (b[1] - a[1])
    int1 <- a[2] - grad1 * a[1]

    grad2 <- if (d[1] == c[1]) Inf else (d[2] - c[2]) / (d[1] - c[1])
    int2 <- c[2] - grad2 * c[1]

    if (is.infinite(grad1)) {
      x <- a[1]
      y <- grad2 * x + int2
    } else if (is.infinite(grad2)) {
      x <- c[1]
      y <- grad1 * x + int1
    } else {
      x <- (int2 - int1) / (grad1 - grad2)
      y <- grad1 * x + int1
    }
    c(x, y)
  }

  # Find adjacent point to 'point' in CH (convex hull), successor = TRUE for next, FALSE for previous
  adjacent_point <- function(point, CH, successor = TRUE) {
    idx <- which(apply(CH, 1, function(row) all(row == point)))
    if (length(idx) == 0) return(NULL)
    idx <- idx[1]  # Pick first match to avoid length > 1 issues

    if (successor) {
      if (idx == nrow(CH)) return(NULL)
      return(CH[idx + 1, ])
    } else {
      if (idx == 1) return(NULL)
      return(CH[idx - 1, ])
    }
  }

  # -----------------------------------------------------------------------

  # Sort data by x
  data <- data[order(data[, 1]), , drop = FALSE]

  n <- nrow(data)
  DataUB <- cbind(data[,1], data[,2] + smallconst)
  DataLB <- cbind(data[,1], data[,2] - smallconst)

  CHPlus <- matrix(NA_real_, nrow = n, ncol = 2)
  CHMinus <- matrix(NA_real_, nrow = n, ncol = 2)
  CHPlus[1:2, ] <- DataUB[1:2, ]
  CHMinus[1:2, ] <- DataLB[1:2, ]

  PPlus <- DataUB[1, ]
  PMinus <- DataLB[1, ]
  LPlus <- DataUB[1, ]
  LMinus <- DataLB[1, ]
  RPlus <- DataUB[2, ]
  RMinus <- DataLB[2, ]

  Inter <- matrix(NA_real_, nrow = n + 2, ncol = 2) # intersection points
  convexCount <- 0

  for (i in 3:n) {
    NextWindow <- FALSE

    # Update Upper Hull
    P <- DataUB[i - 1, ]
    StartPointPlus <- adjacent_point(P, CHPlus, successor = FALSE)
    while (!is.null(StartPointPlus) && angley(DataUB[i, ], P, StartPointPlus) < 0) {
      P <- StartPointPlus
      if (all(P == PPlus)) break
      StartPointPlus <- adjacent_point(P, CHPlus, successor = FALSE)
    }
    if (!is.null(P)) {
      idxP <- which(apply(CHPlus, 1, function(row) all(row == P)))
      if (length(idxP) > 0 && idxP[1] < n) CHPlus[idxP + 1, ] <- DataUB[i, ]
    }

    # Update Lower Hull
    P <- DataLB[i - 1, ]
    StartPointMinus <- adjacent_point(P, CHMinus, successor = FALSE)
    while (!is.null(StartPointMinus) && angley(DataLB[i, ], P, StartPointMinus) > 0) {
      P <- StartPointMinus
      if (all(P == PMinus)) break
      StartPointMinus <- adjacent_point(P, CHMinus, successor = FALSE)
    }
    if (!is.null(P)) {
      idxP <- which(apply(CHMinus, 1, function(row) all(row == P)))
      if (length(idxP) > 0 && idxP[1] < n) CHMinus[idxP + 1, ] <- DataLB[i, ]
    }

    # Intersection and window updates

    if (angley(DataUB[i, ], LPlus, RMinus) > 0 && !NextWindow) {
      convexCount <- convexCount + 1
      Inter[convexCount, ] <- intersection(LPlus, RMinus, PPlus, PMinus)

      PMinus <- RMinus
      PPlus <- intersection(LPlus, RMinus, DataUB[i - 1, ], DataUB[i, ])

      idxP <- which(apply(CHPlus, 1, function(row) all(row == PPlus)))
      if (length(idxP) > 0 && idxP[1] < n) CHPlus[idxP + 1, ] <- DataUB[i, ]

      RPlus <- DataUB[i, ]
      RMinus <- DataLB[i, ]

      LPlus <- PPlus
      LMinus <- PMinus

      StartPointMinus <- adjacent_point(LMinus, CHMinus, successor = TRUE)
      while (!is.null(StartPointMinus) && angley(LMinus, RPlus, StartPointMinus) < 0) {
        LMinus <- StartPointMinus
        if (all(LMinus == DataLB[i, ])) break
        StartPointMinus <- adjacent_point(LMinus, CHMinus, successor = TRUE)
      }
      NextWindow <- TRUE
    }

    if (angley(DataLB[i, ], LMinus, RPlus) < 0 && !NextWindow) {
      convexCount <- convexCount + 1
      Inter[convexCount, ] <- intersection(LMinus, RPlus, PMinus, PPlus)

      PPlus <- RPlus
      PMinus <- intersection(LMinus, RPlus, DataLB[i - 1, ], DataLB[i, ])

      idxP <- which(apply(CHMinus, 1, function(row) all(row == PMinus)))
      if (length(idxP) > 0 && idxP[1] < n) CHMinus[idxP + 1, ] <- DataLB[i, ]

      RMinus <- DataLB[i, ]
      RPlus <- DataUB[i, ]

      LMinus <- PMinus
      LPlus <- PPlus

      StartPointPlus <- adjacent_point(LPlus, CHPlus, successor = TRUE)
      while (!is.null(StartPointPlus) && angley(LPlus, RMinus, StartPointPlus) > 0) {
        LPlus <- StartPointPlus
        if (all(LPlus == DataUB[i, ])) break
        StartPointPlus <- adjacent_point(LPlus, CHPlus, successor = TRUE)
      }
      NextWindow <- TRUE
    }

    if (!NextWindow) {
      if (angley(DataUB[i, ], LMinus, RPlus) > 0) {
        RPlus <- DataUB[i, ]
        StartPointMinus <- adjacent_point(LMinus, CHMinus, successor = TRUE)
        while (!is.null(StartPointMinus) && angley(DataUB[i, ], LMinus, StartPointMinus) > 0) {
          LMinus <- StartPointMinus
          if (all(LMinus == DataLB[i, ])) break
          StartPointMinus <- adjacent_point(LMinus, CHMinus, successor = TRUE)
        }
      }
      if (angley(DataLB[i, ], LPlus, RMinus) < 0) {
        RMinus <- DataLB[i, ]
        StartPointPlus <- adjacent_point(LPlus, CHPlus, successor = TRUE)
        while (!is.null(StartPointPlus) && angley(DataLB[i, ], LPlus, StartPointPlus) < 0) {
          LPlus <- StartPointPlus
          if (all(LPlus == DataUB[i, ])) break
          StartPointPlus <- adjacent_point(LPlus, CHPlus, successor = TRUE)
        }
      }
    }
  }

  # Final intersections at data end
  convexCount <- convexCount + 1
  Inter[convexCount, ] <- intersection(LMinus, RPlus, PPlus, PMinus)
  convexCount <- convexCount + 1
  Inter[convexCount, ] <- intersection(LMinus, RPlus, DataUB[n, ], DataLB[n, ])

  # Build PWL segments: slope, intercept, lower, upper
  if (convexCount < 2) {
    warning("Not enough intersection points for envelope construction")
    return(NULL)
  }

  PWL <- matrix(NA_real_, nrow = convexCount - 1, ncol = 4)
  for (j in 1:(convexCount - 1)) {
    x1 <- Inter[j, 1]; y1 <- Inter[j, 2]
    x2 <- Inter[j + 1, 1]; y2 <- Inter[j + 1, 2]
    slope <- (y2 - y1) / (x2 - x1)
    intercept <- y1 - slope * x1
    PWL[j, ] <- c(slope, intercept, x1, x2)
  }

  list(PWL = PWL, breakpoints = Inter[1:convexCount, , drop = FALSE])
}
