% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hixson_crowell_model.R
\name{hixson_crowell_model}
\alias{hixson_crowell_model}
\title{Hixson-Crowell Drug Release Kinetic Model}
\arguments{
\item{data}{A data frame containing experimental drug release data.}

\item{time_col}{Character string specifying the column name for time (minutes).}

\item{remain_col}{Character string specifying the column name for cumulative
percent drug remaining.}

\item{group_col}{Optional character string specifying a grouping variable
(e.g., formulation, batch).}

\item{pH_col}{Optional character string specifying a column containing pH values.}

\item{plot}{Logical; if TRUE, generates a plot with fitted Hixson-Crowell lines.}

\item{annotate}{Logical; if TRUE, annotates the plot with kHC, intercept, R^2,
and t50 (only if <= 2 groups).}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{Data frame with Hixson-Crowell parameters
        (kHC, intercept), coefficient of determination (R^2), and t50 for each group.}
  \item{\code{data}}{Processed data used for model fitting and plotting.}
}
}
\description{
Fits experimental cumulative drug release data to the Hixson-Crowell cube-root model,
which describes drug release from systems where dissolution occurs with
a change in surface area and particle diameter over time. The model is
based on a linear regression of \eqn{(W_0^{1/3} - W_t^{1/3})} versus time.

The function assumes that the initial drug amount (\eqn{W_0}) is known
and that the remaining drug amount (\eqn{W_t}) can be calculated from
cumulative percent drug remaining data. It supports optional grouping variables
(e.g., formulation, batch) and optional pH-dependent analysis. The function
generates publication-quality plots with experimental data points, fitted
Hixson-Crowell straight lines, and annotations for the Hixson-Crowell rate
constant (kHC), intercept, coefficient of determination (R^2), and the time
required for 50-percent drug release (t50).

Model:
\deqn{W_0^{1/3} - W_t^{1/3} = k_{HC} * t}

where:
- \eqn{W_0} is the initial amount of drug
- \eqn{W_t} is the remaining amount of drug at time \eqn{t}
- \eqn{k_{HC}} is the Hixson-Crowell dissolution rate constant

The Hixson-Crowell model is applicable to dosage forms where drug release
is governed by erosion or dissolution with a decreasing surface area
(e.g., tablets, pellets).
}
\examples{
# Example I: Single formulation
df <- data.frame(
  time = c(0, 15, 30, 45, 60, 90, 120, 150, 180),
  remain = c(100, 88.6, 79.2, 69.2, 60.2, 42.2, 28.0, 15.2, 6.5)
)
hixson_crowell_model(
  data = df,
  time_col = "time",
  remain_col = "remain"
)

# Example II: Two formulations (grouped, not pH-dependent)
df2 <- data.frame(
  time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
  remain = c(
    100, 90, 81, 72, 63, 54, 45, 36, 27, 18,  # Formulation A
    100, 92, 84, 76, 68, 60, 52, 44, 36, 28   # Formulation B
  ),
  formulation = rep(c("Formulation A", "Formulation B"), each = 10)
)
hixson_crowell_model(
  data = df2,
  time_col = "time",
  remain_col = "remain",
  group_col = "formulation"
)

# Example III: pH-dependent release
df_pH <- data.frame(
  time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
  remain = c(
    100, 90, 80, 70, 60, 50, 40, 30, 20, 10,  # pH 7.4
    100, 92, 84, 76, 68, 60, 52, 44, 36, 28   # pH 4.5
  ),
  pH = rep(c(7.4, 4.5), each = 10)
)
hixson_crowell_model(
  data = df_pH,
  time_col = "time",
  remain_col = "remain",
  pH_col = "pH"
)

# Example IV: Two formulations under two pH conditions
df1 <- data.frame(
  time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
  remain = c(
    100, 88, 75, 62, 50, 38, 28, 18, 10, 5,   # pH 4.5
    100, 90, 78, 65, 52, 40, 30, 20, 12, 6    # pH 7.6
  ),
  pH = rep(c(4.5, 7.6), each = 10)
)
df2 <- data.frame(
  time = rep(c(0, 15, 30, 45, 60, 75, 90, 105, 120, 135), 2),
  remain = c(
    100, 90, 78, 66, 54, 44, 34, 25, 16, 8,   # pH 4.5
    100, 92, 80, 68, 56, 44, 34, 24, 15, 7    # pH 7.6
  ),
  pH = rep(c(4.5, 7.6), each = 10)
)
df_all <- rbind(
  cbind(formulation = "Dataset 1", df1),
  cbind(formulation = "Dataset 2", df2)
)
hixson_crowell_model(
  data = df_all,
  time_col = "time",
  remain_col = "remain",
  group_col = "formulation",
  pH_col = "pH"
)
}
\references{
Hixson, A. W., & Crowell, J. H. (1931) <doi:10.1021/ie50260a018>
Dependence of reaction velocity upon surface and agitation. Industrial & Engineering
Chemistry, 23(8), 923–931.
}
\author{
Paul Angelo C. Manlapaz
}
