% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regularization_net.R
\name{regularization_net}
\alias{regularization_net}
\title{Regularized Network Estimation}
\usage{
regularization_net(
  data = NULL,
  ns = NULL,
  mat = NULL,
  likelihood = "obs_based",
  n_calc = "average",
  count_diagonal = TRUE,
  ic_type = NULL,
  extended_gamma = 0.5,
  penalty = "atan",
  vary = "lambda",
  n_lambda = NULL,
  lambda_min_ratio = 0.01,
  n_gamma = 50,
  pen_diag = FALSE,
  lambda = NULL,
  gamma = NULL,
  ordered = FALSE,
  missing_handling = "two-step-em",
  nimp = 20,
  imp_method = "pmm",
  ...
)
}
\arguments{
\item{data}{Optional raw data matrix or data frame containing the variables
to be included in the network. May include missing values. If \code{data} is not
provided (\code{NULL}), a covariance or correlation matrix must be supplied in \code{mat}.}

\item{ns}{Optional numeric sample size specification. Can be a single value
(one sample size for all variables) or a vector (e.g., variable-wise sample
sizes). When \code{data} is provided and \code{ns} is \code{NULL}, sample sizes are derived
automatically from \code{data}. When \code{mat} is supplied instead of raw data,
\code{ns} must be provided and should reflect the sample size underlying \code{mat}.}

\item{mat}{Optional covariance or correlation matrix for the variables to be
included in the network. Used only when \code{data} is \code{NULL}. If both \code{data} and
\code{mat} are supplied, \code{mat} is ignored. When \code{mat} is used, \code{ns} must also be
provided.}

\item{likelihood}{Character string specifying how the log-likelihood
is computed. Possible values are:
\describe{
\item{\code{"obs_based"}}{Uses the observed-data log-likelihood.}
\item{\code{"mat_based"}}{Uses log-likelihood based on the sample correlation matrix.}
}}

\item{n_calc}{Character string specifying how the effective sample size is
determined. When \code{data} are provided, it controls how the observation counts
across variables are aggregated. When \code{ns} is a vector, it controls how the
entries of \code{ns} are combined. If both \code{data} and \code{ns} are supplied, the
values in \code{ns} take precedence. This argument is ignored when \code{ns} is a
single numeric value. Possible values are:
\describe{
\item{\code{"average"}}{Uses the average sample size across variables or across
the entries of \code{ns}.}
\item{\code{"max"}}{Uses the maximum sample size across variables or across
the entries of \code{ns}.}
\item{\code{"total"}}{Uses the total number of observations. Applicable only when
\code{ns} is not provided by the user.}
}}

\item{count_diagonal}{Logical; should observations contributing to the
diagonal elements be included when computing the sample size? Only relevant
when \code{data} is provided and \code{n_calc = "average"}.}

\item{ic_type}{Character string specifying the type of information criterion
used for model selection. Possible values are: \code{"aic"}, \code{"bic"}, and \code{"ebic"}.
If no input is provided, defaults to \code{"ebic"} when \code{penalty = "glasso"} and
\code{"bic"} otherwise.}

\item{extended_gamma}{Numeric gamma parameter used in the extended information
criterion calculation. Only relevant when \code{ic_type = "ebic"}.}

\item{penalty}{Character string indicating the type of penalty used for
regularization. Available options are described in the Details section.}

\item{vary}{Character string specifying which penalty parameter(s) are varied
during regularization to determine the optimal network. Possible values are
\code{"lambda"}, \code{"gamma"}, or \code{"both"}.}

\item{n_lambda}{Number of lambda values to be evaluated. If not specified,
the default is 100 when \code{penalty = "glasso"} and 50 if \code{lambda} is varied for.
If \code{vary == "gamma"}, \code{n_lambda} is set to 1.}

\item{lambda_min_ratio}{Ratio of the smallest to the largest lambda value.}

\item{n_gamma}{Number of gamma values to be evaluated. Is set to 1
if \code{vary == "lambda"}.}

\item{pen_diag}{Logical; should the diagonal elements be penalized in the
regularization process?}

\item{lambda}{Optional user-specified vector of lambda values.}

\item{gamma}{Optional user-specified vector of gamma values.}

\item{ordered}{Logical vector indicating which variables in \code{data} are
treated as ordered (ordinal). Only used when \code{data} is provided. If a single
logical value is supplied, it is recycled to the length of \code{data}.}

\item{missing_handling}{Character string specifying how correlations are
estimated from the \code{data} input in the presence of missing values. Possible
values are:
\describe{
\item{\code{"two-step-em"}}{Uses a classical EM algorithm to estimate the
correlation matrix from \code{data}.}
\item{\code{"stacked-mi"}}{Uses stacked multiple imputation to estimate the
correlation matrix from \code{data}.}
\item{\code{"pairwise"}}{Uses pairwise deletion to compute correlations from
\code{data}.}
\item{\code{"listwise"}}{Uses listwise deletion to compute correlations from
\code{data}.}
}}

\item{nimp}{Number of imputations (default: 20) to be used when
\code{missing_handling = "stacked-mi"}.}

\item{imp_method}{Character string specifying the imputation method to be
used when \code{missing_handling = "stacked-mi"} (default: \code{"pmm"} - predictive
mean matching).}

\item{...}{Further arguments passed to internal functions.}
}
\value{
A list with the following elements:
\describe{
\item{pcor}{Estimated partial correlation matrix corresponding to the
selected (optimal) network.}
\item{n}{Effective sample size used, either supplied
directly via \code{n} or derived based on \code{n_calc}.}
\item{cor_method}{Correlation estimation method used for each
variable pair.}
\item{full_results}{Full set of results returned by the model selection
procedure, including all evaluated networks and their fit statistics.}
\item{args}{A list of settings used in the estimation procedure.}
}
}
\description{
Estimate cross-sectional network structures using regularization. The function first
computes the correlations (if needed),
constructs a grid of tuning parameters tailored to the chosen penalty,
and then selects the final network by minimizing a user‑specified
information criterion.
}
\details{
\loadmathjax
\strong{Penalties}

This function supports a range of convex and nonconvex penalties for
regularized network estimation.

For convex penalties, the graphical lasso can be used via
\code{penalty = "glasso"} \insertCite{friedman.2008}{mantar}.

Another option is the adaptive lasso, specified with
\code{penalty = "adapt"}.
By default, it employs \mjseqn{\gamma = 0.5} \insertCite{zou.2008}{mantar}.
Smaller values of \mjseqn{\gamma} (i.e., \mjseqn{\gamma \to 0}) correspond
to stronger penalization, whereas \mjseqn{\gamma = 1} yields standard
\mjseqn{\ell_1} regularization.

The available nonconvex penalties follow the work of
\insertCite{williams.2020;textual}{mantar}, who identified the
atan penalty as particularly promising. It serves as the default in this
implementation because it has desirable theoretical properties, including
consistency in recovering the true model as \eqn{n \to \infty}.
Additional nonconvex penalties are included for completeness. These were
originally implemented in the now–deprecated \code{R} package \pkg{GGMncv} \insertCite{williams.2021}{mantar},
and the implementation in \pkg{mantar} is based on the corresponding methods
from that package.

Several algorithms exist for nonconvex regularized network estimation.
In \pkg{mantar}, we use the one-step estimator of \insertCite{zou.2008;textual}{mantar}
because of its computational efficiency and its good performance in settings
where \eqn{n > p}, which is typically the case in psychological research.

\itemize{

\item \strong{Atan}: \code{penalty = "atan"} \insertCite{wang.2016}{mantar}.
This is currently the default.

\item \strong{Exponential}: \code{penalty = "exp"}
\insertCite{wang.2018}{mantar}.

\item \strong{Log}: \code{penalty = "log"}
\insertCite{mazumder.2011}{mantar}.

\item \strong{MCP}: \code{penalty = "mcp"}
\insertCite{zhang.2010}{mantar}.

\item \strong{SCAD}: \code{penalty = "scad"}
\insertCite{fan.2001}{mantar}.

\item \strong{Seamless } \mjseqn{\ell_0}: \code{penalty = "selo"}
\insertCite{dicker.2013}{mantar}.

\item \strong{SICA}: \code{penalty = "sica"}
\insertCite{lv.2009}{mantar}.

}

\strong{Information Criteria}

The argument \code{ic_type} specifies which information criterion is computed.
All criteria are computed based on the log-likelihood of the estimated model.

\describe{

\item{\code{"aic"}:}{
Akaike Information Criterion \insertCite{akaike.1974}{mantar};
defined as
\mjseqn{\mathrm{AIC} = -2 \ell + 2k},
where \eqn{\ell} is the log-likelihood of the model and \eqn{k} is the
number of freely estimated edge parameters (non-zero edges).
}

\item{\code{"bic"}:}{
Bayesian Information Criterion \insertCite{schwarz.1978}{mantar};
defined as
\mjseqn{\mathrm{BIC} = -2 \ell + k \log(n)},
where \eqn{\ell} is the log-likelihood of the model, \eqn{k} is the
number of freely estimated edge parameters (non-zero edges), and \eqn{n} is the sample size.
}

\item{\code{"ebic"}:}{
Extended Bayesian Information Criterion \insertCite{chen.2008}{mantar};
particularly useful in high-dimensional settings. Defined as
\mjseqn{\mathrm{EBIC} = -2 \ell + k \log(n) + 4 \gamma k \log(p)},
where \eqn{\ell} is the log-likelihood, \eqn{k} is the number of freely
estimated edges (non-zero edges), \eqn{n} is the sample size, \eqn{p} is the number of
variables, and \eqn{\gamma} is the extended-penalty parameter.
}

}

\strong{Conditional Defaults}

By default, some tuning parameters depend on the chosen penalty.
Specifically, when \code{penalty = "glasso"}, the number of lambda
values \code{n_lambda} defaults to \code{100} and \code{ic_type}
defaults to \code{"ebic"}. For all other penalties, the defaults are
\code{n_lambda = 50} and \code{ic_type = "bic"}. These defaults can
be overridden by specifying \code{n_lambda} and/or \code{ic_type}
explicitly.

\strong{Missing Handling}

To handle missing data, the function offers two approaches: a two-step expectation-maximization
(EM) algorithm and stacked multiple imputation. According to simulations by \insertCite{nehler.2025;textual}{mantar},
stacked multiple imputation performs reliably across a range of sample sizes. In contrast,
the two-step EM algorithm provides accurate results primarily when the sample size is large relative
to the amount of missingness and network complexity - but may still be preferred in such cases due
to its much faster runtime.
Currently, the function only supports variables that are directly included in the network analysis;
auxiliary variables for missing handling are not yet supported. During imputation, all variables
are imputed by default using predictive mean matching \insertCite{@see e.g., @vanbuuren.2018}{mantar},
with all other variables in the data set serving as predictors.
}
\examples{
# Estimate regularized network from full data set
# Using observed-data loglikelihood and atan penalty
result <- regularization_net(mantar_dummy_full_cont,
                            likelihood = "obs_based",
                            penalty = "atan")

# View estimated partial correlation network
result$pcor

# Estimate regularized network from data set with missings
# Using correlation-matrix-based loglikelihood, glasso penalty,
# and stacked multiple imputation to handle missings
# set nimp to 10 for faster computation to in this example (not recommended
# in practice)
result <- regularization_net(mantar_dummy_mis_mix,
                           likelihood = "mat_based",
                           penalty = "glasso",
                           missing_handling = "stacked-mi",
                           nimp = 10,
                           ordered = c(FALSE,FALSE,TRUE,TRUE,
                                       FALSE,FALSE,TRUE,TRUE))

# View used correlation method and effective sample size
result$cor_method
result$n
# View estimated partial correlation network
result$pcor


}
\references{
\insertAllCited{}
}
