% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/losses.R
\name{loss_categorical_focal_crossentropy}
\alias{loss_categorical_focal_crossentropy}
\title{Computes the alpha balanced focal crossentropy loss.}
\usage{
loss_categorical_focal_crossentropy(
  y_true,
  y_pred,
  alpha = 0.25,
  gamma = 2,
  from_logits = FALSE,
  label_smoothing = 0,
  axis = -1L,
  ...,
  reduction = "sum_over_batch_size",
  name = "categorical_focal_crossentropy",
  dtype = NULL
)
}
\arguments{
\item{y_true}{Tensor of one-hot true targets.}

\item{y_pred}{Tensor of predicted targets.}

\item{alpha}{A weight balancing factor for all classes, default is \code{0.25} as
mentioned in the reference. It can be a list of floats or a scalar.
In the multi-class case, alpha may be set by inverse class
frequency by using \code{compute_class_weight} from \code{sklearn.utils}.}

\item{gamma}{A focusing parameter, default is \code{2.0} as mentioned in the
reference. It helps to gradually reduce the importance given to
simple examples in a smooth manner. When \code{gamma} = 0, there is
no focal effect on the categorical crossentropy.}

\item{from_logits}{Whether \code{output} is expected to be a logits tensor. By
default, we consider that \code{output} encodes a probability
distribution.}

\item{label_smoothing}{Float in \verb{[0, 1].} When > 0, label values are smoothed,
meaning the confidence on label values are relaxed. For example, if
\code{0.1}, use \code{0.1 / num_classes} for non-target labels and
\code{0.9 + 0.1 / num_classes} for target labels.}

\item{axis}{The axis along which to compute crossentropy (the features
axis). Defaults to \code{-1}.}

\item{...}{For forward/backward compatability.}

\item{reduction}{Type of reduction to apply to the loss. In almost all cases
this should be \code{"sum_over_batch_size"}. Supported options are
\code{"sum"}, \code{"sum_over_batch_size"}, \code{"mean"},
\code{"mean_with_sample_weight"} or \code{NULL}. \code{"sum"} sums the loss,
\code{"sum_over_batch_size"} and \code{"mean"} sum the loss and divide by the
sample size, and \code{"mean_with_sample_weight"} sums the loss and
divides by the sum of the sample weights. \code{"none"} and \code{NULL}
perform no aggregation. Defaults to \code{"sum_over_batch_size"}.}

\item{name}{Optional name for the loss instance.}

\item{dtype}{The dtype of the loss's computations. Defaults to \code{NULL}, which
means using \code{config_floatx()}. \code{config_floatx()} is a
\code{"float32"} unless set to different value
(via \code{config_set_floatx()}). If a \code{keras$DTypePolicy} is
provided, then the \code{compute_dtype} will be utilized.}
}
\value{
Categorical focal crossentropy loss value.
}
\description{
Use this crossentropy loss function when there are two or more label
classes and if you want to handle class imbalance without using
\code{class_weights}. We expect labels to be provided in a \code{one_hot}
representation.

According to \href{https://arxiv.org/pdf/1708.02002}{Lin et al., 2018}, it
helps to apply a focal factor to down-weight easy examples and focus more on
hard examples. The general formula for the focal loss (FL)
is as follows:

\code{FL(p_t) = (1 - p_t)^gamma * log(p_t)}

where \code{p_t} is defined as follows:
\verb{p_t = output if y_true == 1, else 1 - output}

\code{(1 - p_t)^gamma} is the \code{modulating_factor}, where \code{gamma} is a focusing
parameter. When \code{gamma} = 0, there is no focal effect on the cross entropy.
\code{gamma} reduces the importance given to simple examples in a smooth manner.

The authors use alpha-balanced variant of focal loss (FL) in the paper:
\code{FL(p_t) = -alpha * (1 - p_t)^gamma * log(p_t)}

where \code{alpha} is the weight factor for the classes. If \code{alpha} = 1, the
loss won't be able to handle class imbalance properly as all
classes will have the same weight. This can be a constant or a list of
constants. If alpha is a list, it must have the same length as the number
of classes.

The formula above can be generalized to:
\code{FL(p_t) = alpha * (1 - p_t)^gamma * CrossEntropy(y_true, y_pred)}

where minus comes from \code{CrossEntropy(y_true, y_pred)} (CE).

Extending this to multi-class case is straightforward:
\code{FL(p_t) = alpha * (1 - p_t) ** gamma * CategoricalCE(y_true, y_pred)}

In the snippet below, there is \code{num_classes} floating pointing values per
example. The shape of both \code{y_pred} and \code{y_true} are
\verb{(batch_size, num_classes)}.
}
\section{Examples}{
\if{html}{\out{<div class="sourceCode r">}}\preformatted{y_true <- rbind(c(0, 1, 0), c(0, 0, 1))
y_pred <- rbind(c(0.05, 0.95, 0), c(0.1, 0.8, 0.1))
loss <- loss_categorical_focal_crossentropy(y_true, y_pred)
loss
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([3.2058331e-05 4.6627346e-01], shape=(2), dtype=float32)

}\if{html}{\out{</div>}}

Standalone usage:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{y_true <- rbind(c(0, 1, 0), c(0, 0, 1))
y_pred <- rbind(c(0.05, 0.95, 0), c(0.1, 0.8, 0.1))
# Using 'auto'/'sum_over_batch_size' reduction type.
cce <- loss_categorical_focal_crossentropy()
cce(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.23315276, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Calling with 'sample_weight'.
cce(y_true, y_pred, sample_weight = op_array(c(0.3, 0.7)))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.16320053, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Using 'sum' reduction type.
cce <- loss_categorical_focal_crossentropy(reduction = "sum")
cce(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(0.46630552, shape=(), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Using 'none' reduction type.
cce <- loss_categorical_focal_crossentropy(reduction = NULL)
cce(y_true, y_pred)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([3.2058331e-05 4.6627346e-01], shape=(2), dtype=float32)

}\if{html}{\out{</div>}}

Usage with the \code{compile()} API:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{model \%>\% compile(
  optimizer = 'adam',
  loss = loss_categorical_focal_crossentropy())
}\if{html}{\out{</div>}}
}

\seealso{
Other losses: \cr
\code{\link{Loss}()} \cr
\code{\link{loss_binary_crossentropy}()} \cr
\code{\link{loss_binary_focal_crossentropy}()} \cr
\code{\link{loss_categorical_crossentropy}()} \cr
\code{\link{loss_categorical_generalized_cross_entropy}()} \cr
\code{\link{loss_categorical_hinge}()} \cr
\code{\link{loss_circle}()} \cr
\code{\link{loss_cosine_similarity}()} \cr
\code{\link{loss_ctc}()} \cr
\code{\link{loss_dice}()} \cr
\code{\link{loss_hinge}()} \cr
\code{\link{loss_huber}()} \cr
\code{\link{loss_kl_divergence}()} \cr
\code{\link{loss_log_cosh}()} \cr
\code{\link{loss_mean_absolute_error}()} \cr
\code{\link{loss_mean_absolute_percentage_error}()} \cr
\code{\link{loss_mean_squared_error}()} \cr
\code{\link{loss_mean_squared_logarithmic_error}()} \cr
\code{\link{loss_poisson}()} \cr
\code{\link{loss_sparse_categorical_crossentropy}()} \cr
\code{\link{loss_squared_hinge}()} \cr
\code{\link{loss_tversky}()} \cr
\code{\link{metric_binary_crossentropy}()} \cr
\code{\link{metric_binary_focal_crossentropy}()} \cr
\code{\link{metric_categorical_crossentropy}()} \cr
\code{\link{metric_categorical_focal_crossentropy}()} \cr
\code{\link{metric_categorical_hinge}()} \cr
\code{\link{metric_hinge}()} \cr
\code{\link{metric_huber}()} \cr
\code{\link{metric_kl_divergence}()} \cr
\code{\link{metric_log_cosh}()} \cr
\code{\link{metric_mean_absolute_error}()} \cr
\code{\link{metric_mean_absolute_percentage_error}()} \cr
\code{\link{metric_mean_squared_error}()} \cr
\code{\link{metric_mean_squared_logarithmic_error}()} \cr
\code{\link{metric_poisson}()} \cr
\code{\link{metric_sparse_categorical_crossentropy}()} \cr
\code{\link{metric_squared_hinge}()} \cr
}
\concept{losses}
