/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5TSmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5TSpkg.h"    

#ifdef H5_HAVE_THREADS

#ifdef H5_HAVE_C11_THREADS

herr_t
H5TS_key_create(H5TS_key_t *key, H5TS_key_destructor_func_t dtor)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == key))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(tss_create(key, dtor) != thrd_success))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_key_delete(H5TS_key_t key)
{
    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    
    tss_delete(key);

    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(SUCCEED)
} 

#else
#ifdef H5_HAVE_WIN_THREADS

herr_t
H5TS_key_create(H5TS_key_t *key, H5TS_key_destructor_func_t dtor)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == key))
        HGOTO_DONE(FAIL);

    
    if (NULL != dtor)
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(TLS_OUT_OF_INDEXES == (*key = TlsAlloc())))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_key_delete(H5TS_key_t key)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (TLS_OUT_OF_INDEXES != key)
        if (H5_UNLIKELY(0 == TlsFree(key)))
            HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#else

herr_t
H5TS_key_create(H5TS_key_t *key, H5TS_key_destructor_func_t dtor)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == key))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(pthread_key_create(key, dtor)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_key_delete(H5TS_key_t key)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(pthread_key_delete(key)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

#endif
#endif

#endif 
