/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5TSmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5TSpkg.h"    

#ifdef H5_HAVE_THREADS

#ifdef H5_HAVE_C11_THREADS

herr_t
H5TS_rwlock_init(H5TS_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(mtx_init(&lock->mutex, mtx_plain) != thrd_success))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(cnd_init(&lock->read_cv) != thrd_success))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(cnd_init(&lock->write_cv) != thrd_success))
        HGOTO_DONE(FAIL);

    
    lock->readers       = 0;
    lock->writers       = 0;
    lock->read_waiters  = 0;
    lock->write_waiters = 0;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_rwlock_destroy(H5TS_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    
    mtx_destroy(&lock->mutex);
    cnd_destroy(&lock->read_cv);
    cnd_destroy(&lock->write_cv);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 
#else
#ifdef H5_HAVE_WIN_THREADS

herr_t
H5TS_rwlock_init(H5TS_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    InitializeSRWLock(lock);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_rwlock_destroy(H5TS_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 
#elif defined(__MACH__)

herr_t
H5TS_rwlock_init(H5TS_rwlock_t *lock)
{
    pthread_mutexattr_t  _attr;
    pthread_mutexattr_t *attr      = NULL;
    herr_t               ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(pthread_mutexattr_init(&_attr)))
        HGOTO_DONE(FAIL);
    attr = &_attr;

    
    if (H5_UNLIKELY(pthread_mutexattr_settype(attr, PTHREAD_MUTEX_NORMAL)))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(pthread_mutex_init(&lock->mutex, attr)))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(pthread_cond_init(&lock->read_cv, NULL)))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(pthread_cond_init(&lock->write_cv, NULL)))
        HGOTO_DONE(FAIL);

    
    lock->readers       = 0;
    lock->writers       = 0;
    lock->read_waiters  = 0;
    lock->write_waiters = 0;

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_rwlock_destroy(H5TS_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    
    if (H5_UNLIKELY(pthread_mutex_destroy(&lock->mutex)))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(pthread_cond_destroy(&lock->read_cv)))
        HGOTO_DONE(FAIL);
    if (H5_UNLIKELY(pthread_cond_destroy(&lock->write_cv)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 
#else

herr_t
H5TS_rwlock_init(H5TS_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    if (H5_UNLIKELY(pthread_rwlock_init(lock, NULL)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 

herr_t
H5TS_rwlock_destroy(H5TS_rwlock_t *lock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_NOAPI_NAMECHECK_ONLY

    
    if (H5_UNLIKELY(NULL == lock))
        HGOTO_DONE(FAIL);

    if (H5_UNLIKELY(pthread_rwlock_destroy(lock)))
        HGOTO_DONE(FAIL);

done:
    FUNC_LEAVE_NOAPI_NAMECHECK_ONLY(ret_value)
} 
#endif
#endif

#endif 
