/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5L_FRIEND     
#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5Lpkg.h"      
#include "H5MMprivate.h" 
#include "H5Opkg.h"      

static void *H5O__link_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags, size_t p_size,
                              const uint8_t *p);
static herr_t H5O__link_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                               const void *_mesg);
static void  *H5O__link_copy(const void *_mesg, void *_dest);
static size_t H5O__link_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__link_reset(void *_mesg);
static herr_t H5O__link_free(void *_mesg);
static herr_t H5O__link_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                      const H5O_copy_t *cpy_info, void *udata);
static void  *H5O__link_copy_file(H5F_t *file_src, void *native_src, H5F_t *file_dst, bool *recompute_size,
                                  unsigned *mesg_flags, H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__link_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc,
                                       void *mesg_dst, unsigned *mesg_flags, H5O_copy_t *cpy_info);
static herr_t H5O__link_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_LINK[1] = {{
    H5O_LINK_ID,              
    "link",                   
    sizeof(H5O_link_t),       
    0,                        
    H5O__link_decode,         
    H5O__link_encode,         
    H5O__link_copy,           
    H5O__link_size,           
    H5O__link_reset,          
    H5O__link_free,           
    H5O_link_delete,          
    NULL,                     
    NULL,                     
    NULL,                     
    H5O__link_pre_copy_file,  
    H5O__link_copy_file,      
    H5O__link_post_copy_file, 
    NULL,                     
    NULL,                     
    H5O__link_debug           
}};

#define H5O_LINK_VERSION 1

#define H5O_LINK_NAME_SIZE       0x03 
#define H5O_LINK_STORE_CORDER    0x04 
#define H5O_LINK_STORE_LINK_TYPE 0x08 
#define H5O_LINK_STORE_NAME_CSET 0x10 
#define H5O_LINK_ALL_FLAGS                                                                                   \
    (H5O_LINK_NAME_SIZE | H5O_LINK_STORE_CORDER | H5O_LINK_STORE_LINK_TYPE | H5O_LINK_STORE_NAME_CSET)

#define H5O_LINK_NAME_1 0x00 
#define H5O_LINK_NAME_2 0x01 
#define H5O_LINK_NAME_4 0x02 
#define H5O_LINK_NAME_8 0x03 

H5FL_DEFINE_STATIC(H5O_link_t);

static void *
H5O__link_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                 unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    H5O_link_t    *lnk = NULL;                 
    size_t         len = 0;                    
    unsigned char  link_flags;                 
    const uint8_t *p_end     = p + p_size - 1; 
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*p++ != H5O_LINK_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message");

    
    if (NULL == (lnk = H5FL_CALLOC(H5O_link_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    link_flags = *p++;
    if (link_flags & ~H5O_LINK_ALL_FLAGS)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad flag value for message");

    
    if (link_flags & H5O_LINK_STORE_LINK_TYPE) {
        
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        lnk->type = (H5L_type_t)*p++;
        if (lnk->type < H5L_TYPE_HARD || lnk->type > H5L_TYPE_MAX)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad link type");
    }
    else
        lnk->type = H5L_TYPE_HARD;

    
    if (link_flags & H5O_LINK_STORE_CORDER) {
        if (H5_IS_BUFFER_OVERFLOW(p, 8, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        INT64DECODE(p, lnk->corder);
        lnk->corder_valid = true;
    }
    else {
        lnk->corder       = 0;
        lnk->corder_valid = false;
    }

    
    if (link_flags & H5O_LINK_STORE_NAME_CSET) {
        
        if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        lnk->cset = (H5T_cset_t)*p++;
        if (lnk->cset < H5T_CSET_ASCII || lnk->cset > H5T_CSET_UTF8)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad cset type");
    }
    else
        lnk->cset = H5T_CSET_ASCII;

    
    switch (link_flags & H5O_LINK_NAME_SIZE) {
        case 0: 
            if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            len = *p++;
            break;

        case 1: 
            if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT16DECODE(p, len);
            break;

        case 2: 
            if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT32DECODE(p, len);
            break;

        case 3: 
            if (H5_IS_BUFFER_OVERFLOW(p, 8, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT64DECODE(p, len);
            break;

        default:
            HGOTO_ERROR(H5E_OHDR, H5E_BADVALUE, NULL, "no appropriate size for name length");
    }
    if (len == 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "invalid name length");

    
    if (H5_IS_BUFFER_OVERFLOW(p, len, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (NULL == (lnk->name = (char *)H5MM_malloc(len + 1)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
    H5MM_memcpy(lnk->name, p, len);
    lnk->name[len] = '\0';
    p += len;

    
    switch (lnk->type) {
        case H5L_TYPE_HARD:
            
            if (H5_IS_BUFFER_OVERFLOW(p, H5F_sizeof_addr(f), p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            H5F_addr_decode(f, &p, &(lnk->u.hard.addr));
            break;

        case H5L_TYPE_SOFT:
            
            if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT16DECODE(p, len);
            if (len == 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "invalid link length");

            if (H5_IS_BUFFER_OVERFLOW(p, len, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            if (NULL == (lnk->u.soft.name = (char *)H5MM_malloc((size_t)len + 1)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
            H5MM_memcpy(lnk->u.soft.name, p, len);
            lnk->u.soft.name[len] = '\0';
            p += len;
            break;

        
        case H5L_TYPE_EXTERNAL:
        case H5L_TYPE_ERROR:
        case H5L_TYPE_MAX:
        default:
            if (lnk->type < H5L_TYPE_UD_MIN || lnk->type > H5L_TYPE_MAX)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "unknown link type");

            
            if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
            UINT16DECODE(p, len);
            if (lnk->type == H5L_TYPE_EXTERNAL && len < 3)
                HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "external link information length < 3");
            lnk->u.ud.size = len;
            if (len > 0) {
                if (H5_IS_BUFFER_OVERFLOW(p, len, p_end))
                    HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
                if (NULL == (lnk->u.ud.udata = H5MM_malloc((size_t)len)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
                H5MM_memcpy(lnk->u.ud.udata, p, len);
                p += len;
            }
            else
                lnk->u.ud.udata = NULL;
    }

    
    ret_value = lnk;

done:
    if (!ret_value && lnk) {
        H5MM_xfree(lnk->name);
        if (lnk->type == H5L_TYPE_SOFT && lnk->u.soft.name != NULL)
            H5MM_xfree(lnk->u.soft.name);
        if (lnk->type >= H5L_TYPE_UD_MIN && lnk->u.ud.size > 0 && lnk->u.ud.udata != NULL)
            H5MM_xfree(lnk->u.ud.udata);
        H5FL_FREE(H5O_link_t, lnk);
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__link_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                 const void *_mesg)
{
    const H5O_link_t *lnk = (const H5O_link_t *)_mesg;
    uint64_t          len;        
    unsigned char     link_flags; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(lnk);

    
    len = (uint64_t)strlen(lnk->name);
    assert(len > 0);

    
    *p++ = H5O_LINK_VERSION;

    
    if (len > 4294967295)
        link_flags = H5O_LINK_NAME_8;
    else if (len > 65535)
        link_flags = H5O_LINK_NAME_4;
    else if (len > 255)
        link_flags = H5O_LINK_NAME_2;
    else
        link_flags = H5O_LINK_NAME_1;
    link_flags = (unsigned char)(link_flags | (lnk->corder_valid ? H5O_LINK_STORE_CORDER : 0));
    link_flags = (unsigned char)(link_flags | ((lnk->type != H5L_TYPE_HARD) ? H5O_LINK_STORE_LINK_TYPE : 0));
    link_flags = (unsigned char)(link_flags | ((lnk->cset != H5T_CSET_ASCII) ? H5O_LINK_STORE_NAME_CSET : 0));
    *p++       = link_flags;

    
    if (link_flags & H5O_LINK_STORE_LINK_TYPE)
        *p++ = (uint8_t)lnk->type;

    
    if (lnk->corder_valid)
        INT64ENCODE(p, lnk->corder);

    
    if (link_flags & H5O_LINK_STORE_NAME_CSET)
        *p++ = (uint8_t)lnk->cset;

    
    switch (link_flags & H5O_LINK_NAME_SIZE) {
        case 0: 
            *p++ = (uint8_t)len;
            break;

        case 1: 
            UINT16ENCODE(p, len);
            break;

        case 2: 
            UINT32ENCODE(p, len);
            break;

        case 3: 
            UINT64ENCODE(p, len);
            break;

        default:
            assert(0 && "bad size for name");
    } 

    
    H5MM_memcpy(p, lnk->name, (size_t)len);
    p += len;

    
    switch (lnk->type) {
        case H5L_TYPE_HARD:
            
            H5F_addr_encode(f, &p, lnk->u.hard.addr);
            break;

        case H5L_TYPE_SOFT:
            
            len = (uint16_t)strlen(lnk->u.soft.name);
            assert(len > 0);
            UINT16ENCODE(p, len);
            H5MM_memcpy(p, lnk->u.soft.name, (size_t)len);
            p += len;
            break;

        
        case H5L_TYPE_EXTERNAL:
        case H5L_TYPE_ERROR:
        case H5L_TYPE_MAX:
        default:
            assert(lnk->type >= H5L_TYPE_UD_MIN && lnk->type <= H5L_TYPE_MAX);

            
            len = (uint16_t)lnk->u.ud.size;
            UINT16ENCODE(p, len);
            if (len > 0) {
                H5MM_memcpy(p, lnk->u.ud.udata, (size_t)len);
                p += len;
            }
            break;
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__link_copy(const void *_mesg, void *_dest)
{
    const H5O_link_t *lnk       = (const H5O_link_t *)_mesg;
    H5O_link_t       *dest      = (H5O_link_t *)_dest;
    void             *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    if (!dest && NULL == (dest = H5FL_MALLOC(H5O_link_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *lnk;

    
    assert(lnk->name);
    if (NULL == (dest->name = H5MM_xstrdup(lnk->name)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "can't duplicate link name");

    
    if (lnk->type == H5L_TYPE_SOFT) {
        if (NULL == (dest->u.soft.name = H5MM_xstrdup(lnk->u.soft.name)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "can't duplicate soft link value");
    } 
    else if (lnk->type >= H5L_TYPE_UD_MIN) {
        if (lnk->u.ud.size > 0) {
            if (NULL == (dest->u.ud.udata = H5MM_malloc(lnk->u.ud.size)))
                HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");
            H5MM_memcpy(dest->u.ud.udata, lnk->u.ud.udata, lnk->u.ud.size);
        } 
    }     

    
    ret_value = dest;

done:
    if (NULL == ret_value)
        if (dest) {
            if (dest->name && dest->name != lnk->name)
                dest->name = (char *)H5MM_xfree(dest->name);
            if (NULL == _dest)
                dest = H5FL_FREE(H5O_link_t, dest);
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__link_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_link_t *lnk = (const H5O_link_t *)_mesg;
    uint64_t          name_len;      
    size_t            name_size;     
    size_t            ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(uint64_t) >= sizeof(size_t));

    
    name_len = (uint64_t)strlen(lnk->name);

    
    if (name_len > 4294967295)
        name_size = 8;
    else if (name_len > 65535)
        name_size = 4;
    else if (name_len > 255)
        name_size = 2;
    else
        name_size = 1;

    
    ret_value = 1 +                                            
                1 +                                            
                (lnk->type != H5L_TYPE_HARD ? (size_t)1 : 0) + 
                (lnk->corder_valid ? 8 : 0) +                  
                (lnk->cset != H5T_CSET_ASCII ? 1 : 0) +        
                name_size +                                    
                name_len;                                      

    
    switch (lnk->type) {
        case H5L_TYPE_HARD:
            ret_value += H5F_SIZEOF_ADDR(f);
            break;

        case H5L_TYPE_SOFT:
            ret_value += 2 +                       
                         strlen(lnk->u.soft.name); 
            break;

        case H5L_TYPE_ERROR:
        case H5L_TYPE_EXTERNAL:
        case H5L_TYPE_MAX:
        default: 
            assert(lnk->type >= H5L_TYPE_UD_MIN);
            ret_value += 2 +             
                         lnk->u.ud.size; 
            break;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__link_reset(void *_mesg)
{
    H5O_link_t *lnk = (H5O_link_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    if (lnk) {
        
        if (lnk->type == H5L_TYPE_SOFT)
            lnk->u.soft.name = (char *)H5MM_xfree(lnk->u.soft.name);
        else if (lnk->type >= H5L_TYPE_UD_MIN) {
            if (lnk->u.ud.size > 0)
                lnk->u.ud.udata = H5MM_xfree(lnk->u.ud.udata);
        } 
        lnk->name = (char *)H5MM_xfree(lnk->name);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__link_free(void *_mesg)
{
    H5O_link_t *lnk = (H5O_link_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    assert(lnk);

    lnk = H5FL_FREE(H5O_link_t, lnk);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5O_link_delete(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, void *_mesg)
{
    H5O_link_t *lnk       = (H5O_link_t *)_mesg;
    hid_t       file_id   = -1;      
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(f);
    assert(lnk);

    
    
    if (lnk->type == H5L_TYPE_HARD) {
        H5O_loc_t oloc;

        
        H5O_loc_reset(&oloc);
        oloc.file = f;
        assert(H5_addr_defined(lnk->u.hard.addr));
        oloc.addr = lnk->u.hard.addr;

        
        if (H5O_link(&oloc, -1) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to decrement object link count");
    } 
    
    else if (lnk->type >= H5L_TYPE_UD_MIN) {
        const H5L_class_t *link_class; 

        
        if (NULL == (link_class = H5L_find_class(lnk->type)))
            HGOTO_ERROR(H5E_OHDR, H5E_NOTREGISTERED, FAIL, "link class not registered");

        
        if (link_class->del_func) {
            
            if ((file_id = H5F_get_id(f)) < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CANTGET, FAIL, "unable to get file ID");

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    
                    ret_value = (link_class->del_func)(lnk->name, file_id, lnk->u.ud.udata, lnk->u.ud.size);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_OHDR, H5E_CALLBACK, FAIL, "link deletion callback returned failure");
        } 
    }     

done:
    
    if (file_id > 0 && H5I_dec_ref(file_id) < 0)
        HDONE_ERROR(H5E_OHDR, H5E_CANTCLOSEFILE, FAIL, "can't close file");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__link_pre_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const void H5_ATTR_UNUSED *native_src, bool *deleted,
                        const H5O_copy_t *cpy_info, void H5_ATTR_UNUSED *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(deleted);
    assert(cpy_info);

    
    if (cpy_info->max_depth >= 0 && cpy_info->curr_depth >= cpy_info->max_depth)
        *deleted = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__link_copy_file(H5F_t H5_ATTR_UNUSED *file_src, void *native_src, H5F_t H5_ATTR_UNUSED *file_dst,
                    bool H5_ATTR_UNUSED *recompute_size, unsigned H5_ATTR_UNUSED *mesg_flags,
                    H5O_copy_t H5_ATTR_UNUSED *cpy_info, void H5_ATTR_UNUSED *udata)
{
    H5O_link_t *link_src  = (H5O_link_t *)native_src;
    void       *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(link_src);
    assert(cpy_info);
    assert(cpy_info->max_depth < 0 || cpy_info->curr_depth < cpy_info->max_depth);

    
    if (link_src->type > H5L_TYPE_SOFT && link_src->type < H5L_TYPE_UD_MIN)
        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, NULL, "unrecognized built-in link type");

    
    
    if (NULL == (ret_value = H5FL_CALLOC(H5O_link_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__link_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc, void *mesg_dst,
                         unsigned H5_ATTR_UNUSED *mesg_flags, H5O_copy_t *cpy_info)
{
    const H5O_link_t *link_src  = (const H5O_link_t *)mesg_src;
    H5O_link_t       *link_dst  = (H5O_link_t *)mesg_dst;
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(link_src);
    assert(dst_oloc);
    assert(H5_addr_defined(dst_oloc->addr));
    assert(dst_oloc->file);
    assert(link_dst);
    assert(cpy_info);
    assert(cpy_info->max_depth < 0 || cpy_info->curr_depth < cpy_info->max_depth);

    
    if (H5L__link_copy_file(dst_oloc->file, link_src, src_oloc, link_dst, cpy_info) < 0)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTCOPY, FAIL, "unable to copy link");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__link_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_link_t *lnk       = (const H5O_link_t *)_mesg;
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(lnk);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Link Type:",
            (lnk->type == H5L_TYPE_HARD
                 ? "Hard"
                 : (lnk->type == H5L_TYPE_SOFT
                        ? "Soft"
                        : (lnk->type == H5L_TYPE_EXTERNAL
                               ? "External"
                               : (lnk->type >= H5L_TYPE_UD_MIN ? "User-defined" : "Unknown")))));

    if (lnk->corder_valid)
        Rfprintf(stream, "%*s%-*s %" PRId64 "\n", indent, "", fwidth, "Creation Order:", lnk->corder);

    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "Link Name Character Set:",
            (lnk->cset == H5T_CSET_ASCII ? "ASCII" : (lnk->cset == H5T_CSET_UTF8 ? "UTF-8" : "Unknown")));
    Rfprintf(stream, "%*s%-*s '%s'\n", indent, "", fwidth, "Link Name:", lnk->name);

    
    switch (lnk->type) {
        case H5L_TYPE_HARD:
            Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
                    "Object address:", lnk->u.hard.addr);
            break;

        case H5L_TYPE_SOFT:
            Rfprintf(stream, "%*s%-*s '%s'\n", indent, "", fwidth, "Link Value:", lnk->u.soft.name);
            break;

        case H5L_TYPE_ERROR:
        case H5L_TYPE_EXTERNAL:
        case H5L_TYPE_MAX:
        default:
            if (lnk->type >= H5L_TYPE_UD_MIN) {
                if (lnk->type == H5L_TYPE_EXTERNAL) {
                    const char *objname =
                        (const char *)lnk->u.ud.udata + (strlen((const char *)lnk->u.ud.udata) + 1);

                    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
                            "External File Name:", (const char *)lnk->u.ud.udata);
                    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth, "External Object Name:", objname);
                } 
                else {
                    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth,
                            "User-Defined Link Size:", (unsigned long long)lnk->u.ud.size);
                } 
            }     
            else
                HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "unrecognized link type");
            break;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
