% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smap.R
\name{smap}
\alias{smap}
\alias{smap_vec}
\alias{smap_lgl}
\alias{smap_int}
\alias{smap_dbl}
\alias{smap_chr}
\alias{smap_structure}
\title{Map Functions Over Glycan Structure Vectors}
\usage{
smap(.x, .f, ..., .parallel = FALSE)

smap_vec(.x, .f, ..., .ptype = NULL, .parallel = FALSE)

smap_lgl(.x, .f, ..., .parallel = FALSE)

smap_int(.x, .f, ..., .parallel = FALSE)

smap_dbl(.x, .f, ..., .parallel = FALSE)

smap_chr(.x, .f, ..., .parallel = FALSE)

smap_structure(.x, .f, ..., .parallel = FALSE)
}
\arguments{
\item{.x}{A glycan structure vector (glyrepr_structure).}

\item{.f}{A function that takes an igraph object and returns a result.
Can be a function, purrr-style lambda (\code{~ .x$attr}), or a character string naming a function.}

\item{...}{Additional arguments passed to \code{.f}.}

\item{.parallel}{Logical; whether to use parallel processing. If \code{FALSE} (default),
parallel processing is disabled. Set to \code{TRUE} to enable parallel processing.}

\item{.ptype}{A prototype for the return type (for \code{smap_vec}).}
}
\value{
\itemize{
\item \code{smap()}: A list
\item \code{smap_vec()}: An atomic vector of type specified by \code{.ptype}
\item \code{smap_lgl/int/dbl/chr()}: Atomic vectors of the corresponding type
\item \code{smap_structure()}: A new glyrepr_structure object
}
}
\description{
These functions apply a function to each unique structure in a glycan structure vector,
taking advantage of hash-based deduplication to avoid redundant computation.
Similar to purrr mapping functions, but optimized for glycan structure vectors.
}
\details{
These functions only compute \code{.f} once for each unique structure, then map
the results back to the original vector positions. This is much more efficient
than applying \code{.f} to each element individually when there are duplicate structures.

\strong{Return Types:}
\itemize{
\item \code{smap()}: Returns a list with the same length as \code{.x}
\item \code{smap_vec()}: Returns an atomic vector with the same length as \code{.x}
\item \code{smap_lgl()}: Returns a logical vector
\item \code{smap_int()}: Returns an integer vector
\item \code{smap_dbl()}: Returns a double vector
\item \code{smap_chr()}: Returns a character vector
\item \code{smap_structure()}: Returns a new glycan structure vector (\code{.f} must return igraph objects)
}
}
\examples{
# Create a structure vector with duplicates
core1 <- o_glycan_core_1()
core2 <- n_glycan_core()
structures <- c(core1, core2, core1)  # core1 appears twice

# Map a function that counts vertices - only computed twice, not three times
smap_int(structures, igraph::vcount)

# Map a function that returns logical
smap_lgl(structures, function(g) igraph::vcount(g) > 5)

# Use purrr-style lambda functions  
smap_int(structures, ~ igraph::vcount(.x))
smap_lgl(structures, ~ igraph::vcount(.x) > 5)

# Map a function that modifies structure (must return igraph)
add_vertex_names <- function(g) {
  if (!("name" \%in\% igraph::vertex_attr_names(g))) {
    igraph::set_vertex_attr(g, "name", value = paste0("v", seq_len(igraph::vcount(g))))
  } else {
    g
  }
}
smap_structure(structures, add_vertex_names)

}
