#' Position scales for discrete distributions
#' 
#' These scales allow for discrete distributions to be passed to the x and y position by mapping distribution objects
#' to discrete aesthetics.
#' These scale can be used similarly to the scale_*_discrete functions.
#' If you want to transform your scale, you should apply a transformation through the coord_* functions,
#' as they are applied after the stat, so the existing ggplot infastructure can be used.
#' 
#' @inheritParams ggplot2::scale_x_discrete
#' @inheritParams ggplot2::scale_y_discrete
#' @inheritParams scales::train_discrete
#' @returns A ggplot2 scale
#' @examples
#' library(ggplot2)
#' # ggplot
#' ggplot(smaller_diamonds, aes(x = cut, y = clarity)) +
#'  geom_count(aes(size = after_stat(prop)))
#' # ggdibbler
#' ggplot(smaller_uncertain_diamonds, aes(x = cut, y = clarity)) + 
#'  geom_count_sample(aes(size = after_stat(prop)), times=10, alpha=0.1)
#' @name scale_discrete_distribution
NULL

#' @export
#' @importFrom ggplot2 waiver 
#' @importFrom scales DiscreteRange
#' @rdname scale_discrete_distribution
scale_x_discrete_distribution <- function(
    name = waiver(), 
    palette = seq_len,
    expand = waiver(),
    guide = waiver(), 
    position = "bottom", 
    sec.axis = waiver(),
    continuous.limits = NULL,
    drop = TRUE,
    ...
) {
  sc <- discrete_distribution_scale(
    aesthetics = ggplot_global$x_aes, 
    name = name,
    palette = palette, 
    drop = drop,
    ...,
    expand = expand, 
    guide = guide, 
    position = position,
    continuous.limits = continuous.limits
  )
}


#' @export
#' @importFrom ggplot2 waiver ScaleDiscretePosition discrete_scale ggproto_parent
#' @importFrom distributional parameters generate is_distribution
#' @rdname scale_discrete_distribution
scale_y_discrete_distribution <- function(
    name = waiver(), 
    palette = seq_len,
    expand = waiver(),
    guide = waiver(), 
    position = "left", 
    sec.axis = waiver(),
    continuous.limits = NULL,
    drop = TRUE,
    ...
) {
  sc <- discrete_distribution_scale(
    aesthetics = ggplot_global$y_aes, 
    name = name,
    palette = palette, 
    drop = drop,
    ...,
    expand = expand, 
    guide = guide, 
    position = position,
    continuous.limits = continuous.limits
  )
}



#' @keywords internal
discrete_distribution_scale <- function(
    aesthetics,
    palette,
    continuous.limits = NULL,
    call = rlang::caller_call(),
    guide = "legend",
    drop = TRUE,
    ...) {
  
  # x/y position aesthetics should use ScaleDiscreteDistributionPosition; others use ScaleDiscrete
  if (all(aesthetics %in% c(ggplot_global$x_aes, ggplot_global$y_aes))) {
    scale_class <- ScaleDiscreteDistributionPosition
  } else {
    scale_class <- ggplot2::ScaleDiscretePosition
  }
  
  sc <- ggplot2::discrete_scale(
    super = scale_class,
    aesthetics= aesthetics,
    palette = palette,
    guide = guide,
    call = call,
    drop = drop,
    ...
  )
  sc$range_c <- ContinuousDistributionRange$new()
  sc$continuous_limits <- continuous.limits
  sc
}


#' @keywords internal
DiscreteDistributionRange <- R6::R6Class(
  "DiscreteDistributionRange",
  inherit = scales::DiscreteRange,
  list(
    factor = NULL,
    train = function(x, drop = FALSE, na.rm = FALSE, call = rlang::caller_env()) {
      self$factor <- ifelse(distributional::is_distribution(new),
                            self$factor %||% is.factor(unlist(generate(x,1))),
                            self$factor %||% is.factor(x)) 
      self$range <- train_discrete_distribution(
        x,
        self$range,
        drop,
        na.rm,
        self$factor,
        call = call
      )
    },
    reset = function() {
      self$range <- NULL
      self$factor <- NULL
    }
  )
)

#' @keywords internal
train_discrete_distribution <- function(
    new,
    existing = NULL,
    drop = FALSE,
    na.rm = FALSE,
    fct = NA,
    call = rlang::caller_env()
    ) {
  if (is.null(new)) {
    return(existing)
  }
  if(distributional::is_distribution(new)){
    # make distribution into output
    new <- unlist(distributional::generate(new, 100))
  }
  if (!is_discrete(new)) {
    example <- unique(new)
    example <- example[seq_len(pmin(length(example), 5))]
    cli::cli_abort(
      c(
        "Continuous value supplied to a discrete scale.",
        i = "Example values: {.and {.val {example}}}."
      ),
      call = call
    )
  }
    discrete_range(existing, new, drop = drop, na.rm = na.rm, fct = fct)

}


#' @keywords internal
ScaleDiscreteDistributionPosition <- ggproto( 
  "ScaleDiscreteDistributionPosition", 
  ggplot2::ScaleDiscretePosition,
  
  continuous_limits = NULL,
  
  # Train
  train = function(self, x) {
    if (ifelse(distributional::is_distribution(x), is_discrete(unlist(generate(x,1))),
               is_discrete(x))) {
      self$range$train(x, drop = self$drop, na.rm = !self$na.translate)
    } else {
      self$range_c$train(x)
    }
  },
  
  # Range
  range = DiscreteDistributionRange$new(),
  
  # included this
  clone = function(self) {
    new <- ggproto(NULL, self)
    new$range <- DiscreteDistributionRange$new()
    new
  },
  
  # Map
  # input: x = dist vector, limits = dist vector as limits
  # output:  a vector of mapped values in aesthetics space.
  map = function(self, x, limits = self$get_limits()) {
    if (distributional::is_distribution(x)){
      disc_to_int <- function(x) ggplot2::ggproto_parent(ggplot2::ScaleDiscretePosition, self)$map(x)
      disc_levels <- distributional::parameters(x)$x
      int_to_disc <- function(int) {disc_levels[int]}
      distributional::dist_transformed(x, disc_to_int, int_to_disc)
    } else if (is_discrete(x)){
      ggplot2::ggproto_parent(ggplot2::ScaleDiscretePosition, self)$map(x)
    } else {
      x
    }
  }
)


# stole from ggplot
#' @keywords internal
is_discrete <- function(x) {
  is.factor(x) || is.character(x) || is.logical(x)
}




