% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/edf.R
\name{edf}
\alias{edf}
\title{Expected degrees of freedom}
\usage{
edf(object, nsim = 10000, seed = NULL)
}
\arguments{
\item{object}{A \code{cirls} object or any object inheriting from \code{lm}, see details.}

\item{nsim}{The number of simulations.}

\item{seed}{An optional seed for the random number generator. See \link[=set.seed]{set.seed}.}
}
\value{
A vector of length three with components:
\item{udf}{The \emph{unconstrained} degrees of freedom, i.e. the rank plus any dispersion parameter for \code{glm} objects.}
\item{odf}{The \emph{observed} degrees of freedom, that is \code{udf} minus the number of active constraints.}
\item{edf}{The \emph{expected} degrees of freedom estimated by simulation as described in the details section. For any other object inheriting from \code{lm}, attempts to retrieve the \emph{effective} degrees of freedom.}
For \code{cirls} objects, the vector includes the simulated distribution of the number of active constraints as an \code{actfreq} attribute.
}
\description{
Estimate expected degrees of freedom of a \link[=cirls.fit]{cirls} object through simulations.
}
\details{
Simulates coefficient vectors from their unconstrained distribution, which is the non-truncated multivariate normal distribution. For each simulated vector, counts the number of violated constraints as the number of active constraints under the constrained distribution. The expected degrees of freedom is then the number of parameters minus the average number of active constraints.

This procedure allows to account for the randomness of degrees of freedom for the constrained model. Indeed, the observed degrees of freedom is the number of parameters minus the number of active constraints. However, the number of active constraints is random as, some constraints can be active or not depending on the observed data. For instance, in a model for which the constraints are binding, the expected degrees of freedom will be close to the observed one, while in a model in which the constraints are irrelevant, the expected degrees of freedom will be closer to the unconstrained (usual) ones.
}
\section{Note}{
This function is implemented mainly for \link[=cirls.fit]{cirls} objects and can return idiosyncratic results for other objects inheriting from \code{lm}. In this case, it will attempt to retrieve an 'edf' value, but simply return the rank of the model if this fails. For \code{glm} models for instance, it will return thrice the same value.
}

\examples{
# Simulate a simple dataset
set.seed(5)
x <- rnorm(100)
y <- x + rnorm(100)

#### Model with a sensible constraint
# Reduces edf compared to udf as the constraint is sometimes active
mod1 <- glm(y ~ x, method = "cirls.fit", Cmat = list(x = 1), lb = 1)
edf(mod1)

#### Model with an almost surely binding constraint
# In this case edf is very close to odf as the constraint is often active
mod2 <- glm(y ~ x, method = "cirls.fit", Cmat = list(x = 1), lb = 1.5)
edf(mod2)

#### Model with an irrelevant constraint
# Here the constraint is useless and edf is equal to unconstrained df
mod3 <- glm(y ~ x, method = "cirls.fit", Cmat = list(x = 1), lb = -5)
edf(mod3)
}
\references{
Meyer, M.C., 2013. Semi-parametric additive constrained regression. \emph{Journal of Nonparametric Statistics} \strong{25}, \strong{715–730}. \doi{10.1080/10485252.2013.797577}
}
\seealso{
\link[=logLik.cirls]{logLik.cirls} which internally calls \code{edf} to compute degrees of freedom.
}
