% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bwScv.R
\name{bwScv}
\alias{bwScv}
\title{Smoothed Cross-Validation for Circular Data}
\usage{
bwScv(x, np = 500, lower = 0, upper = 60, tol = 0.1)
}
\arguments{
\item{x}{Data from which the smoothing parameter is to be computed. The object is
coerced to a numeric vector in radians using \code{\link[circular]{circular}}.
Can be a numeric vector or an object of class \code{circular}. Note: computational
complexity scales as O(n²) due to outer product operations; datasets with n > 2000
may cause performance and memory issues.}

\item{np}{An integer specifying the number of points used in numerical integration
to evaluate the SCV criterion. A higher number increases precision but also
computational cost (recommended value is >= 100). Default is 500.}

\item{lower}{Lower boundary of the interval for the optimization of the smoothing
parameter \code{kappa}. Must be a positive numeric value smaller than \code{upper}.
Default is 0.}

\item{upper}{Upper boundary of the interval for the optimization of the smoothing
parameter \code{kappa}. Must be a positive numeric value greater than \code{lower}.
Default is 60.}

\item{tol}{Convergence tolerance for the \code{\link[stats]{optimize}} function, determining the
precision of the optimization process. Also used to detect convergence near boundaries:
if the optimal value is within \code{tol} of \code{lower} or \code{upper}, a warning
is issued suggesting interval adjustment. Default is 0.1.}
}
\value{
The computed optimal smoothing parameter \code{kappa}, a numeric concentration
parameter (analogous to inverse radians) that minimizes the smoothed cross-validation
criterion within the interval \code{[lower, upper]} and the value of objective function
at that point. Higher values indicate sharper, more concentrated kernels and less
smoothing; lower values indicate broader kernels and more smoothing. If the
optimization fails, a warning is issued.
}
\description{
This function computes the optimal smoothing parameter (bandwidth) for circular data using a smoothed cross-validation
(SCV) method (see \url{doi:10.1007/s00180-023-01401-0}).
}
\details{
The smoothed cross-validation (SCV) method is an alternative bandwidth
selection approach, originally introduced by Hall & Marron (1992) for linear
densities and adapted for circular data by Zámečník et al. (2023).

The SCV criterion is given by
\deqn{\mathrm{SCV}(\kappa) = \frac{R(K)}{nh}
 + \frac{1}{n^{2}} \sum_{i=1}^{n} \sum_{j=1}^{n}
    \big(K_{\kappa} * K_{\kappa} * K_{\kappa} * K_{\kappa} - 2K_{\kappa} * K_{\kappa} *K_{\kappa} + K_{\kappa} * K_{\kappa}\big)(\Theta_i - \Theta_j)}
where \eqn{K_\kappa} is the Von Mises kernel with concentration \eqn{\kappa} (for the formula see 3.7, 3.8 in Zámečník et al. (2023)). The optimal bandwidth minimizes the sum
\eqn{ISB(\kappa) + IV(\kappa)} over the interval \code{[lower, upper]}.

The integral expressions involved in the SCV criterion (see Sections 3.2 in Zámečník et al., 2023) are evaluated numerically using the trapezoidal rule
on a uniform grid of length \code{np}.
}
\examples{
# Example with circular data (Lower `nu` = more smoothing; higher = sharper peaks).
library(circular)
x <- rwrappednormal(100, mu = circular(2), rho = 0.5)
bw <- bwScv(x)
print(round(bw$minimum, 2))

x <- rvonmises(100, mu = circular(0.5), kappa = 2)
bw <- bwScv(x)
print(round(bw$minimum, 2))

}
\references{
Zámečník, S., Horová, I., Katina, S., & Hasilová, K. (2023). An adaptive
method for bandwidth selection in circular kernel density estimation.
\emph{Computational Statistics}.
\doi{10.1007/s00180-023-01401-0}

Hall, P., & Marron, J. S. (1992). On the amount of noise inherent in bandwidth
selection for a kernel density estimator. \emph{The Annals of Statistics},
20(1), 163-181.
}
\seealso{
\link{bwTs}, \link{bwLscvg}, \link{bwCcv}
}
