#' Date, time, and OS CLI assertions
#'
#' These functions wrap \pkg{checkmate} checks for dates, POSIXct date-times,
#' and operating systems, returning the underlying check result
#' (`check_*_cli()`) or raising a `cli::cli_abort()` error via
#' [make_assertion()] (`assert_*_cli()`).
#' They are useful in user-facing workflows that depend on temporal objects
#' (e.g., time series, observation timestamps) or that only support specific
#' operating systems, while emitting consistent CLI-styled error messages.
#'
#' @param x Object to be checked.
#'   This is the value whose type, length, or other properties are validated.
#'
#' @param ... Additional arguments passed on to the corresponding
#'   \pkg{checkmate} function (e.g. `lower`, `upper`, `any.missing`,
#'   `all.missing`, `min.len`, `null.ok`).
#'
#' @param .var.name Character scalar used in error messages to refer to
#'   the checked object. Defaults to [checkmate::vname()], which tries to
#'   infer the variable name from the calling context.
#'
#' @param add Optional [checkmate::AssertCollection] to which
#'   assertion failures are added instead of triggering an immediate
#'   error. Defaults to `NULL`, which causes a `cli::cli_abort()` on
#'   failure.
#'
#' - Dates:
#'   - [check_date_cli()], [assert_date_cli()] ensure that `x` is a `Date`
#'     vector and can enforce additional constraints such as length, bounds, or
#'     missingness.
#' - POSIXct date-times:
#'   - [check_posixct_cli()], [assert_posixct_cli()] ensure that `x` is a
#'     `POSIXct` vector, suitable for time stamps with time zones.
#' - Operating system:
#'   - [check_os_cli()], [assert_os_cli()] check that the current OS matches a
#'     given specification (e.g. `"windows"`, `"unix"`).
#'
#' In all `assert_*_cli()` functions, `x` is the object being checked and
#' `.var.name` is used only for error message construction; the return value is
#' always `x` (invisibly) on success.
#'
#' @returns
#'
#' - `check_*_cli()` functions return `TRUE` on success or a character vector
#'   describing the failure, exactly like the corresponding
#'   \pkg{checkmate} checks.
#' - `assert_*_cli()` functions return `x` invisibly on success and either:
#'   - raise a `cli::cli_abort()` error with bullet-style messages, or
#'   - push messages into an [checkmate::AssertCollection] if `add` is supplied.
#'
#' @examples
#'
#' # Dates:
#'
#' # Valid Date vector
#' d <- as.Date(c("2020-01-01", "2020-02-01"))
#' check_date_cli(d)
#'
#' # Character input is not a Date
#' try(check_date_cli(c("2020-01-01", "2020-02-01")))   # failure
#'
#' assert_date_cli(d)
#' try(assert_date_cli(c("2020-01-01", "2020-02-01")))
#'
#' # Single observation date
#' obs_date <- Sys.Date()
#' assert_date_cli(obs_date)
#'
#' # POSIXct date-times:
#'
#' # Valid POSIXct vector
#' t <- as.POSIXct(c("2020-01-01 12:00:00", "2020-01-02 08:30:00"), tz = "UTC")
#' check_posixct_cli(t)
#'
#' # Plain Date is not POSIXct
#' try(check_posixct_cli(as.Date("2020-01-01")))        # failure
#'
#' assert_posixct_cli(t)
#' try(assert_posixct_cli(as.Date("2020-01-01")))
#'
#' # Typical use in logging or time series
#' ts_times <- Sys.time() + 0:9
#' assert_posixct_cli(ts_times)
#'
#' # Operating system:
#'
#' # Check that the current OS is one of the supported ones
#' # (e.g., skip functions not available on Windows)
#' try(check_os_cli("unix"))                           # TRUE on Linux/macOS, failure on Windows
#'
#' # Assert OS:
#' try(assert_os_cli("unix"))
#'
#' @seealso
#'   [checkmate::check_date()], [checkmate::check_posixct()],
#'   [checkmate::check_os()], [make_assertion()], [assert_cli()]
#'
#' @name checkcli-datetime-os
#' @family checkCLI
#' @import checkmate
#' @import cli
#' @importFrom glue glue
#' @importFrom purrr map2

#' @rdname checkcli-datetime-os
#' @export
check_date_cli <- function(...) {
  checkmate::check_date(...)
}

#' @rdname checkcli-datetime-os
#' @export
assert_date_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_date(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-datetime-os
#' @export
check_posixct_cli <- function(...) {
  checkmate::check_posixct(...)
}

#' @rdname checkcli-datetime-os
#' @export
assert_posixct_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_posixct(x, ...),
      .var.name,
      add
    )
  )
}

#' @rdname checkcli-datetime-os
#' @export
check_os_cli <- function(...) {
  checkmate::check_os(...)
}

#' @rdname checkcli-datetime-os
#' @export
assert_os_cli <- function(x, ..., .var.name = checkmate::vname(x), add = NULL) {
  invisible(
    make_assertion(
      x,
      checkmate::check_os(x, ...),
      .var.name,
      add
    )
  )
}



