% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svem_wmt_multi.R
\name{svem_wmt_multi}
\alias{svem_wmt_multi}
\title{Whole-model tests for multiple SVEM responses (WMT wrapper)}
\usage{
svem_wmt_multi(
  formulas,
  data,
  mixture_groups = NULL,
  wmt_transform = c("neglog10", "one_minus_p"),
  wmt_control = list(seed = 123),
  plot = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{formulas}{A single formula or a (preferably named) list of formulas,
one per response. If unnamed, response names are inferred from the
left-hand side of each formula; non-unique names are made unique.}

\item{data}{Data frame containing the predictors and responses referenced
in \code{formulas}.}

\item{mixture_groups}{Optional mixture and simplex constraints passed to
\code{\link{svem_significance_test_parallel}}.}

\item{wmt_transform}{Character; transformation used to convert WMT
p-values into multipliers. One of:
\itemize{
\item \code{"neglog10"}: \eqn{f(p) = [-\log_{10}(p)]^{\text{strength}}},
\item \code{"one_minus_p"}: \eqn{f(p) = (1 - p)^{\text{strength}}}.
}
Currently, \code{strength = 1} is used internally.}

\item{wmt_control}{Optional list of extra arguments passed directly to
\code{\link{svem_significance_test_parallel}}. By default this is
\code{list(seed = 123)} so that WMT calls are reproducible; you may
override or extend this (e.g. \code{list(seed = 999, nPerm = 300)}).
Any entries not recognized by \code{svem_significance_test_parallel}
are ignored by that function.}

\item{plot}{Logical; if \code{TRUE} (default), attempt to plot all
successfully computed WMT objects together via
\code{\link{plot.svem_significance_test}}.}

\item{verbose}{Logical; if \code{TRUE} (default), print progress and a
compact summary of p-values and multipliers.}
}
\value{
A list of class \code{"svem_wmt_multi"} with components:
\describe{
\item{\code{wmt_objects}}{Named list of WMT objects (one per response),
as returned by \code{svem_significance_test_parallel()}. Entries
are \code{NULL} where a WMT call failed.}
\item{\code{p_values}}{Named numeric vector of per-response p-values
(bounded away from 0/1), or \code{NA} when unavailable.}
\item{\code{multipliers}}{Named numeric vector of per-response WMT
multipliers derived from the p-values using \code{wmt_transform}.}
\item{\code{wmt_transform}}{The transformation used.}
\item{\code{wmt_control}}{The list of arguments passed through to
\code{svem_significance_test_parallel()}.}
}
}
\description{
Convenience wrapper around \code{\link{svem_significance_test_parallel}}
for running whole-model tests (WMT) on multiple responses that share the
same dataset and mixture constraints. This helper:
\itemize{
\item takes a formula or a list of formulas and a single \code{data} frame,
\item calls \code{svem_significance_test_parallel()} for each response,
\item extracts per-response p-values and converts them to WMT
multipliers via a chosen transform, and
\item optionally plots the WMT objects together using
\code{\link{plot.svem_significance_test}} and prints a compact
summary of p-values and multipliers.
}

The resulting \code{multipliers} vector is designed to be passed directly
to downstream scoring functions (for example, as an optional WMT argument
to \code{svem_score_random()}), with response names matched by
\code{names()}.
}
\examples{
\donttest{
data(lipid_screen)

spec <- bigexp_terms(
  Potency ~ PEG + Helper + Ionizable + Cholesterol +
    Ionizable_Lipid_Type + N_P_ratio + flow_rate,
  data             = lipid_screen,
  factorial_order  = 3,
  polynomial_order = 3,
  include_pc_2way  = TRUE,
  include_pc_3way  = FALSE
)

form_pot <- bigexp_formula(spec, "Potency")
form_siz <- bigexp_formula(spec, "Size")
form_pdi <- bigexp_formula(spec, "PDI")

mix <- list(list(
  vars  = c("PEG", "Helper", "Ionizable", "Cholesterol"),
  lower = c(0.01, 0.10, 0.10, 0.10),
  upper = c(0.05, 0.60, 0.60, 0.60),
  total = 1.0
))

set.seed(123)
wmt_out <- svem_wmt_multi(
  formulas       = list(Potency = form_pot,
                        Size    = form_siz,
                        PDI     = form_pdi),
  data           = lipid_screen,
  mixture_groups = mix,
  wmt_transform  = "neglog10",
  wmt_control    = list(seed = 123),
  plot           = TRUE
)

wmt_out$p_values
wmt_out$multipliers

## later: pass wmt_out$multipliers into svem_score_random()
}

}
\seealso{
\code{\link{svem_significance_test_parallel}},
\code{\link{plot.svem_significance_test}}
}
