% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MOEIW.R
\name{MOEIW}
\alias{MOEIW}
\title{The Marshall-Olkin Extended Inverse Weibull family}
\usage{
MOEIW(mu.link = "log", sigma.link = "log", nu.link = "log")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default for the mu parameter.}

\item{sigma.link}{defines the sigma.link, with "log" link as the default for the sigma.}

\item{nu.link}{defines the nu.link, with "log" link as the default for the nu parameter.}
}
\value{
Returns a gamlss.family object which can be used to fit a MOEIW distribution in the \code{gamlss()} function.
}
\description{
The Marshall-Olkin Extended Inverse Weibull family
}
\details{
The Marshall-Olkin Extended Inverse Weibull distribution with parameters \code{mu}, 
\code{sigma} and \code{nu} has density given by

\eqn{f(x) = \frac{\mu \sigma \nu x^{-(\sigma + 1)} exp\{{-\mu x^{-\sigma}}\}}{\{\nu -(\nu-1) exp\{{-\mu x ^{-\sigma}}\} \}^{2}},}

for x > 0.
}
\examples{
# Example 1
# Generating some random values with
# known mu, sigma and nu
set.seed(123456)
y <- rMOEIW(n=100, mu=0.6, sigma=1.7, nu=0.3)

# Fitting the model
require(gamlss)

mod <- gamlss(y~1, sigma.fo=~1, nu.fo=~1, family="MOEIW",
              control=gamlss.control(n.cyc=5000, trace=FALSE))

# Extracting the fitted values for mu, sigma and nu
# using the inverse link function
exp(coef(mod, what="mu"))
exp(coef(mod, what="sigma"))
exp(coef(mod, what="nu"))

# Example 2
# Generating random values under some model

# A function to simulate a data set with Y ~ MOEIW
gendat <- function(n) {
  x1 <- runif(n)
  x2 <- runif(n)
  mu    <- exp(-2.02 + 3 * x1) # 0.60 approximately
  sigma <- exp(2.23 - 2 * x2)  # 3.42 approximately
  nu    <- 2
  y <- rMOEIW(n=n, mu=mu, sigma=sigma, nu=nu)
  data.frame(y=y, x1=x1, x2=x2)
}

set.seed(123)
dat <- gendat(n=100)

mod <- gamlss(y~x1, sigma.fo=~x2, nu.fo=~1, 
              family=MOEIW, data=dat,
              control=gamlss.control(n.cyc=5000, trace=FALSE))

coef(mod, what="mu")
coef(mod, what="sigma")
exp(coef(mod, what="nu"))
}
\references{
Okasha, H. M., El-Baz, A. H., Tarabia, A. M. K., & Basheer, A. M. (2017). 
Extended inverse Weibull distribution with reliability application. 
Journal of the Egyptian Mathematical Society, 25(3), 343-349.
}
\seealso{
\link{dMOEIW}
}
\author{
Amylkar Urrea Montoya, \email{amylkar.urrea@udea.edu.co}
}
