#' Compute Softmax Weights
#'
#' Calculates softmax weights for a vector of component scores using a
#' temperature parameter that controls the sharpness of the weight
#' distribution.
#'
#' @param scores Numeric vector of component scores (e.g.,
#'   \code{c(MSG, MRG, cMTG)} or \code{c(MSG, MRG, cMTG, RVS)}).
#' @param temperature Numeric. Temperature parameter controlling weight
#'   concentration. Lower values produce sharper (more concentrated)
#'   weights; higher values approach uniform weighting. Default is 0.13,
#'   calibrated via perplexity targeting to achieve approximately 2.0
#'   effective components out of 3.
#'
#' @return Named numeric vector (if input is named) or unnamed numeric
#'   vector of weights summing to 1.
#'
#' @details
#' The softmax function transforms component scores into a probability
#' distribution:
#' \deqn{W_i = \frac{\exp(C_i / T)}{\sum_j \exp(C_j / T)}}
#'
#' The temperature parameter \eqn{T} serves two roles:
#' \enumerate{
#'   \item \strong{Signal extraction:} Low \eqn{T} (e.g., 0.13) concentrates
#'     weight on the strongest component, extracting the dominant biological
#'     signal.
#'   \item \strong{Zero handling:} When any \eqn{C_k = 0}, the corresponding
#'     weight \eqn{W_k \approx 0} and remaining weights renormalize
#'     automatically, preventing the zero-collapse that occurs with
#'     geometric aggregation.
#' }
#'
#' @section Numerical stability:
#' The implementation uses the log-sum-exp trick for numerical stability
#' with extreme score ratios.
#'
#' @examples
#' # Typical corn profile
#' softmax_weights(c(MSG = 0.8, MRG = 1.0, cMTG = 0.6))
#'
#' # Zero component: weight goes to ~0, others renormalize
#' softmax_weights(c(MSG = 0.8, MRG = 0.0, cMTG = 0.7))
#'
#' # Higher temperature: more uniform weights
#' softmax_weights(c(0.8, 0.5, 0.3), temperature = 1.0)
#'
#' @seealso \code{\link{calibrate_temperature}} for selecting the
#'   temperature parameter.
#'
#' @export
softmax_weights <- function(scores, temperature = 0.13) {
  if (!is.numeric(scores) || length(scores) < 2) {
    stop("scores must be a numeric vector with at least 2 elements.")
  }
  if (!is.numeric(temperature) || length(temperature) != 1 ||
      temperature <= 0) {
    stop("temperature must be a single positive number.")
  }

  # Log-sum-exp trick for numerical stability
  scaled <- scores / temperature
  max_scaled <- max(scaled)
  shifted <- scaled - max_scaled
  exps <- exp(shifted)
  weights <- exps / sum(exps)

  if (!is.null(names(scores))) {
    names(weights) <- names(scores)
  }

  weights
}
