% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RLum.OSL_decomposition.R
\name{RLum.OSL_decomposition}
\alias{RLum.OSL_decomposition}
\title{Separate CW-OSL components in RLum.Analysis data sets}
\usage{
RLum.OSL_decomposition(
  object,
  record_type = "OSL",
  K = 3,
  decay_rates = NULL,
  report = FALSE,
  report_dir = NULL,
  image_format = "pdf",
  open_report = TRUE,
  rmd_path = NULL,
  verbose = TRUE
)
}
\arguments{
\item{object}{\link[Luminescence:RLum.Analysis-class]{Luminescence::RLum.Analysis} or \link{list} of \link[Luminescence:RLum.Analysis-class]{Luminescence::RLum.Analysis}
(\strong{required}):
Data set of one or multiple CW-OSL measured aliquots. The data set must either
contain a list element \verb{$OSL_COMPONENTS} or the parameter \code{decay_rates} must
be defined.}

\item{record_type}{\link{character} (\emph{with default}):
Type of records, selected by the \link[Luminescence:RLum.Analysis-class]{Luminescence::RLum.Analysis} attribute \verb{@recordType}.
Common are: \code{"OSL"},\code{"SGOSL"} or \code{"IRSL"}.}

\item{K}{\link{numeric} (\emph{with default}):
Number of components. Selects the according result table in the \verb{$OSL_COMPONENTS} list item of the data set \code{object}.}

\item{decay_rates}{\link{numeric} vector or \link{data.frame} (\emph{optional}):
User-defined component decay rates. If this parameter is defined, the parameter \code{K} will ignored.
If the input object is a \link{data.frame}, then the decay rates must be stored in the column \verb{$lambda}.}

\item{report}{\link{logical} (\emph{with default}):
Creates a \code{html} report, saves it in the \code{report_dir} directory.
The report contains the results and detailed information on the data processing.}

\item{report_dir}{\link{character} (\emph{optional}):
Path of output directory if \code{report = TRUE}. If \code{report_dir = NULL} (default),
a temporary folder is used which is deleted when the R session is closed.
File paths are also allowed as parameter, then a new directory named after the OSL data file
will be created.}

\item{image_format}{\link{character} (\emph{with default}):
Image format of the automatically saved graphs if \code{report = TRUE} and \code{report_dir} is set.
Allowed are \code{.pdf}, \code{.eps}, \code{.svg} (vector graphics), \code{.jpg}, \code{.png}, \code{.bmp} (pixel graphics)
and more, see \link[ggplot2:ggsave]{ggplot2::ggsave}. The images are saved in the \code{report_dir} subfolder \verb{/report_figures}.
Set \code{image_format = NULL} if no images shall be saved.}

\item{open_report}{\link{logical} (\emph{with default}):
If set to \code{TRUE} a browser window displaying the report will be opened automatically.}

\item{rmd_path}{\link{character} (\emph{with default}):
\strong{For advanced users:} File path to the \link[rmarkdown:rmarkdown-package]{rmarkdown::rmarkdown-package} source code file of the report.
This allows to execute a manipulated version of the report.}

\item{verbose}{\link{logical} (\emph{with default}):
Enables console text output.}
}
\value{
The input \code{object}, a \link{list} of \link[Luminescence:RLum.Analysis-class]{Luminescence::RLum.Analysis} objects is returned but with
a new list element \code{object[["DECOMPOSITION"]]}, containing:

\itemize{
\item \verb{$decompositon.input} \link{data.frame}: Set of input components. Relevant is just the column \verb{$lambda}
\item \verb{$results} \link{data.frame}: Overview table of decomposition
\item \verb{$parameters} \link{list}: Input and algorithm parameters
}

The \link[Luminescence:RLum.Data.Curve-class]{Luminescence::RLum.Data.Curve} attribute \verb{@info} of each CW-OSL record contains the
new entry \verb{$COMPONENTS} with the curve-individual signal component parameters.
It can be read for example by:

\code{object[[i]]@records[[j]]@info[["COMPONENTS"]]}
}
\description{
Calculates the CW-OSL signal component intensities for each CW-OSL measurement
under the requirement that the decay rates are already given. The signal decomposition
process uses an analytical approach described in detail in Mittelstrass (2019) and
Mittelstrass et al. (in preparation). This function processes \link[Luminescence:RLum.Analysis-class]{Luminescence::RLum.Analysis} data sets created within the \link[Luminescence:Luminescence-package]{Luminescence::Luminescence-package} (Kreutzer et al. 2012).

The workflow of this function is as follows:

\enumerate{
\item \link{optimise_OSLintervals}: Approximates the optimal integration intervals. Uses the global
average curve as time axis template. If none global average curve is given, one is automatically created using \link{sum_OSLcurves}.
\item \link{decompose_OSLcurve}: Calculates component intensities for \strong{all} \code{record_type} measurements.
Uses the \code{"det"} algorithm if a background correction was performed with \link{RLum.OSL_correction} or the
\code{"det+nls"} algorithm if no background correction was performed. For error estimation, the \code{"empiric"} approach is used.
\item Creates a \code{html} report to summarize the results (\emph{optional}).
}

Data sets must be formatted as \link[Luminescence:RLum.Analysis-class]{Luminescence::RLum.Analysis} objects and
should have been processed with \link{RLum.OSL_correction} and \link{RLum.OSL_global_fitting} beforehand.
Output objects are also \link[Luminescence:RLum.Analysis-class]{Luminescence::RLum.Analysis} objects and are
meant for equivalent dose determination with \link[Luminescence:analyse_SAR.CWOSL]{Luminescence::analyse_SAR.CWOSL}.

If \code{report = TRUE}, a \code{html} report of the results is rendered by the \link[rmarkdown:rmarkdown-package]{rmarkdown::rmarkdown-package}
and saved in the working directory, which is usually the directory of the data file.
This report can be displayed, shared and published online without any requirements regarding
the operation system or installed software. However, an internet connection is needed to display
the \emph{MathJax} encoded equations and special characters.
The \emph{Rmarkdown} source code of the report can be found with the following command:

\code{system.file("rmd", "report_Step2.Rmd", package = "OSLdecomposition")}
}
\section{Last updates}{


2023-09-01, DM: Improved input checks to return more helpful messages
}

\examples{

#'FB_10Gy' is a dose recovery test with the Fontainebleau quartz
# measured in a lexsyg research with green LED stimulation
data_path <- system.file("examples", "FB_10Gy_SAR.bin", package = "OSLdecomposition")
data_set <- Luminescence::read_BIN2R(data_path, fastForward = TRUE)

# Separate components
data_set_decomposed <- RLum.OSL_decomposition(
data_set, decay_rates = c(0.8, 0.05))

}
\references{
Kreutzer, S., Schmidt, C., Fuchs, M.C., Dietze, M., Fischer, M., Fuchs, M., 2012.
Introducing an R package for luminescence dating analysis. Ancient TL, 30 (1), 1-8.

Mittelstraß, D., 2019. Decomposition of weak optically stimulated luminescence signals and
its application in retrospective dosimetry at quartz (Master thesis). TU Dresden, Dresden.
}
\seealso{
\link{RLum.OSL_global_fitting}, \link{decompose_OSLcurve}, \link{optimise_OSLintervals}, \link[Luminescence:analyse_SAR.CWOSL]{Luminescence::analyse_SAR.CWOSL}
}
\author{
Dirk Mittelstrass, \email{dirk.mittelstrass@luminescence.de}

Please cite the package the following way:

Mittelstraß, D., Schmidt, C., Beyer, J., Heitmann, J. and Straessner, A.:
R package OSLdecomposition: Automated identification and separation of quartz CW-OSL signal components, \emph{in preparation}.
}
