#' Retrieve relative variability metrics from NORMAgene output
#'
#' Retrieves relative variability metrics calculated during normalization.
#'
#' For more information on relative variability metrics, see [NORMAgene-package].
#'
#' @param x
#' An object returned by [norma_gene()].
#'
#' @param type
#' Character string specifying which relative variability metric to return. One of `"by_gene"` or `"by_condition"`.
#'
#' @return Depending on `type`:
#' \describe{
#'   \item{by_gene}{A named numeric matrix of gene-level relative variability values, calculated both within experimental conditions and cumulatively across all experimental conditions.}
#'   \item{by_condition}{A named numeric vector of relative variability calculated within experimental conditions, as well as cumulatively across all experimental conditions, regardless of gene.}
#' }
#'
#' @seealso
#' [norma_gene()] \cr
#' [NORMAgene-package]
#'
#' @examples
#' # load example dataset containing raw CT values and
#' # metadata from a multi-condition experiment
#'
#' data(multi_cond_data)
#' raw_data<-multi_cond_data
#'
#' #normalize CT values via NORMAgene
#'
#' norm_data<-norma_gene(
#'   data = raw_data,
#'   conditions = "Diagnosis",
#'   replicates= "Sample_id"
#' )
#'
#' # retrieve relative variability metrics
#'
#' relative_variability(norm_data, type = "by_gene")
#' relative_variability(norm_data, type = "by_condition")
#'
#' @export

relative_variability<-function(x, type=c("by_gene", "by_condition")){

  type<-match.arg(type)

  #----input parsing and validation----

  if(!inherits(x, "norma_gene")){
    stop("`x` must be an object returned by `norma_gene()`.")
  }

  rel_var<-attr(x, "rel_var", exact=TRUE)

  if(is.null(rel_var)){
    stop("No relative variability information found in `x`.")
  }

  #----dispatch----

  switch(
    type,
    by_gene=rel_var$by_gene,
    by_condition=rel_var$by_cond
  )
}
